/**
 * Copyright (C) 2012 JBoss Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jbpm.formModeler.core.processing.fieldHandlers;

import org.apache.commons.lang.StringUtils;
import org.jbpm.formModeler.api.model.DataHolder;
import org.jbpm.formModeler.api.model.Field;
import org.jbpm.formModeler.api.model.Form;
import org.jbpm.formModeler.core.processing.*;
import org.jbpm.formModeler.core.processing.formRendering.FormErrorMessageBuilder;
import org.jbpm.formModeler.core.rendering.SubformFinderService;

import javax.inject.Inject;
import javax.inject.Named;
import java.util.*;

@Named("org.jbpm.formModeler.core.processing.fieldHandlers.SubformFieldHandler")
public class SubformFieldHandler extends PersistentFieldHandler {
    private static transient org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(SubformFieldHandler.class.getName());

    @Inject
    private SubformFinderService subformFinderService;

    @Inject
    private FormErrorMessageBuilder formErrorMessageBuilder;

    private String pageToIncludeForRendering = "/formModeler/fieldHandlers/Subform/input.jsp";
    private String pageToIncludeForDisplaying = "/formModeler/fieldHandlers/Subform/show.jsp";
    private String pageToIncludeForSearching = "/formModeler/fieldHandlers/Subform/search.jsp";

    private static int maxDepth = 2;

    /**
     * Determine the list of class types this field can generate. That is, normally,
     * a field can generate multiple outputs (an input text can generate Strings,
     * Integers, ...)
     *
     * @return the set of class types that can be generated by this handler.
     */
    public String[] getCompatibleClassNames() {
        return new String[]{Object.class.getName()};
    }


    public Object getValue(Field field, String inputName, Map parametersMap, Map filesMap, String desiredClassName, Object previousValue) throws Exception {
        Form form = getEnterDataForm(inputName, field);
        if (!checkSubformDepthAllowed(form.getId(), inputName)) return null;
        getFormProcessor().setValues(form, inputName, parametersMap, filesMap);
        FormStatusData status = getFormProcessor().read(form, inputName);
        if (status.isValid()) {
            // Check if form status is empty & if the object already exists to avoid null objects creation.
            if (status.isEmpty()) return null;
            Map m = getFormProcessor().getMapRepresentationToPersist(form, inputName);
            return m;
        } else {
            throw new IllegalArgumentException("Subform status is invalid.");
        }
    }

    @Override
    public Object persist(Field field, String inputName, String desiredClass) throws Exception {
        Form form = getEnterDataForm(inputName, field);
        Map representation = getFormProcessor().getMapRepresentationToPersist(form, inputName);

        return getFormProcessor().persistFormHolder(form, inputName, representation, form.getDataHolderByInfo(field.getSubformClass()));
    }

    public Map getParamValue(String inputName, Object value, String pattern) {
        if (value == null)
            return Collections.EMPTY_MAP;
        Map m = new HashMap();

        FormNamespaceData data = getNamespaceManager().getNamespace(inputName);
        Field parentField = data.getForm().getField(data.getFieldNameInParent());
        Form childForm = getEnterDataForm(inputName, parentField);
        DataHolder holder = childForm.getDataHolderByInfo(parentField.getSubformClass());
        for (Iterator it = childForm.getFormFields().iterator(); it.hasNext();) {
            Field childField = (Field) it.next();
            String bindingExpression = StringUtils.defaultIfEmpty(childField.getInputBinding(), childField.getOutputBinding());
            if (!holder.isAssignableForField(childField)) continue;
            try {
                Object val = holder.readFromBindingExperssion(value, bindingExpression);
                FieldHandler fieldManager = getFieldHandlersManager().getHandler(childField.getFieldType());
                Map childrenMap = fieldManager.getParamValue(inputName + FormProcessor.NAMESPACE_SEPARATOR + childForm.getId() + FormProcessor.NAMESPACE_SEPARATOR + childField.getFieldName(), val, childField.getFieldPattern());
                if (childrenMap != null) m.putAll(childrenMap);
            } catch (Exception e) {
                log.warn("Error reading value from field '" + childField.getFieldName() + "': ", e);
            }
        }
        return m;
    }

    public List getWrongChildFieldErrors(String namespace, Field field) {
        return formErrorMessageBuilder.getWrongFormErrors(namespace, getEnterDataForm(namespace, field));
    }


    /**
     * When rendering a form, if field is handled by this handler, determine the
     * page that renders the displaying of the value
     *
     * @return a page to include
     */
    public String getPageToIncludeForDisplaying() {
        return pageToIncludeForDisplaying;
    }

    public boolean isEmpty(Object value) {
        return value == null || "".equals(value);
    }

    public void setPageToIncludeForDisplaying(String pageToIncludeForDisplaying) {
        this.pageToIncludeForDisplaying = pageToIncludeForDisplaying;
    }

    public boolean acceptsPropertyName(String propName) {
        return true;
    }

    /**
     * When rendering a form, if field is handled by this handler, determine the
     * page that renders the input(s)
     *
     * @return a page to include
     */
    public String getPageToIncludeForRendering() {
        return pageToIncludeForRendering;
    }

    public void setPageToIncludeForRendering(String pageToIncludeForRendering) {
        this.pageToIncludeForRendering = pageToIncludeForRendering;
    }

    public String getPageToIncludeForSearching() {
        return pageToIncludeForSearching;
    }

    public void setPageToIncludeForSearching(String pageToIncludeForSearching) {
        this.pageToIncludeForSearching = pageToIncludeForSearching;
    }

    protected Form getEnterDataForm(String namespace, Field field) {
        String formName = field.getDefaultSubform();

        return getForm(formName, namespace);
    }

    private Form getForm(String formPath, String namespace) {
        return subformFinderService.getFormFromPath(formPath, namespace);
    }

    public static boolean checkSubformDepthAllowed(Long id, String namesapce) {
        StringTokenizer token = new StringTokenizer(namesapce, FormProcessor.NAMESPACE_SEPARATOR, false);
        String _id = id.toString();
        int count = 0;
        while (token.hasMoreElements()) {
            String idToCompare = (String) token.nextElement();
            if (idToCompare.equals(_id)) {
                count++;
                if (count >= maxDepth) {
                    return false;
                }
            }

        }
        return true;

    }
}
