/*
 * Copyright The WildFly Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package org.wildfly.clustering.function;

/**
 * An enhanced binary function.
 * @author Paul Ferraro
 * @param <T> the operator type
 */
public interface BinaryOperator<T> extends java.util.function.BinaryOperator<T>, BiFunction<T, T, T> {
	/** An identity function using the former parameter */
	BinaryOperator<?> FORMER_IDENTITY = (value1, value2) -> value1;
	/** An identity function using the latter parameter */
	BinaryOperator<?> LATTER_IDENTITY = (value1, value2) -> value2;
	/** An operator that always returns null. */
	BinaryOperator<?> NULL = (value1, value2) -> null;

	/**
	 * Returns a composed operator that applies the specified operators to each parameter as inputs to this operator.
	 * @param before1 the operator applied to the first parameter
	 * @param before2 the operator applied to the second parameter
	 * @return a composed operator that applies the specified operators to each parameter as inputs to this operator.
	 */
	default BinaryOperator<T> compose(java.util.function.UnaryOperator<T> before1, java.util.function.UnaryOperator<T> before2) {
		return new BinaryOperator<>() {
			@Override
			public T apply(T value1, T value2) {
				return BinaryOperator.this.apply(before1.apply(value1), before2.apply(value2));
			}
		};
	}

	/**
	 * Returns a composed operator that applies the specified operators to each parameter as inputs to this operator.
	 * @param before1 the operator applied to the first parameter
	 * @param before2 the operator applied to the second parameter
	 * @return a composed operator that applies the specified operators to each parameter as inputs to this operator.
	 */
	default UnaryOperator<T> composeUnary(java.util.function.UnaryOperator<T> before1, java.util.function.UnaryOperator<T> before2) {
		return new UnaryOperator<>() {
			@Override
			public T apply(T value) {
				return BinaryOperator.this.apply(before1.apply(value), before2.apply(value));
			}
		};
	}

	/**
	 * An operator variant of {@link BiFunction#andThen(java.util.function.Function)}.
	 * @param after the operator to invoke using the result of this operator.
	 * @return a binary operator that invokes the specified operator using the result of this operator.
	 */
	default BinaryOperator<T> andThen(java.util.function.UnaryOperator<T> after) {
		return new BinaryOperator<>() {
			@Override
			public T apply(T value1, T value2) {
				return after.apply(BinaryOperator.this.apply(value1, value2));
			}
		};
	}

	/**
	 * Returns a function that processes this function with reversed parameter order.
	 * @return a function that processes this function with reversed parameter order.
	 */
	@Override
	default BinaryOperator<T> reverse() {
		return new BinaryOperator<>() {
			@Override
			public T apply(T value2, T value1) {
				return BinaryOperator.this.apply(value1, value2);
			}
		};
	}

	/**
	 * Returns a function that applies this function to the values returned by the specified providers if its parameters do not match the specified predicates.
	 * @param predicate1 a predicate used to determine the first parameter of this function
	 * @param defaultValue1 a provider of the default value of the first parameter
	 * @param predicate2 a predicate used to determine the second parameter of this function
	 * @param defaultValue2 a provider of the default value of the second parameter
	 * @return a function that applies this function to the value returned by the specified provider if its value does not match the specified predicate.
	 */
	@Override
	default BinaryOperator<T> withDefault(java.util.function.Predicate<T> predicate1, java.util.function.Supplier<T> defaultValue1, java.util.function.Predicate<T> predicate2, java.util.function.Supplier<T> defaultValue2) {
		return new BinaryOperator<>() {
			@Override
			public T apply(T value1, T value2) {
				return BinaryOperator.this.apply(predicate1.test(value1) ? value1 : defaultValue1.get(), predicate2.test(value2) ? value2 : defaultValue2.get());
			}
		};
	}

	/**
	 * Returns a function that applies this function if its parameters matches the specified predicate, or returns the value provided by the specified supplier otherwise.
	 * @param predicate a predicate used to determine the parameter of this function
	 * @param defaultResult a provider of the default parameter value
	 * @return a function that applies this function if its parameter matches the specified predicate, or returns the value provided by the specified supplier otherwise.
	 */
	@Override
	default BinaryOperator<T> orDefault(java.util.function.BiPredicate<T, T> predicate, java.util.function.Supplier<T> defaultResult) {
		return new BinaryOperator<>() {
			@Override
			public T apply(T value1, T value2) {
				return predicate.test(value1, value2) ? BinaryOperator.this.apply(value1, value2) : defaultResult.get();
			}
		};
	}

	/**
	 * Returns a function that returns its first parameter.
	 * @param <T> the operating type
	 * @return a function that returns its first parameter.
	 */
	@SuppressWarnings("unchecked")
	static <T> BinaryOperator<T> former() {
		return (BinaryOperator<T>) FORMER_IDENTITY;
	}

	/**
	 * Returns a function that returns its second parameter.
	 * @param <T> the operating type
	 * @return a function that returns its first parameter.
	 */
	@SuppressWarnings("unchecked")
	static <T> BinaryOperator<T> latter() {
		return (BinaryOperator<T>) LATTER_IDENTITY;
	}

	/**
	 * Returns a function that always returns null, ignoring its parameter.
	 * @param <T> the operating type
	 * @return a function that always returns null, ignoring its parameter.
	 */
	@SuppressWarnings("unchecked")
	static <T> BinaryOperator<T> empty() {
		return (BinaryOperator<T>) NULL;
	}

	/**
	 * Returns a function that always returns the specified value, ignoring its parameter.
	 * @param <T> the operating type
	 * @param result the function result
	 * @return a function that always returns the specified value, ignoring its parameter.
	 */
	static <T> BinaryOperator<T> of(T result) {
		return (result != null) ? of(BiConsumer.empty(), Supplier.of(result)) : empty();
	}

	/**
	 * Returns an operator that accepts its parameters via the specified consumer and returns the value returned by the specified supplier.
	 * @param <T> the operator type
	 * @param consumer the consumer of the function parameter
	 * @param supplier the supplier of the function result
	 * @return an operator that accepts its parameters via the specified consumer and returns the value returned by the specified supplier.
	 */
	static <T> BinaryOperator<T> of(java.util.function.BiConsumer<T, T> consumer, java.util.function.Supplier<T> supplier) {
		return new BinaryOperator<>() {
			@Override
			public T apply(T value1, T value2) {
				consumer.accept(value1, value2);
				return supplier.get();
			}
		};
	}

	/**
	 * Returns an operator view of the specified binary function.
	 * @param <T> the operating type
	 * @param function the delegating function
	 * @return an operator view of the specified function.
	 */
	static <T> BinaryOperator<T> apply(java.util.function.BiFunction<? super T, ? super T, T> function) {
		return (function != null) && (function != Function.NULL) ? new BinaryOperator<>() {
			@Override
			public T apply(T value1, T value2) {
				return function.apply(value1, value2);
			}
		} : empty();
	}

	/**
	 * Returns an operator that applies the former parameter to the specified operator.
	 * @param <T> the operating type
	 * @param operator the operator applied to the former parameter
	 * @return an operator that applies the former parameter to the specified operator.
	 */
	static <T> BinaryOperator<T> applyFormer(java.util.function.UnaryOperator<T> operator) {
		return new BinaryOperator<>() {
			@Override
			public T apply(T value, T ignored) {
				return operator.apply(value);
			}
		};
	}

	/**
	 * Returns an operator that applies the latter parameter to the specified operator.
	 * @param <T> the operating type
	 * @param operator the operator applied to the latter parameter
	 * @return an operator that applies the latter parameter to the specified operator.
	 */
	static <T> BinaryOperator<T> applyLatter(java.util.function.UnaryOperator<T> operator) {
		return new BinaryOperator<>() {
			@Override
			public T apply(T ignored, T value) {
				return operator.apply(value);
			}
		};
	}
}
