/*
 * Copyright The WildFly Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package org.wildfly.clustering.function;

/**
 * An enhanced binary function.
 * @author Paul Ferraro
 * @param <T> the former parameter type
 * @param <U> the latter parameter type
 * @param <R> the result type
 */
public interface BiFunction<T, U, R> extends java.util.function.BiFunction<T, U, R> {
	/** An identity function using the former parameter */
	BiFunction<?, ?, ?> FORMER_IDENTITY = (value1, value2) -> value1;
	/** An identity function using the latter parameter */
	BiFunction<?, ?, ?> LATTER_IDENTITY = (value1, value2) -> value2;
	/** A function that always returns null. */
	BiFunction<?, ?, ?> NULL = (value1, value2) -> null;

	/**
	 * Composes a function that applies the specified functions to each parameter as inputs to this function.
	 * @param <V1> the first parameter function type
	 * @param <V2> the second parameter function type
	 * @param before1 the function applied to the first parameter
	 * @param before2 the function applied to the second parameter
	 * @return a composed function that applies the specified functions to each parameter as inputs to this function.
	 */
	default <V1, V2> BiFunction<V1, V2, R> compose(java.util.function.Function<? super V1, ? extends T> before1, java.util.function.Function<? super V2, ? extends U> before2) {
		return new BiFunction<>() {
			@Override
			public R apply(V1 value1, V2 value2) {
				return BiFunction.this.apply(before1.apply(value1), before2.apply(value2));
			}
		};
	}

	/**
	 * Composes a unary function that applies the specified functions to its parameter as inputs to this function.
	 * @param <V> the parameter function type
	 * @param before1 the function applied to the first parameter
	 * @param before2 the function applied to the second parameter
	 * @return a composed function that applies the specified functions to its parameter as inputs to this function.
	 */
	default <V> Function<V, R> composeUnary(java.util.function.Function<? super V, ? extends T> before1, java.util.function.Function<? super V, ? extends U> before2) {
		return new Function<>() {
			@Override
			public R apply(V value) {
				return BiFunction.this.apply(before1.apply(value), before2.apply(value));
			}
		};
	}

	@Override
	default <V> BiFunction<T, U, V> andThen(java.util.function.Function<? super R, ? extends V> after) {
		return new BiFunction<>() {
			@Override
			public V apply(T value1, U value2) {
				return after.apply(BiFunction.this.apply(value1, value2));
			}
		};
	}

	/**
	 * Returns a function that processes this function with reversed parameter order.
	 * @return a function that processes this function with reversed parameter order.
	 */
	default BiFunction<U, T, R> reverse() {
		return new BiFunction<>() {
			@Override
			public R apply(U value2, T value1) {
				return BiFunction.this.apply(value1, value2);
			}
		};
	}

	/**
	 * Returns a function that applies this function to the values returned by the specified providers if its parameters do not match the specified predicates.
	 * @param predicate1 a predicate used to determine the first parameter of this function
	 * @param defaultValue1 a provider of the default value of the first parameter
	 * @param predicate2 a predicate used to determine the second parameter of this function
	 * @param defaultValue2 a provider of the default value of the second parameter
	 * @return a function that applies this function to the value returned by the specified provider if its value does not match the specified predicate.
	 */
	default BiFunction<T, U, R> withDefault(java.util.function.Predicate<T> predicate1, java.util.function.Supplier<T> defaultValue1, java.util.function.Predicate<U> predicate2, java.util.function.Supplier<U> defaultValue2) {
		return new BiFunction<>() {
			@Override
			public R apply(T value1, U value2) {
				return BiFunction.this.apply(predicate1.test(value1) ? value1 : defaultValue1.get(), predicate2.test(value2) ? value2 : defaultValue2.get());
			}
		};
	}

	/**
	 * Returns a function that applies this function if its parameters matches the specified predicate, or returns the value provided by the specified supplier otherwise.
	 * @param predicate a predicate used to determine the parameter of this function
	 * @param defaultResult a provider of the default parameter value
	 * @return a function that applies this function if its parameter matches the specified predicate, or returns the value provided by the specified supplier otherwise.
	 */
	default BiFunction<T, U, R> orDefault(java.util.function.BiPredicate<T, U> predicate, java.util.function.Supplier<R> defaultResult) {
		return new BiFunction<>() {
			@Override
			public R apply(T value1, U value2) {
				return predicate.test(value1, value2) ? BiFunction.this.apply(value1, value2) : defaultResult.get();
			}
		};
	}

	/**
	 * Returns a function that returns its first parameter.
	 * @param <T> the first parameter type
	 * @param <U> the second parameter type
	 * @param <R> the function return type
	 * @return a function that returns its first parameter.
	 */
	@SuppressWarnings("unchecked")
	static <T extends R, U, R> BiFunction<T, U, R> former() {
		return (BiFunction<T, U, R>) FORMER_IDENTITY;
	}

	/**
	 * Returns a function that returns its second parameter.
	 * @param <T> the first parameter type
	 * @param <U> the second parameter type
	 * @param <R> the function return type
	 * @return a function that returns its first parameter.
	 */
	@SuppressWarnings("unchecked")
	static <T, U extends R, R> BiFunction<T, U, R> latter() {
		return (BiFunction<T, U, R>) LATTER_IDENTITY;
	}

	/**
	 * Returns a function that always returns the specified value, ignoring its parameters.
	 * @param <T> the first parameter type
	 * @param <U> the second parameter type
	 * @param <R> the function return type
	 * @return a function that always returns the specified value, ignoring its parameters.
	 */
	@SuppressWarnings("unchecked")
	static <T, U, R> BiFunction<T, U, R> empty() {
		return (BiFunction<T, U, R>) NULL;
	}

	/**
	 * Returns a function that always returns the specified value, ignoring its parameter.
	 * @param <T> the first parameter type
	 * @param <U> the second parameter type
	 * @param <R> the function return type
	 * @param result the function result
	 * @return a function that always returns the specified value, ignoring its parameter.
	 */
	static <T, U, R> BiFunction<T, U, R> of(R result) {
		return (result != null) ? of(BiConsumer.empty(), Supplier.of(result)) : empty();
	}

	/**
	 * Returns a function that accepts its parameters via the specified consumer and returns the value returned by the specified supplier.
	 * @param <T> the former parameter type
	 * @param <U> the latter parameter type
	 * @param <R> the function return type
	 * @param consumer the consumer of the function parameter
	 * @param supplier the supplier of the function result
	 * @return a function that accepts its parameters via the specified consumer and returns the value returned by the specified supplier.
	 */
	static <T, U, R> BiFunction<T, U, R> of(java.util.function.BiConsumer<T, U> consumer, java.util.function.Supplier<R> supplier) {
		return new BiFunction<>() {
			@Override
			public R apply(T value1, U value2) {
				consumer.accept(value1, value2);
				return supplier.get();
			}
		};
	}

	/**
	 * Returns a function that applies the former parameter to the specified function.
	 * @param <T> the former parameter type
	 * @param <U> the latter parameter type
	 * @param <R> the function return type
	 * @param function the function applied to the former parameter
	 * @return a function that applies the former parameter to the specified function.
	 */
	static <T, U, R> BiFunction<T, U, R> applyFormer(java.util.function.Function<T, R> function) {
		return new BiFunction<>() {
			@Override
			public R apply(T value, U ignored) {
				return function.apply(value);
			}
		};
	}

	/**
	 * Returns a function that applies the latter parameter to the specified function.
	 * @param <T> the former parameter type
	 * @param <U> the latter parameter type
	 * @param <R> the function return type
	 * @param function the function applied to the latter parameter
	 * @return a function that applies the latter parameter to the specified function.
	 */
	static <T, U, R> BiFunction<T, U, R> applyLatter(java.util.function.Function<U, R> function) {
		return new BiFunction<>() {
			@Override
			public R apply(T ignored, U value) {
				return function.apply(value);
			}
		};
	}
}
