/*
 * Copyright The WildFly Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package org.wildfly.clustering.session.cache.metadata.fine;

import java.time.Duration;
import java.util.Map;

import org.wildfly.clustering.server.offset.Value;

/**
 * A mutable session access metadata.
 * @author Paul Ferraro
 */
public class MutableSessionAccessMetaData implements SessionAccessMetaData {

	private final ImmutableSessionAccessMetaData metaData;
	private final Value<Duration> sinceCreation;
	private final Value<Duration> lastAccess;

	/**
	 * Creates a mutable session access metadata.
	 * @param metaData the immutable session access metadata
	 * @param values the set of session access metadata values
	 */
	public MutableSessionAccessMetaData(ImmutableSessionAccessMetaData metaData, MutableSessionAccessMetaDataValues values) {
		this.metaData = metaData;
		this.sinceCreation = values.getSinceCreation();
		this.lastAccess = values.getLastAccess();
	}

	@Override
	public boolean isNew() {
		return this.metaData.isNew() && this.lastAccess.get().isZero();
	}

	@Override
	public Duration getSinceCreationDuration() {
		return this.sinceCreation.get();
	}

	@Override
	public Duration getLastAccessDuration() {
		return this.lastAccess.get();
	}

	@Override
	public void setLastAccessDuration(Duration sinceCreation, Duration lastAccess) {
		this.sinceCreation.set(sinceCreation);
		this.lastAccess.set(lastAccess);
	}

	@Override
	public String toString() {
		return Map.of("new", this.isNew(), "since-creation", this.getSinceCreationDuration(), "last-access", this.getLastAccessDuration()).toString();
	}
}
