/*
 * Copyright (C) 2007-2010 Júlio Vilmar Gesser.
 * Copyright (C) 2011, 2013-2016 The JavaParser Team.
 *
 * This file is part of JavaParser.
 *
 * JavaParser can be used either under the terms of
 * a) the GNU Lesser General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 * b) the terms of the Apache License
 *
 * You should have received a copy of both licenses in LICENCE.LGPL and
 * LICENCE.APACHE. Please refer to those files for details.
 *
 * JavaParser is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */
package org.drools.javaparser.ast.stmt;

import org.drools.javaparser.TokenRange;
import org.drools.javaparser.ast.AllFieldsConstructor;
import org.drools.javaparser.ast.Generated;
import org.drools.javaparser.ast.expr.Expression;
import org.drools.javaparser.ast.expr.NameExpr;
import org.drools.javaparser.ast.observer.ObservableProperty;
import org.drools.javaparser.ast.visitor.GenericVisitor;
import org.drools.javaparser.ast.visitor.VoidVisitor;
import org.drools.javaparser.metamodel.OptionalProperty;
import java.util.Optional;
import java.util.function.Consumer;
import org.drools.javaparser.ast.Node;
import org.drools.javaparser.ast.visitor.CloneVisitor;
import org.drools.javaparser.metamodel.BreakStmtMetaModel;
import org.drools.javaparser.metamodel.JavaParserMetaModel;

/**
 * <h1>The break statement</h1>
 * <h2>Java 1.0-11</h2>
 * Break has an optional label:
 * <br/><code>break;</code>
 * <br/><code>break somewhere;</code>
 * <br/>The label is in the "value" property as a NameExpr.
 * <h2>Java 12</h2>
 * Break can now also have any expression,
 * to be used in the switch-expression:
 * <br/><code>break 123+456;</code>
 * <br/><code>break "more or less";</code>
 * <br/>The expression will be in the "value" property.
 *
 * @author Julio Vilmar Gesser
 * @see org.drools.javaparser.ast.expr.SwitchExpr
 */
public final class BreakStmt extends Statement {

    @OptionalProperty
    private Expression value;

    public BreakStmt() {
        this(null, new NameExpr());
    }

    public BreakStmt(final String label) {
        this(null, new NameExpr(label));
    }

    @AllFieldsConstructor
    public BreakStmt(final Expression value) {
        this(null, value);
    }

    /**
     * This constructor is used by the parser and is considered private.
     */
    @Generated("org.drools.javaparser.generator.core.node.MainConstructorGenerator")
    public BreakStmt(TokenRange tokenRange, Expression value) {
        super(tokenRange);
        setValue(value);
        customInitialization();
    }

    @Override
    @Generated("org.drools.javaparser.generator.core.node.AcceptGenerator")
    public <R, A> R accept(final GenericVisitor<R, A> v, final A arg) {
        return v.visit(this, arg);
    }

    @Override
    @Generated("org.drools.javaparser.generator.core.node.AcceptGenerator")
    public <A> void accept(final VoidVisitor<A> v, final A arg) {
        v.visit(this, arg);
    }

    @Generated("org.drools.javaparser.generator.core.node.PropertyGenerator")
    public Optional<Expression> getValue() {
        return Optional.ofNullable(value);
    }

    /**
     * Sets the label
     *
     * @param value the label or the expression, can be null
     * @return this, the BreakStmt
     */
    @Generated("org.drools.javaparser.generator.core.node.PropertyGenerator")
    public BreakStmt setValue(final Expression value) {
        if (value == this.value) {
            return (BreakStmt) this;
        }
        notifyPropertyChange(ObservableProperty.VALUE, this.value, value);
        if (this.value != null)
            this.value.setParentNode(null);
        this.value = value;
        setAsParentNodeOf(value);
        return this;
    }

    @Override
    @Generated("org.drools.javaparser.generator.core.node.RemoveMethodGenerator")
    public boolean remove(Node node) {
        if (node == null)
            return false;
        if (value != null) {
            if (node == value) {
                removeValue();
                return true;
            }
        }
        return super.remove(node);
    }

    @Generated("org.drools.javaparser.generator.core.node.RemoveMethodGenerator")
    public BreakStmt removeValue() {
        return setValue((Expression) null);
    }

    @Override
    @Generated("org.drools.javaparser.generator.core.node.CloneGenerator")
    public BreakStmt clone() {
        return (BreakStmt) accept(new CloneVisitor(), null);
    }

    @Override
    @Generated("org.drools.javaparser.generator.core.node.GetMetaModelGenerator")
    public BreakStmtMetaModel getMetaModel() {
        return JavaParserMetaModel.breakStmtMetaModel;
    }

    @Override
    @Generated("org.drools.javaparser.generator.core.node.ReplaceMethodGenerator")
    public boolean replace(Node node, Node replacementNode) {
        if (node == null)
            return false;
        if (value != null) {
            if (node == value) {
                setValue((Expression) replacementNode);
                return true;
            }
        }
        return super.replace(node, replacementNode);
    }

    @Override
    @Generated("org.drools.javaparser.generator.core.node.TypeCastingGenerator")
    public boolean isBreakStmt() {
        return true;
    }

    @Override
    @Generated("org.drools.javaparser.generator.core.node.TypeCastingGenerator")
    public BreakStmt asBreakStmt() {
        return this;
    }

    @Generated("org.drools.javaparser.generator.core.node.TypeCastingGenerator")
    public void ifBreakStmt(Consumer<BreakStmt> action) {
        action.accept(this);
    }

    @Override
    @Generated("org.drools.javaparser.generator.core.node.TypeCastingGenerator")
    public Optional<BreakStmt> toBreakStmt() {
        return Optional.of(this);
    }
}
