/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * Copyright (c) 2008, Red Hat Middleware LLC or third-party contributors as
 * indicated by the @author tags or express copyright attribution
 * statements applied by the authors.  All third-party contributions are
 * distributed under license by Red Hat Middleware LLC.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this distribution; if not, write to:
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301  USA
 */
package org.hibernate.envers.synchronization.work;

import java.io.Serializable;
import java.util.List;
import java.util.Map;

import org.hibernate.envers.configuration.AuditConfiguration;
import org.hibernate.envers.configuration.AuditEntitiesConfiguration;
import org.hibernate.envers.entities.mapper.PersistentCollectionChangeData;

import org.hibernate.Session;
import org.hibernate.persister.collection.AbstractCollectionPersister;
import org.hibernate.engine.CollectionEntry;
import org.hibernate.engine.SessionImplementor;
import org.hibernate.collection.PersistentCollection;

/**
 * @author Adam Warski (adam at warski dot org)
 */
public class PersistentCollectionChangeWorkUnit extends AbstractAuditWorkUnit implements AuditWorkUnit {
    private final List<PersistentCollectionChangeData> collectionChanges;
    private final String referencingPropertyName;

    public PersistentCollectionChangeWorkUnit(SessionImplementor sessionImplementor, String entityName,
											  AuditConfiguration verCfg, PersistentCollection collection,
											  CollectionEntry collectionEntry, Serializable snapshot, Serializable id) {
        super(sessionImplementor, entityName, verCfg, null);

		String ownerEntityName = ((AbstractCollectionPersister) collectionEntry.getLoadedPersister()).getOwnerEntityName();
		referencingPropertyName = collectionEntry.getRole().substring(ownerEntityName.length() + 1);

        collectionChanges = verCfg.getEntCfg().get(getEntityName()).getPropertyMapper()
                .mapCollectionChanges(referencingPropertyName, collection, snapshot, id);
    }

    public boolean containsWork() {
        return collectionChanges != null && collectionChanges.size() != 0;
    }

    @SuppressWarnings({"unchecked"})
    public void perform(Session session, Object revisionData) {
        AuditEntitiesConfiguration entitiesCfg = verCfg.getAuditEntCfg();

        for (PersistentCollectionChangeData persistentCollectionChangeData : collectionChanges) {
            // Setting the revision number
            ((Map<String, Object>) persistentCollectionChangeData.getData().get(entitiesCfg.getOriginalIdPropName()))
                    .put(entitiesCfg.getRevisionFieldName(), revisionData);

            session.save(persistentCollectionChangeData.getEntityName(), persistentCollectionChangeData.getData());
        }
    }

    public String getReferencingPropertyName() {
        return referencingPropertyName;
    }

    public List<PersistentCollectionChangeData> getCollectionChanges() {
        return collectionChanges;
    }

    public KeepCheckResult check(AddWorkUnit second) {
        return null;
    }

    public KeepCheckResult check(ModWorkUnit second) {
        return null;
    }

    public KeepCheckResult check(DelWorkUnit second) {
        return null;
    }

    public KeepCheckResult check(CollectionChangeWorkUnit second) {
        return null;
    }

    public KeepCheckResult dispatch(KeepCheckVisitor first) {
        return null;
    }
}
