/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * Copyright (c) 2008, 2013, Red Hat Inc. or third-party contributors as
 * indicated by the @author tags or express copyright attribution
 * statements applied by the authors.  All third-party contributions are
 * distributed under license by Red Hat Inc.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this distribution; if not, write to:
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301  USA
 */
package org.hibernate.envers.configuration.internal;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.hibernate.envers.configuration.EnversSettings;
import org.hibernate.envers.strategy.DefaultAuditStrategy;
import org.hibernate.internal.util.config.ConfigurationHelper;

/**
 * Configuration of versions entities - names of fields, entities and tables created to store versioning information.
 *
 * @author Adam Warski (adam at warski dot org)
 * @author Stephanie Pau at Markit Group Plc
 */
public class  AuditEntitiesConfiguration {
	private final String auditTablePrefix;
	private final String auditTableSuffix;

	private final String auditStrategyName;
	private final String originalIdPropName;

	private final String revisionFieldName;
	private final String revisionNumberPath;
	private final String revisionPropBasePath;

	private final String revisionTypePropName;
	private final String revisionTypePropType;

	private final String revisionInfoEntityName;

	private final Map<String, String> customAuditTablesNames;

	private final String revisionEndFieldName;

	private final boolean revisionEndTimestampEnabled;
	private final String revisionEndTimestampFieldName;

	private final String embeddableSetOrdinalPropertyName;

	public AuditEntitiesConfiguration(Properties properties, String revisionInfoEntityName) {
		this.revisionInfoEntityName = revisionInfoEntityName;

		auditTablePrefix = ConfigurationHelper.getString( EnversSettings.AUDIT_TABLE_PREFIX, properties, "" );
		auditTableSuffix = ConfigurationHelper.getString( EnversSettings.AUDIT_TABLE_SUFFIX, properties, "_AUD" );

		auditStrategyName = ConfigurationHelper.getString(
				EnversSettings.AUDIT_STRATEGY, properties, DefaultAuditStrategy.class.getName()
		);

		originalIdPropName = "originalId";

		revisionFieldName = ConfigurationHelper.getString( EnversSettings.REVISION_FIELD_NAME, properties, "REV" );

		revisionTypePropName = ConfigurationHelper.getString(
				EnversSettings.REVISION_TYPE_FIELD_NAME, properties, "REVTYPE"
		);
		revisionTypePropType = "byte";

		revisionEndFieldName = ConfigurationHelper.getString(
				EnversSettings.AUDIT_STRATEGY_VALIDITY_END_REV_FIELD_NAME, properties, "REVEND"
		);

		revisionEndTimestampEnabled = ConfigurationHelper.getBoolean(
				EnversSettings.AUDIT_STRATEGY_VALIDITY_STORE_REVEND_TIMESTAMP, properties, false
		);

		if ( revisionEndTimestampEnabled ) {
			revisionEndTimestampFieldName = ConfigurationHelper.getString(
					EnversSettings.AUDIT_STRATEGY_VALIDITY_REVEND_TIMESTAMP_FIELD_NAME, properties, "REVEND_TSTMP"
			);
		}
		else {
			revisionEndTimestampFieldName = null;
		}

		customAuditTablesNames = new HashMap<String, String>();

		revisionNumberPath = originalIdPropName + "." + revisionFieldName + ".id";
		revisionPropBasePath = originalIdPropName + "." + revisionFieldName + ".";

		embeddableSetOrdinalPropertyName = ConfigurationHelper.getString(
				EnversSettings.EMBEDDABLE_SET_ORDINAL_FIELD_NAME, properties, "SETORDINAL"
		);
	}

	public String getOriginalIdPropName() {
		return originalIdPropName;
	}

	public String getRevisionFieldName() {
		return revisionFieldName;
	}

	public boolean isRevisionEndTimestampEnabled() {
		return revisionEndTimestampEnabled;
	}

	public String getRevisionEndTimestampFieldName() {
		return revisionEndTimestampFieldName;
	}

	public String getRevisionNumberPath() {
		return revisionNumberPath;
	}

	/**
	 * @param propertyName Property of the revision entity.
	 *
	 * @return A path to the given property of the revision entity associated with an audit entity.
	 */
	public String getRevisionPropPath(String propertyName) {
		return revisionPropBasePath + propertyName;
	}

	public String getRevisionTypePropName() {
		return revisionTypePropName;
	}

	public String getRevisionTypePropType() {
		return revisionTypePropType;
	}

	public String getRevisionInfoEntityName() {
		return revisionInfoEntityName;
	}

	public void addCustomAuditTableName(String entityName, String tableName) {
		customAuditTablesNames.put( entityName, tableName );
	}

	public String getAuditEntityName(String entityName) {
		return auditTablePrefix + entityName + auditTableSuffix;
	}

	public String getAuditTableName(String entityName, String tableName) {
		final String customHistoryTableName = customAuditTablesNames.get( entityName );
		if ( customHistoryTableName == null ) {
			return auditTablePrefix + tableName + auditTableSuffix;
		}

		return customHistoryTableName;
	}

	public String getAuditStrategyName() {
		return auditStrategyName;
	}

	public String getRevisionEndFieldName() {
		return revisionEndFieldName;
	}

	public String getEmbeddableSetOrdinalPropertyName() {
		return embeddableSetOrdinalPropertyName;
	}
}
