/*
 * Hibernate Search, full-text search for your domain model
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.search.elasticsearch.schema.impl;

import org.hibernate.search.elasticsearch.schema.impl.model.IndexMetadata;

/**
 * An object responsible for validating type mappings retrieved from an existing Elasticsearch instance
 * against Hibernate Search-generated type mappings.
 *
 * @author Yoann Rodiere
 */
public interface ElasticsearchSchemaValidator {

	/**
	 * Retrieves and validate actual index metadata, throwing an exception if validation fails.
	 * <p>
	 * The metadata mainly contain the type mappings and the index settings.
	 *
	 * @param expectedIndexMetadata The expected metadata, generated by Hibernate Search.
	 * @param executionOptions The execution options, giving more context information.
	 * @throws ElasticsearchSchemaValidationException If a validation error occurs.
	 */
	void validate(IndexMetadata expectedIndexMetadata, ExecutionOptions executionOptions);

	/**
	 * Retrieves and validate actual index settings, returning a boolean indicating
	 * whether the actual settings are valid or not.
	 *
	 * @param expectedIndexMetadata The expected metadata, generated by Hibernate Search.
	 * @param executionOptions The execution options, giving more context information.
	 * @return {@code true} if the actual settings are valid, {@code false} otherwise.
	 * @throws ElasticsearchSchemaValidationException If a validation error occurs.
	 */
	boolean isSettingsValid(IndexMetadata expectedIndexMetadata, ExecutionOptions executionOptions);

}
