/*
 * Decompiled with CFR 0.152.
 */
package org.hibernate.search.elasticsearch.test;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.StandardCharsets;
import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.http.nio.ContentEncoder;
import org.apache.http.nio.IOControl;
import org.hibernate.search.elasticsearch.impl.JsonBuilder;
import org.hibernate.search.elasticsearch.util.impl.GsonHttpEntity;
import org.hibernate.search.testsupport.TestForIssue;
import org.junit.Assert;
import org.junit.Test;

@TestForIssue(jiraKey="HSEARCH-2818")
public class GsonStreamedEncodingTest {
    public static final int MAX_TESTING_BUFFER_BYTES = 4000;
    private static final String JSON_TEST_PAYLOAD_VERSION = "{\"version\":{\"number\":\"5.5.0\"}}\n";
    private static final String JSON_TEST_PAYLOAD_EMPTY = "{}\n";
    private static final int BULK_BATCH_SIZE = 100;
    private static final String JSON_TEST_PAYLOAD_LARGE_BULK = GsonStreamedEncodingTest.produceLargeBukJSONContent();
    private static final Gson gson = new Gson();

    @Test
    public void testEmptyJSON() {
        List<JsonObject> list = Collections.singletonList(GsonStreamedEncodingTest.buildEmptyJSON());
        this.verifyProducedContent(list);
        this.verifySha256Signature(list);
        this.verifyOutput(JSON_TEST_PAYLOAD_EMPTY, list);
    }

    @Test
    public void testSinglePropertyJSON() {
        List<JsonObject> list = Collections.singletonList(GsonStreamedEncodingTest.buildVersionJSON());
        this.verifyProducedContent(list);
        this.verifySha256Signature(list);
        this.verifyOutput(JSON_TEST_PAYLOAD_VERSION, list);
    }

    @Test
    public void testTripleBulkJSON() {
        ArrayList<JsonObject> list = new ArrayList<JsonObject>(3);
        list.add(GsonStreamedEncodingTest.buildEmptyJSON());
        list.add(GsonStreamedEncodingTest.buildVersionJSON());
        list.add(GsonStreamedEncodingTest.buildEmptyJSON());
        this.verifyProducedContent(list);
        this.verifySha256Signature(list);
        this.verifyOutput("{}\n{\"version\":{\"number\":\"5.5.0\"}}\n{}\n", list);
    }

    @Test
    public void testHugeBulkJSON() {
        List<JsonObject> list = GsonStreamedEncodingTest.produceLargeBulkJSON();
        this.verifyProducedContent(list);
        this.verifySha256Signature(list);
        this.verifyOutput(JSON_TEST_PAYLOAD_LARGE_BULK, list);
    }

    @Test
    @TestForIssue(jiraKey="HSEARCH-2886")
    public void testSplitUnicodeSurrogatePair() {
        List<JsonObject> list = GsonStreamedEncodingTest.produceUnicodeSplitSurrogatePairJSON();
        this.verifyProducedContent(list);
        this.verifySha256Signature(list);
    }

    @Test
    public void testContentIsRepeatable() {
        ArrayList<JsonObject> list = new ArrayList<JsonObject>(3);
        list.add(GsonStreamedEncodingTest.buildEmptyJSON());
        list.add(GsonStreamedEncodingTest.buildVersionJSON());
        list.add(GsonStreamedEncodingTest.buildEmptyJSON());
        try (GsonHttpEntity entity = new GsonHttpEntity(gson, list);){
            byte[] productionOne = this.produceContentWithCustomEncoder(entity);
            entity.close();
            byte[] productionTwo = this.produceContentWithCustomEncoder(entity);
            Assert.assertArrayEquals((byte[])productionOne, (byte[])productionTwo);
        }
        catch (IOException e) {
            throw new RuntimeException("We're mocking IO operations, this should not happen?", e);
        }
    }

    @Test
    public void testDigestToTriggerLengthComputation() {
        Throwable throwable;
        GsonHttpEntity entity;
        List<JsonObject> list = GsonStreamedEncodingTest.produceLargeBulkJSON();
        try {
            entity = new GsonHttpEntity(gson, list);
            throwable = null;
            try {
                Assert.assertEquals((long)-1L, (long)entity.getContentLength());
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (entity != null) {
                    if (throwable != null) {
                        try {
                            entity.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                    } else {
                        entity.close();
                    }
                }
            }
        }
        catch (IOException e) {
            throw new RuntimeException("We're mocking IO operations, this should not happen?", e);
        }
        try {
            entity = new GsonHttpEntity(gson, list);
            throwable = null;
            try {
                MessageDigest digest = DigestUtils.getSha256Digest();
                OutputStream discardingStream = new OutputStream(){

                    @Override
                    public void write(int b) throws IOException {
                    }
                };
                DigestOutputStream digestStream = new DigestOutputStream(discardingStream, digest);
                entity.writeTo((OutputStream)digestStream);
                Assert.assertNotEquals((long)-1L, (long)entity.getContentLength());
                byte[] content = this.produceContentWithCustomEncoder(entity);
                Assert.assertEquals((long)content.length, (long)entity.getContentLength());
            }
            catch (Throwable throwable4) {
                throwable = throwable4;
                throw throwable4;
            }
            finally {
                if (entity != null) {
                    if (throwable != null) {
                        try {
                            entity.close();
                        }
                        catch (Throwable throwable5) {
                            throwable.addSuppressed(throwable5);
                        }
                    } else {
                        entity.close();
                    }
                }
            }
        }
        catch (IOException e) {
            throw new RuntimeException("We're mocking IO operations, this should not happen?", e);
        }
    }

    @Test
    public void testShortMessageIsNotChunked() {
        ArrayList<JsonObject> list = new ArrayList<JsonObject>(3);
        list.add(GsonStreamedEncodingTest.buildEmptyJSON());
        list.add(GsonStreamedEncodingTest.buildVersionJSON());
        list.add(GsonStreamedEncodingTest.buildEmptyJSON());
        byte[] traditionalEncoding = this.traditionalEncoding(list);
        try (GsonHttpEntity entity = new GsonHttpEntity(gson, list);){
            Assert.assertEquals((long)traditionalEncoding.length, (long)entity.getContentLength());
        }
        catch (IOException e) {
            throw new RuntimeException("We're mocking IO operations, this should not happen?", e);
        }
    }

    private void verifyOutput(String expected, List<JsonObject> list) {
        Assert.assertEquals((Object)expected, (Object)this.encodeToString(list));
    }

    private void verifySha256Signature(List<JsonObject> jsonObjects) {
        String optimisedEncoding = this.optimisedSha256(jsonObjects);
        String traditionalEncoding = this.traditionalSha256(jsonObjects);
        Assert.assertEquals((String)"SHA-256 signatures not matching", (Object)traditionalEncoding, (Object)optimisedEncoding);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private String optimisedSha256(List<JsonObject> bodyParts) {
        this.notEmpty(bodyParts);
        try (GsonHttpEntity entity = new GsonHttpEntity(gson, bodyParts);){
            MessageDigest digest = DigestUtils.getSha256Digest();
            OutputStream discardingStream = new OutputStream(){

                @Override
                public void write(int b) throws IOException {
                }
            };
            DigestOutputStream digestStream = new DigestOutputStream(discardingStream, digest);
            entity.writeTo((OutputStream)digestStream);
            String string = Hex.encodeHexString((byte[])digest.digest());
            return string;
        }
        catch (IOException e) {
            throw new RuntimeException("We're mocking IO operations, this should not happen?", e);
        }
    }

    private String traditionalSha256(List<JsonObject> jsonObjects) {
        return DigestUtils.sha256Hex((byte[])this.traditionalEncoding(jsonObjects));
    }

    private void verifyProducedContent(List<JsonObject> jsonObjects) {
        byte[] optimised;
        byte[] expected = this.traditionalEncoding(jsonObjects);
        if (!Arrays.equals(expected, optimised = this.optimisedEncoding(jsonObjects))) {
            CharBuffer decodedExpected = StandardCharsets.UTF_8.decode(ByteBuffer.wrap(expected));
            CharBuffer decodedOptimised = StandardCharsets.UTF_8.decode(ByteBuffer.wrap(optimised));
            System.out.println("Rendered :\n" + decodedOptimised);
            System.out.println("Should be:\n" + decodedExpected);
        }
        Assert.assertArrayEquals((byte[])expected, (byte[])optimised);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private byte[] optimisedEncoding(List<JsonObject> bodyParts) {
        this.notEmpty(bodyParts);
        try (GsonHttpEntity entity = new GsonHttpEntity(gson, bodyParts);){
            byte[] firstRun = this.produceContentWithCustomEncoder(entity);
            entity.close();
            byte[] secondRun = this.produceContentWithCustomEncoder(entity);
            Assert.assertArrayEquals((String)"Being repeatable, we expect it to be able to reproduce all content even after being closed", (byte[])firstRun, (byte[])secondRun);
            byte[] byArray = secondRun;
            return byArray;
        }
        catch (IOException e) {
            throw new RuntimeException("We're mocking IO operations, this should not happen?", e);
        }
    }

    private byte[] produceContentWithCustomEncoder(GsonHttpEntity entity) throws IOException {
        FakeIOControl fakeIO = new FakeIOControl();
        HeapContentEncoder sink = new HeapContentEncoder();
        int loopCounter = 0;
        while (!sink.isCompleted()) {
            entity.produceContent((ContentEncoder)sink, (IOControl)fakeIO);
            sink.setNextAcceptedBytesSize(loopCounter++ % 3);
        }
        return sink.flipAndRead();
    }

    private void notEmpty(List<JsonObject> bodyParts) {
        Assert.assertFalse((String)"Pointless to test this, we don't use this strategy for empty blocks", (boolean)bodyParts.isEmpty());
    }

    byte[] traditionalEncoding(List<JsonObject> bodyParts) {
        return this.encodeToString(bodyParts).getBytes(StandardCharsets.UTF_8);
    }

    private String encodeToString(List<JsonObject> bodyParts) {
        this.notEmpty(bodyParts);
        StringBuilder builder = new StringBuilder();
        for (JsonObject bodyPart : bodyParts) {
            gson.toJson((JsonElement)bodyPart, (Appendable)builder);
            builder.append('\n');
        }
        return builder.toString();
    }

    private static List<JsonObject> produceLargeBulkJSON() {
        ArrayList<JsonObject> list = new ArrayList<JsonObject>(100);
        for (int i = 0; i < 100; ++i) {
            list.add(GsonStreamedEncodingTest.buildVersionJSON());
        }
        return list;
    }

    private static List<JsonObject> produceUnicodeSplitSurrogatePairJSON() {
        String splitObjectPrefix = "{\"p\":\"";
        String surrogatePair = "\ud802\udd04";
        String splitObjectSuffix = "\"}";
        int splitIndex = 1024;
        int lengthToFill = splitIndex - splitObjectPrefix.length() - 1;
        ArrayList<JsonObject> list = new ArrayList<JsonObject>();
        StringBuilder sb = new StringBuilder();
        sb.append(splitObjectPrefix);
        for (int i = 0; i < lengthToFill; ++i) {
            sb.append('a');
        }
        sb.append(surrogatePair);
        sb.append(splitObjectSuffix);
        JsonObject splitObject = (JsonObject)gson.fromJson(sb.toString(), JsonObject.class);
        list.add(splitObject);
        return list;
    }

    private static JsonObject buildEmptyJSON() {
        return JsonBuilder.object().build();
    }

    private static JsonObject buildVersionJSON() {
        return JsonBuilder.object().add("version", JsonBuilder.object().addProperty("number", "5.5.0")).build();
    }

    private static String produceLargeBukJSONContent() {
        StringBuilder content = new StringBuilder(100 * JSON_TEST_PAYLOAD_VERSION.length());
        for (int i = 0; i < 100; ++i) {
            content.append(JSON_TEST_PAYLOAD_VERSION);
        }
        return content.toString();
    }

    private static final class FakeIOControl
    implements IOControl {
        private FakeIOControl() {
        }

        public void requestInput() {
            Assert.fail((String)"Should not invoke this");
        }

        public void suspendInput() {
            Assert.fail((String)"Should not invoke this");
        }

        public void requestOutput() {
            Assert.fail((String)"Should not invoke this");
        }

        public void suspendOutput() {
            Assert.fail((String)"Should not invoke this");
        }

        public void shutdown() throws IOException {
            Assert.fail((String)"Should not invoke this");
        }
    }

    private static final class HeapContentEncoder
    implements ContentEncoder {
        private boolean contentComplete = false;
        private int nextWriteAcceptLimit = 0;
        private ByteBuffer buf = ByteBuffer.allocate(4000);
        private boolean lastWriteWasZeroLength = false;
        private boolean closed = false;

        private HeapContentEncoder() {
        }

        public int write(ByteBuffer byteBuffer) throws IOException {
            Assert.assertFalse((boolean)this.closed);
            this.lastWriteWasZeroLength = !byteBuffer.hasRemaining();
            int toRead = Math.min(byteBuffer.remaining(), this.nextWriteAcceptLimit);
            byte[] currentRead = new byte[toRead];
            byteBuffer.get(currentRead);
            this.buf.put(currentRead);
            return toRead;
        }

        public void complete() throws IOException {
            Assert.assertFalse((String)"A final zero-length write was detected - this should never happen. See HSEARCH-2854.", (boolean)this.lastWriteWasZeroLength);
            Assert.assertFalse((boolean)this.closed);
            Assert.assertFalse((String)"Can't mark it 'complete' multiple times", (boolean)this.contentComplete);
            this.contentComplete = true;
        }

        public boolean isCompleted() {
            return this.contentComplete;
        }

        public byte[] flipAndRead() {
            Assert.assertFalse((String)"can read the buffer only once", (boolean)this.closed);
            this.closed = true;
            this.buf.flip();
            byte[] currentRead = new byte[this.buf.remaining()];
            this.buf.get(currentRead);
            return currentRead;
        }

        public void setNextAcceptedBytesSize(int size) {
            this.nextWriteAcceptLimit = size;
        }
    }
}

