// $Id: DecimalMaxValidatorForNumber.java 17421 2009-08-26 12:25:39Z hardy.ferentschik $
/*
* JBoss, Home of Professional Open Source
* Copyright 2008, Red Hat Middleware LLC, and individual contributors
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
* http://www.apache.org/licenses/LICENSE-2.0
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
package org.hibernate.validator.constraints.impl;

import java.math.BigDecimal;
import java.math.BigInteger;
import javax.validation.ConstraintDeclarationException;
import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;
import javax.validation.constraints.DecimalMax;

/**
 * Check that the number being validated is less than or equal to the maximum
 * value specified.
 *
 * @author Hardy Ferentschik
 */
public class DecimalMaxValidatorForNumber implements ConstraintValidator<DecimalMax, Number> {

	private BigDecimal maxValue;

	public void initialize(DecimalMax maxValue) {
		try {
			this.maxValue = new BigDecimal( maxValue.value() );
		}
		catch ( NumberFormatException nfe ) {
			throw new ConstraintDeclarationException( maxValue.value() + " does not represent a valid BigDemcimal formt" );
		}
	}

	public boolean isValid(Number value, ConstraintValidatorContext constraintValidatorContext) {

		//null values are valid
		if ( value == null ) {
			return true;
		}

		if ( value instanceof BigDecimal ) {
			return ( ( BigDecimal ) value ).compareTo( maxValue ) != 1;
		}
		else if ( value instanceof BigInteger ) {
			return ( new BigDecimal( ( BigInteger ) value ) ).compareTo( maxValue ) != 1;
		}
		else {
			return ( new BigDecimal( value.doubleValue() ).compareTo( maxValue ) ) != 1;
		}
	}
}