/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.loaders.jdbc;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Arrays;
import java.util.Locale;
import org.infinispan.config.ConfigurationException;
import org.infinispan.loaders.CacheLoaderException;
import org.infinispan.loaders.jdbc.DatabaseType;
import org.infinispan.loaders.jdbc.JdbcUtil;
import org.infinispan.loaders.jdbc.connectionfactory.ConnectionFactory;
import org.infinispan.loaders.jdbc.logging.Log;
import org.infinispan.util.logging.LogFactory;

public class TableManipulation
implements Cloneable {
    private static final Log log = (Log)LogFactory.getLog(TableManipulation.class, Log.class);
    public static final int DEFAULT_FETCH_SIZE = 100;
    public static final int DEFAULT_BATCH_SIZE = 100;
    private String idColumnName;
    private String idColumnType;
    private String tableName;
    private String tableNamePrefix;
    private String cacheName;
    private String dataColumnName;
    private String dataColumnType;
    private String timestampColumnName;
    private String timestampColumnType;
    private int fetchSize = 100;
    private int batchSize = 100;
    boolean createTableOnStart = true;
    boolean dropTableOnExit = false;
    private ConnectionFactory connectionFactory;
    private String insertRowSql;
    private String updateRowSql;
    private String selectRowSql;
    private String deleteRowSql;
    private String loadAllRowsSql;
    private String loadAllNonExpiredRowsSql;
    private String deleteAllRows;
    private String selectExpiredRowsSql;
    private String deleteExpiredRowsSql;
    private String loadSomeRowsSql;
    public DatabaseType databaseType;
    private String loadAllKeysBinarySql;
    private String loadAllKeysStringSql;

    public TableManipulation(String idColumnName, String idColumnType, String tableNamePrefix, String dataColumnName, String dataColumnType, String timestampColumnName, String timestampColumnType) {
        this.idColumnName = idColumnName;
        this.idColumnType = idColumnType;
        this.tableNamePrefix = tableNamePrefix;
        this.dataColumnName = dataColumnName;
        this.dataColumnType = dataColumnType;
        this.timestampColumnName = timestampColumnName;
        this.timestampColumnType = timestampColumnType;
    }

    public TableManipulation() {
    }

    public boolean tableExists(Connection connection, String tableName) throws CacheLoaderException {
        boolean bl;
        this.assertNotNull(this.getTableName(), "table name is mandatory");
        ResultSet rs = null;
        try {
            DatabaseMetaData dmd = connection.getMetaData();
            String catalog = connection.getCatalog();
            String schema = null;
            String quote = dmd.getIdentifierQuoteString();
            if (tableName.startsWith(quote)) {
                if (!tableName.endsWith(quote)) {
                    throw new IllegalStateException("Mismatched quote in table name: " + tableName);
                }
                int quoteLength = quote.length();
                tableName = tableName.substring(quoteLength, tableName.length() - quoteLength);
                if (dmd.storesLowerCaseQuotedIdentifiers()) {
                    tableName = TableManipulation.toLowerCase(tableName);
                } else if (dmd.storesUpperCaseQuotedIdentifiers()) {
                    tableName = TableManipulation.toUpperCase(tableName);
                }
            } else if (dmd.storesLowerCaseIdentifiers()) {
                tableName = TableManipulation.toLowerCase(tableName);
            } else if (dmd.storesUpperCaseIdentifiers()) {
                tableName = TableManipulation.toUpperCase(tableName);
            }
            int dotIndex = tableName.indexOf(46);
            if (dotIndex != -1) {
                schema = tableName.substring(0, dotIndex);
                tableName = tableName.substring(dotIndex + 1);
            }
            rs = dmd.getTables(catalog, schema, tableName, null);
            bl = rs.next();
        }
        catch (SQLException e) {
            try {
                throw new CacheLoaderException("Error while checking if table already exists " + tableName, (Throwable)e);
            }
            catch (Throwable throwable) {
                JdbcUtil.safeClose(rs);
                throw throwable;
            }
        }
        JdbcUtil.safeClose(rs);
        return bl;
    }

    public void createTable(Connection conn) throws CacheLoaderException {
        this.assertMandatoryElementsPresent();
        String createTableDdl = "CREATE TABLE " + this.getTableName() + "(" + this.idColumnName + " " + this.idColumnType + " NOT NULL, " + this.dataColumnName + " " + this.dataColumnType + ", " + this.timestampColumnName + " " + this.timestampColumnType + ", PRIMARY KEY (" + this.idColumnName + "))";
        if (log.isTraceEnabled()) {
            log.tracef("Creating table with following DDL: '%s'.", createTableDdl);
        }
        this.executeUpdateSql(conn, createTableDdl);
    }

    private void assertMandatoryElementsPresent() throws CacheLoaderException {
        this.assertNotNull(this.idColumnType, "idColumnType needed in order to create table");
        this.assertNotNull(this.idColumnName, "idColumnName needed in order to create table");
        this.assertNotNull(this.tableNamePrefix, "tableNamePrefix needed in order to create table");
        this.assertNotNull(this.cacheName, "cacheName needed in order to create table");
        this.assertNotNull(this.dataColumnName, "dataColumnName needed in order to create table");
        this.assertNotNull(this.dataColumnType, "dataColumnType needed in order to create table");
        this.assertNotNull(this.timestampColumnName, "timestampColumnName needed in order to create table");
        this.assertNotNull(this.timestampColumnType, "timestampColumnType needed in order to create table");
    }

    private void assertNotNull(String keyColumnType, String message) throws CacheLoaderException {
        if (keyColumnType == null || keyColumnType.trim().length() == 0) {
            throw new CacheLoaderException(message);
        }
    }

    private void executeUpdateSql(Connection conn, String sql) throws CacheLoaderException {
        Statement statement = null;
        try {
            statement = conn.createStatement();
            statement.executeUpdate(sql);
        }
        catch (SQLException e) {
            log.errorCreatingTable(sql, e);
            throw new CacheLoaderException((Throwable)e);
        }
        finally {
            JdbcUtil.safeClose(statement);
        }
    }

    public void dropTable(Connection conn) throws CacheLoaderException {
        String dropTableDdl = "DROP TABLE " + this.getTableName();
        String clearTable = "DELETE FROM " + this.getTableName();
        this.executeUpdateSql(conn, clearTable);
        if (log.isTraceEnabled()) {
            log.tracef("Dropping table with following DDL '%s'", dropTableDdl);
        }
        this.executeUpdateSql(conn, dropTableDdl);
    }

    private static String toLowerCase(String s) {
        return s.toLowerCase(Locale.ENGLISH);
    }

    private static String toUpperCase(String s) {
        return s.toUpperCase(Locale.ENGLISH);
    }

    public void setIdColumnName(String idColumnName) {
        this.idColumnName = idColumnName;
    }

    public void setIdColumnType(String idColumnType) {
        this.idColumnType = idColumnType;
    }

    public void setTableNamePrefix(String tableNamePrefix) {
        this.tableNamePrefix = tableNamePrefix;
    }

    public void setDataColumnName(String dataColumnName) {
        this.dataColumnName = dataColumnName;
    }

    public void setDataColumnType(String dataColumnType) {
        this.dataColumnType = dataColumnType;
    }

    public void setTimestampColumnName(String timestampColumnName) {
        this.timestampColumnName = timestampColumnName;
    }

    public void setTimestampColumnType(String timestampColumnType) {
        this.timestampColumnType = timestampColumnType;
    }

    public boolean isCreateTableOnStart() {
        return this.createTableOnStart;
    }

    public void setCreateTableOnStart(boolean createTableOnStart) {
        this.createTableOnStart = createTableOnStart;
    }

    public boolean isDropTableOnExit() {
        return this.dropTableOnExit;
    }

    public void setDropTableOnExit(boolean dropTableOnExit) {
        this.dropTableOnExit = dropTableOnExit;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void start(ConnectionFactory connectionFactory) throws CacheLoaderException {
        this.connectionFactory = connectionFactory;
        if (this.isCreateTableOnStart()) {
            Connection conn = this.connectionFactory.getConnection();
            try {
                if (!this.tableExists(conn, this.getTableName())) {
                    this.createTable(conn);
                }
            }
            finally {
                this.connectionFactory.releaseConnection(conn);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stop() throws CacheLoaderException {
        if (this.isDropTableOnExit()) {
            Connection conn = this.connectionFactory.getConnection();
            try {
                this.dropTable(conn);
            }
            finally {
                this.connectionFactory.releaseConnection(conn);
            }
        }
    }

    public String getInsertRowSql() {
        if (this.insertRowSql == null) {
            this.insertRowSql = "INSERT INTO " + this.getTableName() + " (" + this.dataColumnName + ", " + this.timestampColumnName + ", " + this.idColumnName + ") VALUES(?,?,?)";
        }
        return this.insertRowSql;
    }

    public String getUpdateRowSql() {
        if (this.updateRowSql == null) {
            this.updateRowSql = "UPDATE " + this.getTableName() + " SET " + this.dataColumnName + " = ? , " + this.timestampColumnName + "=? WHERE " + this.idColumnName + " = ?";
        }
        return this.updateRowSql;
    }

    public String getSelectRowSql() {
        if (this.selectRowSql == null) {
            this.selectRowSql = "SELECT " + this.idColumnName + ", " + this.dataColumnName + " FROM " + this.getTableName() + " WHERE " + this.idColumnName + " = ?";
        }
        return this.selectRowSql;
    }

    public String getDeleteRowSql() {
        if (this.deleteRowSql == null) {
            this.deleteRowSql = "DELETE FROM " + this.getTableName() + " WHERE " + this.idColumnName + " = ?";
        }
        return this.deleteRowSql;
    }

    public String getLoadNonExpiredAllRowsSql() {
        if (this.loadAllNonExpiredRowsSql == null) {
            this.loadAllNonExpiredRowsSql = "SELECT " + this.dataColumnName + "," + this.idColumnName + ", " + this.timestampColumnName + " FROM " + this.getTableName() + " WHERE " + this.timestampColumnName + " > ? OR " + this.timestampColumnName + " < 0";
        }
        return this.loadAllNonExpiredRowsSql;
    }

    public String getLoadAllRowsSql() {
        if (this.loadAllRowsSql == null) {
            this.loadAllRowsSql = "SELECT " + this.dataColumnName + "," + this.idColumnName + " FROM " + this.getTableName();
        }
        return this.loadAllRowsSql;
    }

    public String getDeleteAllRowsSql() {
        if (this.deleteAllRows == null) {
            this.deleteAllRows = "DELETE FROM " + this.getTableName();
        }
        return this.deleteAllRows;
    }

    public String getSelectExpiredRowsSql() {
        if (this.selectExpiredRowsSql == null) {
            this.selectExpiredRowsSql = this.getLoadAllRowsSql() + " WHERE " + this.timestampColumnName + "< ?";
        }
        return this.selectExpiredRowsSql;
    }

    public String getDeleteExpiredRowsSql() {
        if (this.deleteExpiredRowsSql == null) {
            this.deleteExpiredRowsSql = "DELETE FROM " + this.getTableName() + " WHERE " + this.timestampColumnName + "< ? AND " + this.timestampColumnName + "> 0";
        }
        return this.deleteExpiredRowsSql;
    }

    public TableManipulation clone() {
        try {
            return (TableManipulation)super.clone();
        }
        catch (CloneNotSupportedException e) {
            throw new IllegalStateException(e);
        }
    }

    public String getTableName() {
        if (this.tableName == null) {
            if (this.tableNamePrefix == null || this.cacheName == null) {
                throw new IllegalStateException("Both tableNamePrefix and cacheName must be non null at this point!");
            }
            this.tableName = this.tableNamePrefix + "_" + this.cacheName.replace(".", "_");
        }
        return this.tableName;
    }

    public String getTableNamePrefix() {
        return this.tableNamePrefix;
    }

    public boolean tableExists(Connection connection) throws CacheLoaderException {
        return this.tableExists(connection, this.tableName);
    }

    public String getIdColumnName() {
        return this.idColumnName;
    }

    public String getIdColumnType() {
        return this.idColumnType;
    }

    public String getDataColumnName() {
        return this.dataColumnName;
    }

    public String getDataColumnType() {
        return this.dataColumnType;
    }

    public String getTimestampColumnName() {
        return this.timestampColumnName;
    }

    public String getTimestampColumnType() {
        return this.timestampColumnType;
    }

    public int getFetchSize() {
        return this.fetchSize;
    }

    public void setFetchSize(int fetchSize) {
        this.fetchSize = fetchSize;
    }

    public int getBatchSize() {
        return this.batchSize;
    }

    public void setBatchSize(int batchSize) {
        this.batchSize = batchSize;
    }

    public void setCacheName(String cacheName) {
        this.cacheName = cacheName;
        this.tableName = null;
    }

    public boolean isVariableLimitSupported() {
        DatabaseType type = this.getDatabaseType();
        return type != DatabaseType.DB2_390;
    }

    public String getLoadSomeRowsSql() {
        if (this.loadSomeRowsSql == null) {
            switch (this.getDatabaseType()) {
                case ORACLE: {
                    this.loadSomeRowsSql = String.format("SELECT %s, %s FROM (SELECT %s, %s FROM %s) WHERE ROWNUM <= ?", this.dataColumnName, this.idColumnName, this.dataColumnName, this.idColumnName, this.getTableName());
                    break;
                }
                case DB2: 
                case DB2_390: {
                    this.loadSomeRowsSql = String.format("SELECT %s, %s FROM %s FETCH FIRST ? ROWS ONLY", this.dataColumnName, this.idColumnName, this.getTableName());
                    break;
                }
                case INFORMIX: 
                case INTERBASE: 
                case FIREBIRD: {
                    this.loadSomeRowsSql = String.format("SELECT FIRST ? %s, %s FROM %s", this.dataColumnName, this.idColumnName, this.getTableName());
                    break;
                }
                case SQL_SERVER: 
                case ACCESS: 
                case HSQL: {
                    this.loadSomeRowsSql = String.format("SELECT TOP ? %s, %s FROM %s", this.dataColumnName, this.idColumnName, this.getTableName());
                    break;
                }
                default: {
                    this.loadSomeRowsSql = String.format("SELECT %s, %s FROM %s LIMIT ?", this.dataColumnName, this.idColumnName, this.getTableName());
                }
            }
        }
        return this.loadSomeRowsSql;
    }

    public String getLoadAllKeysBinarySql() {
        if (this.loadAllKeysBinarySql == null) {
            this.loadAllKeysBinarySql = String.format("SELECT %s FROM %s", this.dataColumnName, this.getTableName());
        }
        return this.loadAllKeysBinarySql;
    }

    public String getLoadAllKeysStringSql() {
        if (this.loadAllKeysStringSql == null) {
            this.loadAllKeysStringSql = String.format("SELECT %s FROM %s", this.idColumnName, this.getTableName());
        }
        return this.loadAllKeysStringSql;
    }

    private DatabaseType getDatabaseType() {
        if (this.databaseType == null) {
            String dbProduct;
            try {
                dbProduct = this.connectionFactory.getConnection().getMetaData().getDatabaseProductName();
                this.databaseType = this.guessDatabaseType(dbProduct);
            }
            catch (Exception e) {
                log.debug("Unable to guess database type from JDBC metadata.", e);
            }
            if (this.databaseType == null) {
                log.debug("Unable to detect database type using connection metadata.  Attempting to guess on driver name.");
            }
            try {
                dbProduct = this.connectionFactory.getConnection().getMetaData().getDriverName();
                this.databaseType = this.guessDatabaseType(dbProduct);
            }
            catch (Exception e) {
                log.debug("Unable to guess database type from JDBC driver name.", e);
            }
            if (this.databaseType == null) {
                throw new ConfigurationException("Unable to detect database type from JDBC driver name or connection metadata.  Please provide this manually using the 'databaseType' property in your configuration.  Supported database type strings are " + Arrays.toString((Object[])DatabaseType.values()));
            }
            log.debugf("Guessing database type as '%s'.  If this is incorrect, please specify the correct type using the 'databaseType' property in your configuration.  Supported database type strings are %s", (Object)this.databaseType, Arrays.toString((Object[])DatabaseType.values()));
        }
        return this.databaseType;
    }

    private DatabaseType guessDatabaseType(String name) {
        DatabaseType type = null;
        if (name != null) {
            if (name.toLowerCase().contains("mysql")) {
                type = DatabaseType.MYSQL;
            } else if (name.toLowerCase().contains("postgres")) {
                type = DatabaseType.POSTGRES;
            } else if (name.toLowerCase().contains("derby")) {
                type = DatabaseType.DERBY;
            } else if (name.toLowerCase().contains("hsql") || name.toLowerCase().contains("hypersonic")) {
                type = DatabaseType.HSQL;
            } else if (name.toLowerCase().contains("h2")) {
                type = DatabaseType.H2;
            } else if (name.toLowerCase().contains("sqlite")) {
                type = DatabaseType.SQLITE;
            } else if (name.toLowerCase().contains("db2")) {
                type = DatabaseType.DB2;
            } else if (name.toLowerCase().contains("informix")) {
                type = DatabaseType.INFORMIX;
            } else if (name.toLowerCase().contains("interbase")) {
                type = DatabaseType.INTERBASE;
            } else if (name.toLowerCase().contains("firebird")) {
                type = DatabaseType.FIREBIRD;
            } else if (name.toLowerCase().contains("sqlserver") || name.toLowerCase().contains("microsoft")) {
                type = DatabaseType.SQL_SERVER;
            } else if (name.toLowerCase().contains("access")) {
                type = DatabaseType.ACCESS;
            } else if (name.toLowerCase().contains("oracle")) {
                type = DatabaseType.ORACLE;
            }
        }
        return type;
    }
}

