package org.infinispan.factories;

import org.infinispan.commons.equivalence.Equivalence;
import org.infinispan.commons.util.EntrySizeCalculator;
import org.infinispan.configuration.cache.EvictionConfiguration;
import org.infinispan.configuration.cache.MemoryConfiguration;
import org.infinispan.container.DataContainer;
import org.infinispan.container.DefaultDataContainer;
import org.infinispan.container.StorageType;
import org.infinispan.container.entries.PrimitiveEntrySizeCalculator;
import org.infinispan.container.offheap.OffHeapDataContainer;
import org.infinispan.eviction.EvictionType;
import org.infinispan.factories.annotations.DefaultFactoryFor;
import org.infinispan.marshall.core.WrappedByteArraySizeCalculator;

/**
 * Constructs the data container
 *
 * @author Manik Surtani (<a href="mailto:manik@jboss.org">manik@jboss.org</a>)
 * @author Vladimir Blagojevic
 * @since 4.0
 */
@DefaultFactoryFor(classes = DataContainer.class)
public class DataContainerFactory extends AbstractNamedCacheComponentFactory implements
         AutoInstantiableFactory {

   @Override
   @SuppressWarnings("unchecked")
   public <T> T construct(Class<T> componentType) {
      if (configuration.dataContainer().dataContainer() != null) {
         return (T) configuration.dataContainer().dataContainer();
      } else {
         int level = configuration.locking().concurrencyLevel();

         long thresholdSize = configuration.memory().size();


         //handle case when < 0 value signifies unbounded container
         if(thresholdSize < 0) {
            if (configuration.memory().storageType() == StorageType.OFF_HEAP) {
               return (T) new OffHeapDataContainer(configuration.memory().addressCount());
            } else {
               return (T) DefaultDataContainer.unBoundedDataContainer(level);
            }
         }

         DataContainer dataContainer;
         if (configuration.memory().storageType() == StorageType.OFF_HEAP) {
            if (configuration.eviction().type() == EvictionType.MEMORY) {
               // TODO: need to bound container
               dataContainer = new OffHeapDataContainer(configuration.memory().addressCount());
            } else {
               dataContainer = new OffHeapDataContainer(configuration.memory().addressCount());
            }
         } else {
            dataContainer = DefaultDataContainer.boundedDataContainer(level, thresholdSize,
                  configuration.eviction().type());
         }
         configuration.eviction().attributes().attribute(EvictionConfiguration.SIZE).addListener((newSize, old) ->
               configuration.memory().size(newSize.get()));
         configuration.memory().attributes().attribute(MemoryConfiguration.SIZE).addListener((newSize, old) ->
               dataContainer.resize(newSize.get()));
         return (T) dataContainer;
      }
   }
}
