/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.remoting.rpc;

import java.text.NumberFormat;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.infinispan.commands.CommandsFactory;
import org.infinispan.commands.ReplicableCommand;
import org.infinispan.commands.TopologyAffectedCommand;
import org.infinispan.commands.remote.CacheRpcCommand;
import org.infinispan.commons.CacheException;
import org.infinispan.configuration.cache.Configuration;
import org.infinispan.distribution.ch.ConsistentHash;
import org.infinispan.factories.annotations.Inject;
import org.infinispan.factories.annotations.Start;
import org.infinispan.jmx.JmxStatisticsExposer;
import org.infinispan.jmx.annotations.DataType;
import org.infinispan.jmx.annotations.DisplayType;
import org.infinispan.jmx.annotations.MBean;
import org.infinispan.jmx.annotations.ManagedAttribute;
import org.infinispan.jmx.annotations.ManagedOperation;
import org.infinispan.jmx.annotations.MeasurementType;
import org.infinispan.jmx.annotations.Parameter;
import org.infinispan.jmx.annotations.Units;
import org.infinispan.remoting.inboundhandler.DeliverOrder;
import org.infinispan.remoting.responses.Response;
import org.infinispan.remoting.rpc.ResponseMode;
import org.infinispan.remoting.rpc.RpcManager;
import org.infinispan.remoting.rpc.RpcOptions;
import org.infinispan.remoting.rpc.RpcOptionsBuilder;
import org.infinispan.remoting.transport.Address;
import org.infinispan.remoting.transport.Transport;
import org.infinispan.statetransfer.StateTransferManager;
import org.infinispan.topology.CacheTopology;
import org.infinispan.util.TimeService;
import org.infinispan.util.concurrent.CompletableFutures;
import org.infinispan.util.logging.Log;
import org.infinispan.util.logging.LogFactory;

@MBean(objectName="RpcManager", description="Manages all remote calls to remote cache instances in the cluster.")
public class RpcManagerImpl
implements RpcManager,
JmxStatisticsExposer {
    private static final Log log = LogFactory.getLog(RpcManagerImpl.class);
    private static final boolean trace = log.isTraceEnabled();
    private Transport t;
    private final AtomicLong replicationCount = new AtomicLong(0L);
    private final AtomicLong replicationFailures = new AtomicLong(0L);
    private final AtomicLong totalReplicationTime = new AtomicLong(0L);
    private boolean statisticsEnabled = false;
    private Configuration configuration;
    private CommandsFactory cf;
    private StateTransferManager stateTransferManager;
    private TimeService timeService;

    @Inject
    public void injectDependencies(Transport t, Configuration cfg, CommandsFactory cf, StateTransferManager stateTransferManager, TimeService timeService) {
        this.t = t;
        this.configuration = cfg;
        this.cf = cf;
        this.stateTransferManager = stateTransferManager;
        this.timeService = timeService;
    }

    @Start(priority=9)
    private void start() {
        this.statisticsEnabled = this.configuration.jmxStatistics().enabled();
        if (this.configuration.transaction().transactionProtocol().isTotalOrder()) {
            this.t.checkTotalOrderSupported();
        }
    }

    @ManagedAttribute(description="Retrieves the committed view.", displayName="Committed view", dataType=DataType.TRAIT)
    public String getCommittedViewAsString() {
        CacheTopology cacheTopology = this.stateTransferManager.getCacheTopology();
        if (cacheTopology == null) {
            return "N/A";
        }
        return cacheTopology.getCurrentCH().getMembers().toString();
    }

    @ManagedAttribute(description="Retrieves the pending view.", displayName="Pending view", dataType=DataType.TRAIT)
    public String getPendingViewAsString() {
        CacheTopology cacheTopology = this.stateTransferManager.getCacheTopology();
        if (cacheTopology == null) {
            return "N/A";
        }
        ConsistentHash pendingCH = cacheTopology.getPendingCH();
        return pendingCH != null ? pendingCH.getMembers().toString() : "null";
    }

    @Override
    public CompletableFuture<Map<Address, Response>> invokeRemotelyAsync(Collection<Address> recipients, ReplicableCommand rpc, RpcOptions options) {
        CompletableFuture<Map<Address, Response>> invocation;
        if (trace) {
            log.tracef("%s invoking %s to recipient list %s with options %s", this.t.getAddress(), rpc, recipients, options);
        }
        if (!this.configuration.clustering().cacheMode().isClustered()) {
            throw new IllegalStateException("Trying to invoke a remote command but the cache is not clustered");
        }
        this.setTopologyId(rpc);
        CacheRpcCommand cacheRpc = rpc instanceof CacheRpcCommand ? (CacheRpcCommand)rpc : this.cf.buildSingleRpcCommand(rpc);
        long startTimeNanos = this.statisticsEnabled ? this.timeService.time() : 0L;
        try {
            invocation = this.t.invokeRemotelyAsync(recipients, cacheRpc, options.responseMode(), options.timeUnit().toMillis(options.timeout()), options.responseFilter(), options.deliverOrder(), this.configuration.clustering().cacheMode().isDistributed());
        }
        catch (Exception e) {
            log.unexpectedErrorReplicating(e);
            if (this.statisticsEnabled) {
                this.replicationFailures.incrementAndGet();
            }
            return (CompletableFuture)this.rethrowAsCacheException(e);
        }
        return invocation.handle((responseMap, throwable) -> {
            if (this.statisticsEnabled) {
                long timeTaken = this.timeService.timeDuration(startTimeNanos, TimeUnit.MILLISECONDS);
                this.totalReplicationTime.getAndAdd(timeTaken);
            }
            if (throwable == null) {
                if (this.statisticsEnabled) {
                    this.replicationCount.incrementAndGet();
                }
                if (trace) {
                    log.tracef("Response(s) to %s is %s", (Object)rpc, responseMap);
                }
                return responseMap;
            }
            if (this.statisticsEnabled) {
                this.replicationFailures.incrementAndGet();
            }
            return (Map)this.rethrowAsCacheException((Throwable)throwable);
        });
    }

    private <T> T rethrowAsCacheException(Throwable throwable) {
        if (throwable.getCause() != null && throwable instanceof CompletionException) {
            throwable = throwable.getCause();
        }
        if (throwable instanceof CacheException) {
            log.trace("Replication exception", throwable);
            throw (CacheException)throwable;
        }
        log.unexpectedErrorReplicating(throwable);
        throw new CacheException(throwable);
    }

    @Override
    public Map<Address, Response> invokeRemotely(Collection<Address> recipients, ReplicableCommand rpc, RpcOptions options) {
        CompletableFuture<Map<Address, Response>> future = this.invokeRemotelyAsync(recipients, rpc, options);
        try {
            return CompletableFutures.await(future);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new CacheException("Thread interrupted while invoking RPC", e);
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof CacheException) {
                throw (CacheException)cause;
            }
            throw new CacheException("Unexpected exception replicating command", cause);
        }
    }

    @Override
    public Map<Address, Response> invokeRemotely(Map<Address, ReplicableCommand> rpcs, RpcOptions options) {
        if (trace) {
            log.tracef("%s invoking %s with options %s", (Object)this.t.getAddress(), (Object)rpcs, (Object)options);
        }
        if (!this.configuration.clustering().cacheMode().isClustered()) {
            throw new IllegalStateException("Trying to invoke a remote command but the cache is not clustered");
        }
        HashMap<Address, ReplicableCommand> replacedCommands = null;
        for (Map.Entry<Address, ReplicableCommand> entry : rpcs.entrySet()) {
            ReplicableCommand rpc = entry.getValue();
            this.setTopologyId(rpc);
            if (rpc instanceof CacheRpcCommand) continue;
            rpc = this.cf.buildSingleRpcCommand(rpc);
            if (replacedCommands == null) {
                replacedCommands = new HashMap<Address, ReplicableCommand>();
            }
            replacedCommands.put(entry.getKey(), rpc);
        }
        if (replacedCommands != null) {
            rpcs.putAll(replacedCommands);
        }
        long startTimeNanos = 0L;
        if (this.statisticsEnabled) {
            startTimeNanos = this.timeService.time();
        }
        try {
            Map<Address, Response> result = this.t.invokeRemotely(rpcs, options.responseMode(), options.timeUnit().toMillis(options.timeout()), options.responseFilter(), options.deliverOrder(), this.configuration.clustering().cacheMode().isDistributed());
            if (this.statisticsEnabled) {
                this.replicationCount.incrementAndGet();
            }
            if (trace) {
                log.tracef("Response(s) to %s is %s", (Object)rpcs, (Object)result);
            }
            Map<Address, Response> map = result;
            return map;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new CacheException("Thread interrupted while invoking RPC", e);
        }
        catch (CacheException e) {
            log.trace("replication exception: ", e);
            if (this.statisticsEnabled) {
                this.replicationFailures.incrementAndGet();
            }
            throw e;
        }
        catch (Throwable th) {
            log.unexpectedErrorReplicating(th);
            if (this.statisticsEnabled) {
                this.replicationFailures.incrementAndGet();
            }
            throw new CacheException(th);
        }
        finally {
            if (this.statisticsEnabled) {
                long timeTaken = this.timeService.timeDuration(startTimeNanos, TimeUnit.MILLISECONDS);
                this.totalReplicationTime.getAndAdd(timeTaken);
            }
        }
    }

    private CacheRpcCommand toCacheRpcCommand(ReplicableCommand command) {
        return command instanceof CacheRpcCommand ? (CacheRpcCommand)command : this.cf.buildSingleRpcCommand(command);
    }

    @Override
    public void sendTo(Address destination, ReplicableCommand command, DeliverOrder deliverOrder) {
        if (trace) {
            log.tracef("%s invoking %s to %s ordered by %s", new Object[]{this.t.getAddress(), command, destination, deliverOrder});
        }
        this.setTopologyId(command);
        CacheRpcCommand cacheRpc = this.toCacheRpcCommand(command);
        try {
            this.t.sendTo(destination, cacheRpc, deliverOrder);
        }
        catch (Exception e) {
            this.errorReplicating(e);
        }
    }

    @Override
    public void sendToMany(Collection<Address> destinations, ReplicableCommand command, DeliverOrder deliverOrder) {
        if (trace) {
            log.tracef("%s invoking %s to list %s ordered by %s", new Object[]{this.t.getAddress(), command, destinations, deliverOrder});
        }
        this.setTopologyId(command);
        CacheRpcCommand cacheRpc = this.toCacheRpcCommand(command);
        try {
            this.t.sendToMany(destinations, cacheRpc, deliverOrder);
        }
        catch (Exception e) {
            this.errorReplicating(e);
        }
    }

    private void errorReplicating(Exception e) {
        log.unexpectedErrorReplicating(e);
        if (this.statisticsEnabled) {
            this.replicationFailures.incrementAndGet();
        }
        this.rethrowAsCacheException(e);
    }

    @Override
    public Transport getTransport() {
        return this.t;
    }

    private void setTopologyId(ReplicableCommand command) {
        TopologyAffectedCommand topologyAffectedCommand;
        if (command instanceof TopologyAffectedCommand && (topologyAffectedCommand = (TopologyAffectedCommand)command).getTopologyId() == -1) {
            int currentTopologyId = this.stateTransferManager.getCacheTopology().getTopologyId();
            if (trace) {
                log.tracef("Topology id missing on command %s, setting it to %d", (Object)command, (Object)currentTopologyId);
            }
            topologyAffectedCommand.setTopologyId(currentTopologyId);
        }
    }

    @Override
    @ManagedOperation(description="Resets statistics gathered by this component", displayName="Reset statistics")
    public void resetStatistics() {
        this.replicationCount.set(0L);
        this.replicationFailures.set(0L);
        this.totalReplicationTime.set(0L);
    }

    @ManagedAttribute(description="Number of successful replications", displayName="Number of successful replications", measurementType=MeasurementType.TRENDSUP, displayType=DisplayType.SUMMARY)
    public long getReplicationCount() {
        if (!this.isStatisticsEnabled()) {
            return -1L;
        }
        return this.replicationCount.get();
    }

    @ManagedAttribute(description="Number of failed replications", displayName="Number of failed replications", measurementType=MeasurementType.TRENDSUP, displayType=DisplayType.SUMMARY)
    public long getReplicationFailures() {
        if (!this.isStatisticsEnabled()) {
            return -1L;
        }
        return this.replicationFailures.get();
    }

    @ManagedAttribute(description="Enables or disables the gathering of statistics by this component", displayName="Statistics enabled", dataType=DataType.TRAIT, writable=true)
    public boolean isStatisticsEnabled() {
        return this.statisticsEnabled;
    }

    @Override
    public boolean getStatisticsEnabled() {
        return this.isStatisticsEnabled();
    }

    @Override
    @Deprecated
    @ManagedOperation(displayName="Enable/disable statistics. Deprecated, use the statisticsEnabled attribute instead.")
    public void setStatisticsEnabled(@Parameter(name="enabled", description="Whether statistics should be enabled or disabled (true/false)") boolean statisticsEnabled) {
        this.statisticsEnabled = statisticsEnabled;
    }

    @ManagedAttribute(description="Successful replications as a ratio of total replications", displayName="Successful replications ratio")
    public String getSuccessRatio() {
        if (this.replicationCount.get() == 0L || !this.statisticsEnabled) {
            return "N/A";
        }
        double ration = this.calculateSuccessRatio() * 100.0;
        return NumberFormat.getInstance().format(ration) + "%";
    }

    @ManagedAttribute(description="Successful replications as a ratio of total replications in numeric double format", displayName="Successful replication ratio", units=Units.PERCENTAGE, displayType=DisplayType.SUMMARY)
    public double getSuccessRatioFloatingPoint() {
        if (this.replicationCount.get() == 0L || !this.statisticsEnabled) {
            return 0.0;
        }
        return this.calculateSuccessRatio();
    }

    private double calculateSuccessRatio() {
        double totalCount = this.replicationCount.get() + this.replicationFailures.get();
        return (double)this.replicationCount.get() / totalCount;
    }

    @ManagedAttribute(description="The average time spent in the transport layer, in milliseconds", displayName="Average time spent in the transport layer", units=Units.MILLISECONDS, displayType=DisplayType.SUMMARY)
    public long getAverageReplicationTime() {
        if (this.replicationCount.get() == 0L) {
            return 0L;
        }
        return this.totalReplicationTime.get() / this.replicationCount.get();
    }

    public void setTransport(Transport t) {
        this.t = t;
    }

    @Override
    public Address getAddress() {
        return this.t != null ? this.t.getAddress() : null;
    }

    @Override
    public int getTopologyId() {
        CacheTopology cacheTopology = this.stateTransferManager.getCacheTopology();
        return cacheTopology != null ? cacheTopology.getTopologyId() : -1;
    }

    @Override
    public RpcOptionsBuilder getRpcOptionsBuilder(ResponseMode responseMode) {
        return this.getRpcOptionsBuilder(responseMode, responseMode.isSynchronous() ? DeliverOrder.NONE : DeliverOrder.PER_SENDER);
    }

    @Override
    public RpcOptionsBuilder getRpcOptionsBuilder(ResponseMode responseMode, DeliverOrder deliverOrder) {
        return new RpcOptionsBuilder(this.configuration.clustering().remoteTimeout(), TimeUnit.MILLISECONDS, responseMode, deliverOrder);
    }

    @Override
    public RpcOptions getDefaultRpcOptions(boolean sync) {
        return this.getDefaultRpcOptions(sync, sync ? DeliverOrder.NONE : DeliverOrder.PER_SENDER);
    }

    @Override
    public RpcOptions getDefaultRpcOptions(boolean sync, DeliverOrder deliverOrder) {
        return this.getRpcOptionsBuilder(sync ? ResponseMode.SYNCHRONOUS : ResponseMode.ASYNCHRONOUS, deliverOrder).build();
    }

    @Override
    public List<Address> getMembers() {
        return this.stateTransferManager.getCacheTopology().getMembers();
    }
}

