package org.infinispan.query.partitionhandling;

import static org.infinispan.hibernate.search.spi.InfinispanIntegration.DEFAULT_INDEXESDATA_CACHENAME;
import static org.infinispan.hibernate.search.spi.InfinispanIntegration.DEFAULT_INDEXESMETADATA_CACHENAME;
import static org.infinispan.hibernate.search.spi.InfinispanIntegration.DEFAULT_LOCKING_CACHENAME;
import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertTrue;

import java.util.stream.IntStream;

import org.infinispan.Cache;
import org.infinispan.configuration.cache.CacheMode;
import org.infinispan.configuration.cache.Configuration;
import org.infinispan.configuration.cache.ConfigurationBuilder;
import org.infinispan.manager.EmbeddedCacheManager;
import org.infinispan.partitionhandling.AvailabilityException;
import org.infinispan.partitionhandling.BasePartitionHandlingTest;
import org.infinispan.partitionhandling.PartitionHandling;
import org.infinispan.protostream.SerializationContextInitializer;
import org.infinispan.query.Search;
import org.infinispan.query.dsl.IndexedQueryMode;
import org.infinispan.query.dsl.Query;
import org.infinispan.query.indexmanager.InfinispanIndexManager;
import org.infinispan.query.test.Person;
import org.infinispan.query.test.QueryTestSCI;
import org.testng.annotations.Test;

/**
 * @since 9.3
 */
@Test(groups = "functional", testName = "query.partitionhandling.SharedIndexTest")
public class SharedIndexTest extends BasePartitionHandlingTest {

   protected int totalEntries = 100;

   public SharedIndexTest() {
      numMembersInCluster = 3;
      cacheMode = CacheMode.DIST_SYNC;
      cleanup = CleanupPhase.AFTER_TEST;
   }

   @Override
   protected ConfigurationBuilder cacheConfiguration() {
      ConfigurationBuilder configurationBuilder = new ConfigurationBuilder();
      configurationBuilder.indexing()
                          .enable()
                          .addIndexedEntity(Person.class)
                          .addProperty("default.indexmanager", InfinispanIndexManager.class.getName());
      return configurationBuilder;
   }

   @Override
   protected SerializationContextInitializer serializationContextInitializer() {
      return QueryTestSCI.INSTANCE;
   }

   @Override
   protected void amendCacheManagerBeforeStart(EmbeddedCacheManager cm) {
      Configuration replConfig = new ConfigurationBuilder()
            .clustering().cacheMode(CacheMode.REPL_SYNC).partitionHandling()
            .whenSplit(PartitionHandling.DENY_READ_WRITES)
            .indexing().enabled(false).build();

      Configuration distConfig = new ConfigurationBuilder()
            .clustering().cacheMode(CacheMode.DIST_SYNC).partitionHandling()
            .whenSplit(PartitionHandling.DENY_READ_WRITES)
            .indexing().enabled(false).build();

      cm.defineConfiguration(DEFAULT_LOCKING_CACHENAME, replConfig);
      cm.defineConfiguration(DEFAULT_INDEXESDATA_CACHENAME, distConfig);
      cm.defineConfiguration(DEFAULT_INDEXESMETADATA_CACHENAME, replConfig);
   }

   @Test(expectedExceptions = AvailabilityException.class)
   public void shouldThrowExceptionInDegradedMode() {
      Cache<Integer, Person> cache = cache(0);
      IntStream.range(0, totalEntries).forEach(i -> cache.put(i, new Person("Person " + i, "", i)));

      executeQueries();

      splitCluster(new int[]{0}, new int[]{1, 2});
      partition(0).assertDegradedMode();

      executeQueries();
   }

   protected void assertAllNodesQueryResults(int results) {
      assertEquals(results, totalEntries);
   }

   protected void assertSingleNodeQueryResults(int results) {
      assertTrue(results > 0);
   }

   private void executeQueries() {
      String q = getQuery();
      caches().forEach(c -> {
         Query allNodesQuery = Search.getQueryFactory(c).create(q, getIndexedQueryMode());
         assertAllNodesQueryResults(allNodesQuery.getResultSize());
      });
      Query singleNodeQuery = Search.getQueryFactory(cache(0)).create(q);
      assertSingleNodeQueryResults(singleNodeQuery.list().size());
   }

   protected IndexedQueryMode getIndexedQueryMode() {
      return IndexedQueryMode.FETCH;
   }

   protected String getQuery() {
      return "from " + Person.class.getName() + " p where p.name:'person*'";
   }
}
