/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.api.factory;

import org.jboss.bootstrap.api.server.Server;

/**
 * ServerFactory
 * 
 * Factory of generic {@link Server} implementations
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public class ServerFactory
{

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   private ServerFactory()
   {
      // No external access
   }

   //-------------------------------------------------------------------------------------||
   // Factory Methods --------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Creates a Server from the specified implementation class name, using the
    * Thread Context ClassLoader.  The returned server will additionally be created using
    * the TCCL.
    * 
    * @param implClassName The fully-qualified name of the implementation class
    * @throws IllegalArgumentException If the implementation class name is null, blank, 
    *       can not be found on the ClassLoader, or is not assignable to {@link Server} 
    * @throws InstantiationException If some error occurred in constructing the Server
    * @return The newly-created Server
    */
   public static Server<?, ?> createServer(final String implClassName) throws IllegalArgumentException,
         InstantiationException
   {
      return createServer(implClassName, SecurityActions.getTccl());
   }

   /**
    * Creates a Server from the specified implementation class name, using the
    * specified ClassLoader.  The returned server will additionally be created using
    * the ClassLoader denoted. 
    * 
    * @param implClassName The fully-qualified name of the implementation class
    * @param cl The ClassLoader to use
    * @throws IllegalArgumentException If the ClassLoader is null, implementation class 
    *       name is null, blank, can not be found on the ClassLoader, 
    *       or is not assignable to {@link Server}.
    * @throws InstantiationException If some error occurred in constructing the Server
    * @return The newly-created Server
    */
   public static Server<?, ?> createServer(final String implClassName, final ClassLoader cl)
         throws IllegalArgumentException, InstantiationException
   {
      return GenericFactory.createInstance(implClassName, cl, Server.class);
   }
}
