/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.impl.as.config;

import java.io.File;
import java.lang.reflect.Type;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

import org.jboss.bootstrap.impl.base.config.AbstractBasicServerConfig;
import org.jboss.bootstrap.spi.as.config.JBossASServerConfig;
import org.jboss.logging.Logger;
import org.jboss.managed.api.annotation.ManagementComponent;
import org.jboss.managed.api.annotation.ManagementObject;
import org.jboss.managed.api.annotation.ManagementProperties;
import org.jboss.managed.api.annotation.ManagementProperty;
import org.jboss.metatype.api.annotations.MetaMapping;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.types.SimpleMetaType;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.SimpleValue;
import org.jboss.metatype.api.values.SimpleValueSupport;
import org.jboss.metatype.spi.values.MetaMapper;

/**
 * BasicJBossASServerConfig
 * 
 * Basic object-backed implementation of the {@link JBossASServerConfig}.  
 * As this is exported from the Server, this implementation 
 * is Thread-safe.
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
@ManagementObject(name = "jboss.system:type=ServerConfig", isRuntime = true, properties = ManagementProperties.EXPLICIT, description = "provides a view of server config information", componentType = @ManagementComponent(type = "MCBean", subtype = "*"))
public class BasicJBossASServerConfig extends AbstractBasicServerConfig<JBossASServerConfig>
      implements
         JBossASServerConfig
{
   //-------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   @SuppressWarnings("unused")
   private static final Logger log = Logger.getLogger(BasicJBossASServerConfig.class);

   /**
    * Property for JGroups denoting the UDP Multicast address.  Should be set alongside
    * {@link JBossASServerConfig#PROP_KEY_JBOSSAS_PARTITION_UDP_GROUP}
    */
   private static String PROP_KEY_JGROUPS_UDP_MCAST_ADDRESS = "jgroups.udp.mcast_addr";

   //-------------------------------------------------------------------------------||
   // Instance Members -------------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /**
    * Bind address for the server.  Synchronized on
    * "this".  Volatile so we don't have to block
    * when simply setting the value.
    */
   private volatile String bindAddress;

   /**
    * $JBOSS_HOME, root of the AS installation
    * Synchronized on "this".   Must
    * not be exported (so copy on return).
    */
   private URL jbossHome;

   /**
    * The server configuration name. Synchronized on
    * "this".
    */
   private String serverName;

   /**
    * The AS Boot Library Location
    * Synchronized on "this".   Must
    * not be exported (so copy on return).
    */
   private URL bootLibraryLocation;

   /**
    * The Location from which server homes, by default, decend
    * Synchronized on "this".   Must
    * not be exported (so copy on return).
    */
   private URL serverBaseLocation;

   /**
    * The location of the server home
    * Synchronized on "this".   Must
    * not be exported (so copy on return).
    */
   private URL serverHomeLocation;

   /**
    * The location of the common base
    * Synchronized on "this".   Must
    * not be exported (so copy on return).
    */
   private URL commonBaseLocation;

   /**
    * The location of the common lib
    * Synchronized on "this".   Must
    * not be exported (so copy on return).
    */
   private URL commonLibLocation;

   /**
    * Location in which the server logs reside
    * Synchronized on "this".   Must
    * not be exported (so copy on return).
    */
   private URL serverLogLocation;

   /**
    * Location in which the server configuration resides
    * Synchronized on "this".   Must
    * not be exported (so copy on return).
    */
   private URL serverConfLocation;

   /**
    * Location in which the server libraries reside
    * Synchronized on "this".   Must
    * not be exported (so copy on return).
    */
   private URL serverLibLocation;

   /**
    * Location in which the server persistent data resides
    * Synchronized on "this".   Must
    * not be exported (so copy on return).
    */
   private URL serverDataLocation;

   /**
    * Location in which the server temp data resides
    * Synchronized on "this".   Must
    * not be exported (so copy on return).
    */
   private URL serverTempLocation;

   /**
    * Partition name this instance uses when taking
    * part in clustering.  Synchronized on "this".
    * Volatile so we don't have to block
    * when simply setting the value.
    */
   private volatile String partitionName;

   /**
    * UDP Group this instance uses when taking
    * part in clustering.  Synchronized on "this".
    * Volatile so we don't have to block
    * when simply setting the value.
    */
   private volatile String udpGroup;

   /**
    * UDP Port this instance uses when taking
    * part in clustering.  Synchronized on "this".
    * Volatile so we don't have to block
    * when simply setting the value.
    */
   private volatile Integer udpPort;

   /**
    * Whether or not to load the native libraries.
    * Synchronized on "this".
    * Volatile so we don't have to block
    * when simply setting the value.
    */
   private volatile Boolean loadNative;

   /**
    * The AS Native Library Location (where
    * native libs are unpacked).
    * Synchronized on "this".   Must
    * not be exported (so copy on return).
    */
   private URL nativeLibraryLocation;

   /**
    * Whether or not to use the platform MBean Server
    * on JDK 1.5+.
    * Synchronized on "this".
    * Volatile so we don't have to block
    * when simply setting the value.
    */
   private volatile Boolean usePlatformMBeanServer;

   //-------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   public BasicJBossASServerConfig()
   {
      super(JBossASServerConfig.class);
   }

   //-------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#bindAddress(java.lang.String)
    */
   public synchronized JBossASServerConfig bindAddress(final String bindAddress) throws IllegalArgumentException
   {
      // Set
      this.bindAddress = bindAddress;

      // Set property
      this.setPropertyForString(PROP_KEY_JBOSSAS_BIND_ADDRESS, bindAddress);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getBindAddress()
    */
   @ManagementProperty(description = "The address to which server sockets should bind", readOnly = true)
   public synchronized String getBindAddress()
   {
      return this.bindAddress;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getServerName()
    */
   @ManagementProperty(description = "the name of the active profile the sever is using", readOnly = true)
   public String getServerName()
   {
      return this.serverName;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#jbossHome(java.lang.String)
    */
   public JBossASServerConfig jbossHome(final String jbossHome) throws IllegalArgumentException
   {
      // If null, just pass along
      if (jbossHome == null)
      {
         return this.jbossHome((URL) null);
      }

      // Convert this String into a URL
      final URL jbossHomeUrl = this.getUrlFromString(jbossHome);

      // Return
      return this.jbossHome(jbossHomeUrl);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#jbossHome(java.net.URL)
    */
   public synchronized JBossASServerConfig jbossHome(final URL jbossHome)
   {
      // Set
      this.jbossHome = jbossHome;

      // Set properties      
      this.setPropertyForUrlAndDir(PROP_KEY_JBOSSAS_HOME_URL, PROP_KEY_JBOSSAS_HOME, jbossHome);
      this.setPropertyForUrlAndDir(PROP_KEY_JBOSSAS_HOME_URL, PROP_KEY_JBOSSAS_HOME_DIR, jbossHome);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverName(java.lang.String)
    */
   public synchronized JBossASServerConfig serverName(final String serverName)
   {
      // Set
      this.serverName = serverName;

      // Set property
      this.setPropertyForString(PROP_KEY_JBOSSAS_SERVER_NAME, serverName);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getJBossHome()
    */
   @ManagementProperty(description = "the local home directory which the server is running from", readOnly = true, name = "homeUrl")
   public URL getJBossHome()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.jbossHome;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#bootLibraryLocation(java.lang.String)
    */
   public JBossASServerConfig bootLibraryLocation(final String bootLibraryLocation) throws IllegalArgumentException
   {
      // If null, just pass along
      if (bootLibraryLocation == null)
      {
         return this.bootLibraryLocation((URL) null);
      }

      // Convert this String into a URL via File
      final URL url = this.getUrlFromString(bootLibraryLocation);

      // Return
      return this.bootLibraryLocation(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#bootLibraryLocation(java.net.URL)
    */
   @ManagementProperty(description = "the bootstrap library URL for the server.", readOnly = true)
   public synchronized JBossASServerConfig bootLibraryLocation(final URL bootLibraryLocation)
   {
      // Set
      this.bootLibraryLocation = bootLibraryLocation;

      // Set properties
      this.setPropertyForUrl(PROP_KEY_JBOSSAS_BOOT_LIBRARY_URL, bootLibraryLocation);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getBootLibraryLocation()
    */
   @ManagementProperty(description = "The location of the boot libraries", readOnly = true)
   public URL getBootLibraryLocation()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.bootLibraryLocation;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getServerBaseLocation()
    */
   @ManagementProperty(description = "the base URL for calculating server home URLs", readOnly = true, name = "serverBaseURL")
   public URL getServerBaseLocation()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.serverBaseLocation;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverBaseLocation(java.lang.String)
    */
   public JBossASServerConfig serverBaseLocation(final String serverBaseLocation) throws IllegalArgumentException
   {
      // If null, just pass along
      if (serverBaseLocation == null)
      {
         return this.serverBaseLocation((URL) null);
      }

      // Convert this String into a URL via File
      final URL url = this.getUrlFromString(serverBaseLocation);

      // Return
      return this.serverBaseLocation(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverBaseLocation(java.net.URL)
    */
   public synchronized JBossASServerConfig serverBaseLocation(final URL serverBaseLocation)
   {
      // Set
      this.serverBaseLocation = serverBaseLocation;

      // Set properties
      this.setPropertyForUrlAndDir(PROP_KEY_JBOSSAS_SERVER_BASE_URL, PROP_KEY_JBOSSAS_SERVER_BASE_DIR,
            serverBaseLocation);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverHomeLocation()
    */
   @ManagementProperty(description = "the server home URL", readOnly = true, name = "serverHomeURL")
   public URL getServerHomeLocation()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.serverHomeLocation;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverHomeLocation(java.lang.String)
    */
   public JBossASServerConfig serverHomeLocation(final String serverHomeLocation) throws IllegalArgumentException
   {
      // If null, just pass along
      if (serverHomeLocation == null)
      {
         return this.serverHomeLocation((URL) null);
      }

      // Convert this String into a URL via File
      final URL url = this.getUrlFromString(serverHomeLocation);

      // Return
      return this.serverHomeLocation(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverHomeLocation(java.net.URL)
    */
   public synchronized JBossASServerConfig serverHomeLocation(final URL serverHomeLocation)
   {
      // Set
      this.serverHomeLocation = serverHomeLocation;

      // Set properties
      this.setPropertyForUrlAndDir(PROP_KEY_JBOSSAS_SERVER_HOME_URL, PROP_KEY_JBOSSAS_SERVER_HOME_DIR,
            serverHomeLocation);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#commonBaseLocation(java.lang.String)
    */
   public JBossASServerConfig commonBaseLocation(final String commonBaseLocation) throws IllegalArgumentException
   {
      // If null, just pass along
      if (commonBaseLocation == null)
      {
         return this.commonBaseLocation((URL) null);
      }

      // Convert this String into a URL via File
      final URL url = this.getUrlFromString(commonBaseLocation);

      // Return
      return this.commonBaseLocation(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#commonBaseLocation(java.net.URL)
    */
   public synchronized JBossASServerConfig commonBaseLocation(final URL commonBaseLocation)
   {
      // Set
      this.commonBaseLocation = commonBaseLocation;

      // Set properties
      this.setPropertyForUrl(PROP_KEY_JBOSSAS_COMMON_BASE_URL, commonBaseLocation);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getCommonBaseLocation()
    */
   @ManagementProperty(description = "the common library URL base", readOnly = true, name = "commonBaseURL")
   public URL getCommonBaseLocation()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.commonBaseLocation;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#commonLibLocation(java.lang.String)
    */
   public JBossASServerConfig commonLibLocation(final String commonLibLocation) throws IllegalArgumentException
   {
      // If null, just pass along
      if (commonLibLocation == null)
      {
         return this.commonLibLocation((URL) null);
      }

      // Convert this String into a URL via File
      final URL url = this.getUrlFromString(commonLibLocation);

      // Return
      return this.commonLibLocation(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#commonLibLocation(java.net.URL)
    */
   public synchronized JBossASServerConfig commonLibLocation(final URL commonLibLocation)
   {
      // Set
      this.commonLibLocation = commonLibLocation;

      // Set properties
      this.setPropertyForUrl(PROP_KEY_JBOSSAS_COMMON_LIBRARY_URL, commonLibLocation);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getCommonLibLocation()
    */
   @ManagementProperty(description = "the URL for the common library jars", readOnly = true, name = "commonLibraryURL")
   public URL getCommonLibLocation()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.commonLibLocation;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getServerLogLocation()
    */
   @ManagementProperty(description = "The location of the server logs", readOnly = true)
   public URL getServerLogLocation()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.serverLogLocation;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverLogLocation(java.lang.String)
    */
   public JBossASServerConfig serverLogLocation(final String serverLogLocation) throws IllegalArgumentException
   {
      // If null, just pass along
      if (serverLogLocation == null)
      {
         return this.serverLogLocation((URL) null);
      }

      // Convert this String into a URL via File
      final URL url = this.getUrlFromString(serverLogLocation);

      // Return
      return this.serverLogLocation(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverLogLocation(java.net.URL)
    */
   public synchronized JBossASServerConfig serverLogLocation(final URL serverLogLocation)
   {
      // Set
      this.serverLogLocation = serverLogLocation;

      // Get the absolute path
      String location = null;
      if (serverLogLocation != null)
      {
         URI uri = null;
         try
         {
            uri = serverLogLocation.toURI();
         }
         catch (URISyntaxException e)
         {
            throw new RuntimeException("Error in format of " + uri, e);
         }
         location = new File(uri).getAbsolutePath();
      }

      // Set properties
      this.setPropertyForString(PROP_KEY_JBOSSAS_SERVER_LOG_DIR, location);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getServerConfigLocation()
    */
   @ManagementProperty(description = "The location of the server configuration files", readOnly = true)
   public URL getServerConfLocation()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.serverConfLocation;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverConfigLocation(java.lang.String)
    */
   public JBossASServerConfig serverConfLocation(final String serverConfigLocation) throws IllegalArgumentException
   {
      // If null, just pass along
      if (serverConfigLocation == null)
      {
         return this.serverConfLocation((URL) null);
      }

      // Convert this String into a URL via File
      final URL url = this.getUrlFromString(serverConfigLocation);

      // Return
      return this.serverConfLocation(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverConfigLocation(java.net.URL)
    */
   public synchronized JBossASServerConfig serverConfLocation(final URL serverConfigLocation)
   {
      // Set
      this.serverConfLocation = serverConfigLocation;

      // Set properties
      this.setPropertyForUrl(PROP_KEY_JBOSSAS_SERVER_CONF_URL, serverConfigLocation);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getServerLibLocation()
    */
   @ManagementProperty(description = "the server library URL", readOnly = true, name = "serverLibraryURL")
   public URL getServerLibLocation()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.serverLibLocation;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverLibLocation(java.lang.String)
    */
   public JBossASServerConfig serverLibLocation(final String serverLibLocation) throws IllegalArgumentException
   {
      // If null, just pass along
      if (serverLibLocation == null)
      {
         return this.serverLibLocation((URL) null);
      }

      // Convert this String into a URL via File
      final URL url = this.getUrlFromString(serverLibLocation);

      // Return
      return this.serverLibLocation(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverLibLocation(java.net.URL)
    */
   public synchronized JBossASServerConfig serverLibLocation(final URL serverLibLocation)
   {
      // Set
      this.serverLibLocation = serverLibLocation;

      // Set properties
      this.setPropertyForUrl(PROP_KEY_JBOSSAS_SERVER_LIBRARY_URL, serverLibLocation);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getServerDataLocation()
    */
   @ManagementProperty(description = "The location into which data files are written", readOnly = true)
   public URL getServerDataLocation()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.serverDataLocation;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverDataLocation(java.lang.String)
    */
   public JBossASServerConfig serverDataLocation(final String serverDataLocation) throws IllegalArgumentException
   {
      // If null, just pass along
      if (serverDataLocation == null)
      {
         return this.serverDataLocation((URL) null);
      }

      // Convert this String into a URL via File
      final URL url = this.getUrlFromString(serverDataLocation);

      // Return
      return this.serverDataLocation(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverDataLocation(java.net.URL)
    */
   public synchronized JBossASServerConfig serverDataLocation(final URL serverDataLocation)
   {
      // Set
      this.serverDataLocation = serverDataLocation;

      // Get the absolute path
      String location = null;
      if (serverDataLocation != null)
      {
         URI uri = null;
         try
         {
            uri = serverDataLocation.toURI();
         }
         catch (URISyntaxException e)
         {
            throw new RuntimeException("Error in format of " + uri, e);
         }
         location = new File(uri).getAbsolutePath();
      }

      // Set properties
      this.setPropertyForString(PROP_KEY_JBOSSAS_SERVER_DATA_DIR, location);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getServerTempLocation()
    */
   @ManagementProperty(description = "The location into which temp files may be written", readOnly = true)
   public URL getServerTempLocation()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.serverTempLocation;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverTempLocation(java.lang.String)
    */
   public JBossASServerConfig serverTempLocation(final String serverTempLocation) throws IllegalArgumentException
   {
      // If null, just pass along
      if (serverTempLocation == null)
      {
         return this.serverTempLocation((URL) null);
      }

      // Convert this String into a URL via File
      final URL url = this.getUrlFromString(serverTempLocation);

      // Return
      return this.serverTempLocation(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#serverTempLocation(java.net.URL)
    */
   public synchronized JBossASServerConfig serverTempLocation(final URL serverTempLocation)
   {
      // Set
      this.serverTempLocation = serverTempLocation;

      // Get the absolute path
      String location = null;
      if (serverTempLocation != null)
      {
         URI uri = null;
         try
         {
            uri = serverTempLocation.toURI();
         }
         catch (URISyntaxException e)
         {
            throw new RuntimeException("Error in format of " + uri, e);
         }
         location = new File(uri).getAbsolutePath();
      }

      // Set properties
      this.setPropertyForString(PROP_KEY_JBOSSAS_SERVER_TEMP_DIR, location);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getPartitionName()
    */
   @ManagementProperty(description = "The name of the clustering partition", readOnly = true)
   public String getPartitionName()
   {
      return this.partitionName;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#partitionName(java.lang.String)
    */
   public synchronized JBossASServerConfig partitionName(final String partitionName)
   {
      // Set
      this.partitionName = partitionName;

      // Set property
      this.setPropertyForString(PROP_KEY_JBOSSAS_PARTITION_NAME, partitionName);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getUdpGroup()
    */
   @ManagementProperty(description = "The UDP Group used in clustering", readOnly = true)
   public String getUdpGroup()
   {
      return this.udpGroup;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#udpGroup(java.lang.String)
    */
   public synchronized JBossASServerConfig udpGroup(final String udpGroup)
   {
      // Set
      this.udpGroup = udpGroup;

      // Set properties
      this.setPropertyForString(PROP_KEY_JBOSSAS_PARTITION_UDP_GROUP, udpGroup);
      this.setPropertyForString(PROP_KEY_JGROUPS_UDP_MCAST_ADDRESS, udpGroup);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getUdpPort()
    */
   @ManagementProperty(description = "The UDP Port used in clustering", readOnly = true)
   public Integer getUdpPort()
   {
      return this.udpPort;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#udpGroup(java.lang.Integer)
    */
   public synchronized JBossASServerConfig udpPort(final Integer udpPort)
   {
      // Set
      this.udpPort = udpPort;

      // Set property
      this.setPropertyForString(PROP_KEY_JBOSSAS_PARTITION_UDP_PORT, udpPort);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#isLoadNative()
    */
   @ManagementProperty(description = "Whether or not to load the native libraries", readOnly = true)
   public Boolean isLoadNative()
   {
      return this.loadNative;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#loadNative(java.lang.Boolean)
    */
   public synchronized JBossASServerConfig loadNative(Boolean loadNative)
   {
      // Set
      this.loadNative = loadNative;

      // Set property
      this.setPropertyForString(PROP_KEY_JBOSSAS_NATIVE_LOAD, loadNative);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#getNativeLibraryLocation()
    */
   @ManagementProperty(description = "The location in which the native libraries reside", readOnly = true)
   public URL getNativeLibraryLocation()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.nativeLibraryLocation;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#nativeLibraryLocation(java.lang.String)
    */
   public JBossASServerConfig nativeLibraryLocation(final String nativeLibraryLocation) throws IllegalArgumentException
   {
      // If null, just pass along
      if (nativeLibraryLocation == null)
      {
         return this.nativeLibraryLocation((URL) null);
      }

      // Convert this String into a URL via File
      final URL url = this.getUrlFromString(nativeLibraryLocation);

      // Return
      return this.nativeLibraryLocation(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.config.JBossASServerConfig#nativeLibraryLocation(java.net.URL)
    */
   public synchronized JBossASServerConfig nativeLibraryLocation(final URL nativeLibraryLocation)
   {
      // Set
      this.nativeLibraryLocation = nativeLibraryLocation;

      // Get the absolute path
      String location = null;
      if (nativeLibraryLocation != null)
      {
         URI uri = null;
         try
         {
            uri = nativeLibraryLocation.toURI();
         }
         catch (URISyntaxException e)
         {
            throw new RuntimeException("Error in format of " + uri, e);
         }
         location = new File(uri).getAbsolutePath();
      }

      // Set properties
      this.setPropertyForString(PROP_KEY_JBOSSAS_NATIVE_DIR, location);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.as.config.JBossASServerConfig#isUsePlatformMBeanServer()
    */
   @ManagementProperty(description = "Whether or not to use the Platform MBean Server for JMX Operations", readOnly = true)
   public Boolean isUsePlatformMBeanServer()
   {
      return this.usePlatformMBeanServer;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.as.config.JBossASServerConfig#usePlatformMBeanServer(java.lang.Boolean)
    */
   public synchronized JBossASServerConfig usePlatformMBeanServer(final Boolean usePlatformMBeanServer)
   {
      // Set
      this.usePlatformMBeanServer = usePlatformMBeanServer;

      // Set property
      this.setPropertyForString(PROP_KEY_JBOSSAS_PLATFORM_MBEANSERVER, usePlatformMBeanServer);

      // Return
      return this.covarientReturn();
   }

   //-------------------------------------------------------------------------------||
   // Managed Metadata Operations --------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /*
    * These overridden implementations are to append the management object
    * metadata required by AS and the Consoles
    * 
    * JBBOOT-84
    */

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.base.config.AbstractBasicServerConfig#getBootstrapUrl()
    */
   @ManagementProperty(description = "the bootstrap url", readOnly = true, name = "bootstrapURL")
   @Override
   public URL getBootstrapUrl()
   {
      return super.getBootstrapUrl();
   }

   @ManagementProperty(description = "the local home directory which the server is running from", readOnly = true)
   @MetaMapping(FileMetaMapper.class)
   public File getHomeDir()
   {
      return this.urlToFile(this.getJBossHome());
   }

   @ManagementProperty(description = "the base directory for calculating server home directories", readOnly = true)
   @MetaMapping(FileMetaMapper.class)
   public File getServerBaseDir()
   {
      return this.urlToFile(this.getServerBaseLocation());
   }

   @ManagementProperty(description = "the server home directory", readOnly = true)
   @MetaMapping(FileMetaMapper.class)
   public File getServerHomeDir()
   {
      return this.urlToFile(this.getServerHomeLocation());
   }

   @ManagementProperty(description = "the server log directory", readOnly = true)
   @MetaMapping(FileMetaMapper.class)
   public File getServerLogDir()
   {
      return this.urlToFile(this.getServerLogLocation());
   }

   @ManagementProperty(description = "the directory where temporary files will be stored", readOnly = true)
   @MetaMapping(FileMetaMapper.class)
   public File getServerTempDir()
   {
      return this.urlToFile(this.getServerTempLocation());
   }

   @ManagementProperty(description = "the directory where local data will be stored", readOnly = true)
   @MetaMapping(FileMetaMapper.class)
   public File getServerDataDir()
   {
      return this.urlToFile(this.getServerDataLocation());
   }

   @ManagementProperty(description = "the directory for platform native files", readOnly = true)
   @MetaMapping(FileMetaMapper.class)
   public File getServerNativeDir()
   {
      return this.urlToFile(this.getNativeLibraryLocation());
   }

   /**
    * Represents the specified URL as a file.  Throws unchecked
    * exception upon type errors 
    * 
    * @param url
    * @return The File representation, or null if null was passed in
    */
   private File urlToFile(final URL url)
   {
      // If null
      if (url == null)
      {
         return null;
      }

      // Convert to File
      try
      {
         return new File(url.toURI());
      }
      catch (URISyntaxException e)
      {
         throw new RuntimeException(e);
      }
   }

   //-------------------------------------------------------------------------------||
   // Internal Helper Methods ------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /**
    * Obtains a URL from the specified String, first detecting whether the 
    * String represents a URL format already (ie. has protocol).  Prevents
    * direct URL-ing of something already in URL format 
    * (ie. "http://something/http://newValue") 
    *  
    * @param arg
    * @return
    * @throws IllegalArgumentException
    */
   private URL getUrlFromString(final String arg) throws IllegalArgumentException
   {

      // See if this String represents a valid URL
      URL resolvedUrl = null;
      try
      {
         resolvedUrl = new URL(arg);
         return resolvedUrl;
      }
      catch (MalformedURLException murle)
      {
         // That's fine, perhaps it's an absolute path reference
      }

      // Convert this String into a URL via File
      final File file = new File(arg);
      try
      {
         resolvedUrl = file.toURI().toURL();
      }
      catch (MalformedURLException e)
      {
         throw new IllegalArgumentException("Cannot construct URL for argument given: " + arg, e);
      }

      // Return
      return resolvedUrl;
   }

   /**
    * Sets the specified URL's value upon the specified URL Property name and 
    * Directory property name, using appropriate syntax for each (ie. URLs use 
    * a protocol, Directories have just an absolute path) 
    * 
    * @param urlPropName
    * @param dirPropName
    * @param url
    * @throws IllegalArgumentException If either the urlPropName or dirPropName are not
    *   specified
    */
   private void setPropertyForUrlAndDir(final String urlPropName, final String dirPropName, final URL url)
         throws IllegalArgumentException
   {
      // Precondition checks
      if (urlPropName == null || urlPropName.length() == 0)
      {
         throw new IllegalArgumentException("urlPropName is required");
      }
      if (dirPropName == null || dirPropName.length() == 0)
      {
         throw new IllegalArgumentException("dirPropName is required");
      }

      // Set property for the URL
      this.setPropertyForUrl(urlPropName, url);

      // If null
      if (url == null)
      {
         // Set the dir property to null
         this.setPropertyForString(dirPropName, (String) null);
      }
      // Convert the dir property to strip the protocol, and set
      else
      {
         URI uri;
         try
         {
            uri = url.toURI();
         }
         catch (URISyntaxException e)
         {
            throw new RuntimeException("Error in format of " + urlPropName + ": + url", e);
         }
         final File file = new File(uri);
         final String location = file.getAbsolutePath();
         this.setPropertyForString(dirPropName, location);
      }
   }

   //-------------------------------------------------------------------------------||
   // Inner Classes ----------------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /**
    * TODO: move to managed project, JBMAN-49
    */
   public static class FileMetaMapper extends MetaMapper<File>
   {
      @Override
      public MetaValue createMetaValue(MetaType metaType, File object)
      {
         return SimpleValueSupport.wrap(object.getAbsolutePath());
      }

      @Override
      public MetaType getMetaType()
      {
         return SimpleMetaType.STRING;
      }

      @Override
      public Type mapToType()
      {
         return File.class;
      }

      @Override
      public File unwrapMetaValue(MetaValue metaValue)
      {
         SimpleValue svalue = (SimpleValue) metaValue;
         File f = null;
         if (svalue != null)
         {
            f = new File(svalue.getValue().toString());
         }
         return f;
      }
   }

}
