/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.impl.as.server;

import java.util.Date;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.jboss.bootstrap.impl.as.config.BasicJBossASServerConfig;
import org.jboss.bootstrap.impl.as.config.JBossASConfigurationInitializerImpl;
import org.jboss.bootstrap.impl.as.config.JBossASConfigurationValidator;
import org.jboss.bootstrap.impl.as.lifecycle.KernelStartEventLifecycleEventHandler;
import org.jboss.bootstrap.impl.as.lifecycle.KernelStopEventLifecycleEventHandler;
import org.jboss.bootstrap.impl.as.lifecycle.VfsInitializingLifecycleEventHandler;
import org.jboss.bootstrap.impl.mc.server.AbstractMCServerBase;
import org.jboss.bootstrap.spi.as.config.JBossASConfigurationInitializer;
import org.jboss.bootstrap.spi.as.config.JBossASServerConfig;
import org.jboss.bootstrap.spi.as.server.JBossASServer;
import org.jboss.bootstrap.spi.config.ConfigurationValidator;
import org.jboss.bootstrap.spi.config.InvalidConfigurationException;
import org.jboss.bootstrap.spi.lifecycle.LifecycleEventException;
import org.jboss.bootstrap.spi.lifecycle.LifecycleEventHandler;
import org.jboss.bootstrap.spi.lifecycle.LifecycleState;
import org.jboss.kernel.plugins.bootstrap.basic.BasicBootstrap;
import org.jboss.logging.Logger;
import org.jboss.managed.api.annotation.ManagementComponent;
import org.jboss.managed.api.annotation.ManagementObject;
import org.jboss.managed.api.annotation.ManagementProperties;
import org.jboss.managed.api.annotation.ManagementProperty;

/**
 * JBossASServerImpl
 * 
 * Implementation of a JBossAS Server
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
@ManagementObject(name = "jboss.system:type=MCServer", isRuntime = true, properties = ManagementProperties.EXPLICIT, description = "the MCServer bootstrap view", componentType = @ManagementComponent(type = "MCBean", subtype = "*"))
public class JBossASServerImpl extends AbstractMCServerBase<JBossASServer, JBossASServerConfig>
      implements
         JBossASServer
{
   //-------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /**
    * Logger
    */
   private static final Logger log = Logger.getLogger(JBossASServerImpl.class);

   /**
    * The validator instance 
    */
   private static final ConfigurationValidator<JBossASServerConfig> VALIDATOR;

   /**
    * The server initializer instance
    */
   private static final JBossASServerInitializer SERVER_INITIALIZER;

   /**
    * The config validator instance
    */
   private static final JBossASConfigurationInitializer CONFIG_INITIALIZER;

   /*
    * Initialize constants
    */
   static
   {
      VALIDATOR = new JBossASConfigurationValidator();
      SERVER_INITIALIZER = new JBossASServerInitializer();
      CONFIG_INITIALIZER = new JBossASConfigurationInitializerImpl();
   }

   //-------------------------------------------------------------------------------||
   // Instance Members -------------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /** 
    * Container for version information. 
    */
   private final Version version = Version.getInstance();

   /**
    * The date in which the server was started.  
    * Must not be exported (as it's mutable).  Synchronized
    * on "this" (as implicit from start() lifecycle) and volatile
    * so that we don't need to block in {@link JBossASServerImpl#getStartDate()}
    */
   private volatile Date startDate;

   /** 
    * The optional configuration metadata for the server.  No sync required as the backing
    * instance is final.  Requires Thread-safe impl (as it's exported). 
    */
   private final Map<String, Object> metadata = new ConcurrentHashMap<String, Object>();

   //-------------------------------------------------------------------------------||
   // Constructors -----------------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /**
    * Constructor
    * 
    * Constructs a new JBossAS Server using a new default configuration, 
    * which will be automatically created and set
    * 
    * @param config
    */
   public JBossASServerImpl() throws IllegalArgumentException
   {
      // Use other ctor
      this(null);
   }

   /**
    * Constructor
    * 
    * Constructs a new JBossAS Server with the specified underlying configuration
    * 
    * @param config
    */
   public JBossASServerImpl(final JBossASServerConfig config) throws IllegalArgumentException
   {
      // Invoke super
      super(JBossASServer.class, config);

      // Set properties
      this.setServerInitializer(SERVER_INITIALIZER);
      this.setConfigInitializer(CONFIG_INITIALIZER);
      this.setValidator(VALIDATOR);
   }

   //-------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.base.server.AbstractServer#getDefaultServerConfigClass()
    */
   @Override
   protected Class<? extends JBossASServerConfig> getDefaultServerConfigClass()
   {
      return BasicJBossASServerConfig.class;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.server.JBossASServer#getBuildDate()
    */
   @ManagementProperty(description = "The server build date", readOnly = true)
   public String getBuildDate()
   {
      return version.getBuildDate();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.server.JBossASServer#getBuildID()
    */
   @ManagementProperty(description = "The server build ID", readOnly = true)
   public String getBuildID()
   {
      return version.getBuildID();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.server.JBossASServer#getBuildJVM()
    */
   @ManagementProperty(description = "The server build JVM", readOnly = true)
   public String getBuildJVM()
   {
      return version.getBuildJVM();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.server.JBossASServer#getBuildNumber()
    */
   @ManagementProperty(description = "The server build number", readOnly = true)
   public String getBuildNumber()
   {
      return version.getBuildNumber();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.server.JBossASServer#getBuildOS()
    */
   @ManagementProperty(description = "The server build OS", readOnly = true)
   public String getBuildOS()
   {
      return version.getBuildOS();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.server.JBossASServer#getStartDate()
    */
   @ManagementProperty(description = "The server start time", readOnly = true)
   public Date getStartDate()
   {
      // We copy so we don't export the mutable state, JBBOOT-73
      final Date copyDate = (Date) startDate.clone();
      return copyDate;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.server.JBossASServer#getVersion()
    */
   @ManagementProperty(description = "The server version string", readOnly = true)
   public String getVersion()
   {
      return version.toString();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.server.JBossASServer#getVersionName()
    */
   @ManagementProperty(description = "The server version name", readOnly = true)
   public String getVersionName()
   {
      return version.getName();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.as.server.JBossASServer#getVersionNumber()
    */
   @ManagementProperty(description = "The server version number string", readOnly = true)
   public String getVersionNumber()
   {
      return version.getVersionNumber();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#getMetaData()
    */
   public Map<String, Object> getMetaData()
   {
      return this.metadata;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.as.server.JBossASServer#isStarted()
    */
   /*
    * JBBOOT-80 Remove this when jboss-bootstrap is for AS6 only (no
    * more AS5.x support)
    */
   @Deprecated
   public boolean isStarted()
   {
      final LifecycleState state = this.getState();
      return state.equals(LifecycleState.STARTED);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.mc.server.AbstractMCServerBase#initialize()
    */
   @Override
   public synchronized void initialize() throws IllegalStateException, InvalidConfigurationException,
         LifecycleEventException
   {
      // Call Super
      super.initialize();

      // JBBOOT-68
      //TODO Remove once VFS is init'd from something else
      // Register an event handler to init VFS alongside server start
      @SuppressWarnings("deprecation")
      final LifecycleEventHandler initVfsHandler = new VfsInitializingLifecycleEventHandler();
      this.registerEventHandler(initVfsHandler, LifecycleState.STARTING);

      // Create and Register handlers
      final BasicBootstrap bootstrap = this.getBootstrap();
      final LifecycleEventHandler startHandler = new KernelStartEventLifecycleEventHandler(bootstrap);
      final LifecycleEventHandler stopHandler = new KernelStopEventLifecycleEventHandler(bootstrap);
      this.registerEventHandler(startHandler, LifecycleState.STARTED);
      this.registerEventHandler(stopHandler, LifecycleState.STOPPED);

      // Log the server info
      this.logServerInfo();
   }

   /**
    * Logs out information from the underlying server configuration
    */
   protected void logServerInfo()
   {
      // Initialize
      final StringBuffer sb = new StringBuffer();
      final char newline = '\n';
      final char tab = '\t';

      // Get the config
      final JBossASServerConfig config = this.getConfiguration();

      // Log
      sb.append("Server Configuration:");
      sb.append(newline).append(newline).append(tab);
      sb.append("Bootstrap URL: " + config.getBootstrapUrl());
      sb.append(newline).append(tab);
      sb.append("JBOSS_HOME URL: " + config.getJBossHome());
      sb.append(newline).append(tab);
      sb.append("Common Base URL: " + config.getCommonBaseLocation());
      sb.append(newline).append(tab);
      sb.append("Common Library URL: " + config.getCommonLibLocation());
      sb.append(newline).append(tab);
      sb.append("Server Name: " + config.getServerName());
      sb.append(newline).append(tab);
      sb.append("Server Base URL: " + config.getServerBaseLocation());
      sb.append(newline).append(tab);
      sb.append("Server Library URL: " + config.getServerLibLocation());
      sb.append(newline).append(tab);
      sb.append("Server Config URL: " + config.getServerConfLocation());
      sb.append(newline).append(tab);
      sb.append("Server Home URL: " + config.getServerHomeLocation());
      sb.append(newline).append(tab);
      sb.append("Server Data URL: " + config.getServerDataLocation());
      sb.append(newline).append(tab);
      sb.append("Server Log URL: " + config.getServerLogLocation());
      sb.append(newline).append(tab);
      sb.append("Server Temp URL: " + config.getServerTempLocation());
      sb.append(newline);

      // Log
      log.info(sb.toString());
   }

   //-------------------------------------------------------------------------------||
   // Overridden Implementations ---------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.mc.server.AbstractMCServerBase#doStart()
    */
   @Override
   protected void doStart() throws Exception
   {
      // Call super
      super.doStart();

      // Mark the start date
      this.setStartDate(new Date());
   }

   //-------------------------------------------------------------------------------||
   // Accessors / Mutators ---------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /**
    * @param startDate the startDate to set
    */
   protected void setStartDate(final Date startDate)
   {
      // Clone so we don't give callers access to internal state
      this.startDate = (Date) startDate.clone();
   }

   @Override
   public String toString()
   {
      StringBuilder sb = new StringBuilder();
      sb.append("JBoss Server[");
      sb.append(this.version.toString());
      sb.append("]");
      return sb.toString();
   }
}
