/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.impl.as.server;

import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

import org.jboss.bootstrap.impl.base.server.AbstractBasicServerInitializer;
import org.jboss.bootstrap.spi.as.config.JBossASServerConfig;
import org.jboss.bootstrap.spi.as.server.JBossASServer;
import org.jboss.bootstrap.spi.server.ServerInitializer;
import org.jboss.logging.Logger;

/**
 * JBossASServerInitializer
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public class JBossASServerInitializer extends AbstractBasicServerInitializer<JBossASServer, JBossASServerConfig>
      implements
         ServerInitializer<JBossASServer, JBossASServerConfig>
{

   //-------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   private static final Logger log = Logger.getLogger(JBossASServerInitializer.class);

   //-------------------------------------------------------------------------------||
   // Overridden Implementations ---------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.AbstractBasicServerInitializer#initialize(org.jboss.bootstrap.spi.server.Server)
    */
   @Override
   public void initialize(JBossASServer server) throws IllegalArgumentException, IllegalStateException
   {
      // Log
      if (log.isTraceEnabled())
      {
         log.trace("Initializing: " + this + " ...");
      }

      // Call super
      super.initialize(server);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.AbstractBasicServerInitializer#setSystemProperties(org.jboss.bootstrap.spi.config.ServerConfig)
    */
   @Override
   protected void setSystemProperties(JBossASServerConfig configuration)
   {
      // Log
      if (log.isTraceEnabled())
      {
         log.trace("Setting system properties for " + this + " ...");
      }

      // Super
      super.setSystemProperties(configuration);

      // Get values to set
      final URL jbossHome = configuration.getJBossHome();
      final String jbossHomeDir = this.getAbsoluteLocationOfUrl(jbossHome);
      final URL bootLibraryUrl = configuration.getBootLibraryLocation();
      final String bootLibraryString = bootLibraryUrl == null ? null : bootLibraryUrl.toExternalForm();
      final URL serverBaseUrl = configuration.getServerBaseLocation();
      final String serverBaseDir = this.getAbsoluteLocationOfUrl(serverBaseUrl);
      final URL serverHomeUrl = configuration.getServerHomeLocation();
      final String serverHomeDir = this.getAbsoluteLocationOfUrl(serverHomeUrl);
      final URL commonBaseUrl = configuration.getCommonBaseLocation();
      final URL commonLibUrl = configuration.getCommonLibLocation();
      final String serverLogDir = this.getAbsoluteLocationOfUrl(configuration.getServerLogLocation());
      final URL serverConfDir = configuration.getServerConfLocation();
      final URL serverLibDir = configuration.getServerLibLocation();
      final String serverDataDir = this.getAbsoluteLocationOfUrl(configuration.getServerDataLocation());
      final String serverTempDir = this.getAbsoluteLocationOfUrl(configuration.getServerTempLocation());
      final String partitionName = configuration.getPartitionName();
      final String udpGroup = configuration.getUdpGroup();
      final Integer udpPort = configuration.getUdpPort();
      final Boolean loadNative = configuration.isLoadNative();
      final String nativeLibLocation = this.getAbsoluteLocationOfUrl(configuration.getNativeLibraryLocation());
      final Boolean usePlatformMBeanServer = configuration.isUsePlatformMBeanServer();

      // Set our system properties
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_HOME, jbossHomeDir);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_HOME_DIR, jbossHomeDir);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_HOME_URL, jbossHome);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_BIND_ADDRESS, configuration.getBindAddress());
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_SERVER_NAME, configuration.getServerName());
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_BOOT_LIBRARY_URL, bootLibraryString);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_SERVER_BASE_URL, serverBaseUrl);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_SERVER_BASE_DIR, serverBaseDir);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_SERVER_HOME_URL, serverHomeUrl);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_SERVER_HOME_DIR, serverHomeDir);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_COMMON_BASE_URL, commonBaseUrl);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_COMMON_LIBRARY_URL, commonLibUrl);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_SERVER_LOG_DIR, serverLogDir);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_SERVER_CONF_URL, serverConfDir);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_SERVER_LIBRARY_URL, serverLibDir);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_SERVER_DATA_DIR, serverDataDir);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_SERVER_TEMP_DIR, serverTempDir);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_PARTITION_NAME, partitionName);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_PARTITION_UDP_GROUP, udpGroup);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_PARTITION_UDP_PORT, udpPort);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_NATIVE_LOAD, loadNative);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_NATIVE_DIR, nativeLibLocation);
      this.setSystemProperty(JBossASServerConfig.PROP_KEY_JBOSSAS_PLATFORM_MBEANSERVER, usePlatformMBeanServer);
   }

   //-------------------------------------------------------------------------------||
   // Functional Methods -----------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /**
    * Returns the absolute location of the given URL, stripping
    * the protocol
    * 
    * @param url
    * @return
    */
   private String getAbsoluteLocationOfUrl(final URL url)
   {
      // Convert to URI
      URI uri = null;
      try
      {
         uri = url.toURI();
      }
      catch (URISyntaxException urise)
      {
         throw new RuntimeException("URL be made into URI, formatting "
               + "error should have been caught by config validation", urise);
      }

      // Construct a File
      final File file = new File(uri);

      // Get the absolute path
      final String location = file.getAbsolutePath();

      // Return it
      return location;
   }

}
