/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.impl.base.config;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collections;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;

import org.jboss.bootstrap.spi.config.ServerConfig;
import org.jboss.logging.Logger;


/**
 * BasicServerConfig
 * 
 * Base for simple Object-backed implementations of a Server 
 * Configuration.  As this is exported from the Server, this 
 * is Thread-safe.
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public abstract class AbstractBasicServerConfig<T extends ServerConfig<T>> implements ServerConfig<T>
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   private static final Logger log = Logger.getLogger(AbstractBasicServerConfig.class);

   private static final String TRAILING_SLASH = "/";

   //-------------------------------------------------------------------------------------||
   // Instance Members -------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Actual class used in casting for covarient return in
    * method chaining.  Synchronized on "this".
    */
   private Class<T> actualClass;

   /**
    * URL of the bootstrap to run.  Synchronized on
    * "this".   Must
    * not be exported (so copy on return).
    */
   private URL bootstrapUrl;

   /**
    * Bootstrap configuration home (URLs referenced in the 
    * main bootstrap are relative to this URL).
    * Synchronized on "this".  Must
    * not be exported (so copy on return).
    */
   private URL bootstrapConfLocation;

   /**
    * Bootstrap home, used in defaulting/construction
    * of other properties.  Synchronized on "this".  Must
    * not be exported (so copy on return).
    */
   private URL bootstrapHome;

   /**
    * Name of the bootstrap file.  Synchronized on
    * "this".
    */
   private String bootstrapName;

   /**
    * Properties for this configuration.  Must be backed
    * by a Thread-safe impl.
    */
   private ConcurrentMap<String, String> properties;

   /**
    * Whether or not this configuration is frozen
    */
   private AtomicBoolean frozen;

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Constructor
    * 
    * @param actualClass The actual class of this Server Configuration
    *       to be used in casting for covarient return types
    */
   public AbstractBasicServerConfig(final Class<T> actualClass)
   {
      // Use no override properties
      this(actualClass, null);
   }

   /**
    * Constructor
    * 
    * @param actualClass The actual class of this Server Configuration
    *       to be used in casting for covarient return types
    * @param overrideProperties A Map of properties to override those found 
    *           in System Properties
    */
   public AbstractBasicServerConfig(final Class<T> actualClass, final Map<String, String> overrideProperties)
   {
      /*
       * Initialize properties
       */
      Properties sysProps = System.getProperties();
      ConcurrentMap<String, String> properties = new ConcurrentHashMap<String, String>();
      Set<Object> sysPropKeys = sysProps.keySet();
      for (Object sysPropKey : sysPropKeys)
      {
         // Cast, all Property keys are Strings
         String sysPropKeyString = (String) sysPropKey;
         String value = sysProps.getProperty(sysPropKeyString);
         properties.put(sysPropKeyString, value);
      }

      /*
       * Override properties
       */
      if (overrideProperties != null)
      {
         properties.putAll(overrideProperties);
      }

      /*
       * Set properties
       */
      this.properties = properties;
      synchronized (this)
      {
         this.actualClass = actualClass;
      }
      this.frozen = new AtomicBoolean(false);
   }

   //-------------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#getBootstrapConfLocation()
    */
   public URL getBootstrapConfLocation()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.bootstrapConfLocation;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#getBootstrapHome()
    */
   public URL getBootstrapHome()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.bootstrapHome;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#getBootstrapName()
    */
   public synchronized String getBootstrapName()
   {
      return this.bootstrapName;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#getBootstrapUrl()
    */
   public URL getBootstrapUrl()
   {
      URL url = null;
      synchronized (this)
      {
         url = this.bootstrapUrl;
      }
      return this.copyURL(url);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#getProperties()
    */
   public Map<String, String> getProperties()
   {
      return Collections.unmodifiableMap(this.properties);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#setBootstrapConfLocation(java.net.URL)
    */
   public synchronized T bootstrapConfLocation(final URL confLocation) throws IllegalArgumentException
   {
      // Precondition check
      this.checkMutable();

      // Adjust
      final URL newUrl = this.adjustToDirectory(confLocation);

      // Set property
      this.setPropertyForUrl(PROP_KEY_BOOTSTRAP_CONF_URL, newUrl);

      // Set URL
      this.bootstrapConfLocation = newUrl;

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#setBootstrapHome(java.net.URL)
    */
   public synchronized T bootstrapHome(final URL bootstrapHome) throws IllegalArgumentException
   {
      // Precondition check
      this.checkMutable();

      // Adjust
      final URL newUrl = this.adjustToDirectory(bootstrapHome);

      // Set property
      this.setPropertyForUrl(PROP_KEY_BOOTSTRAP_HOME_URL, newUrl);

      // Set URL
      this.bootstrapHome = newUrl;

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#setBootstrapName(java.lang.String)
    */
   public synchronized T bootstrapName(final String name) throws IllegalArgumentException
   {
      // Precondition check
      this.checkMutable();

      // Set property
      this.setPropertyForString(PROP_KEY_BOOTSTRAP_NAME, name);

      // Set URL
      this.bootstrapName = name;

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#setBootstrapUrl(java.net.URL)
    */
   public synchronized T bootstrapUrl(final URL bootstrapLocation) throws IllegalArgumentException
   {
      // Precondition check
      this.checkMutable();

      // Set property
      this.setPropertyForUrl(PROP_KEY_BOOTSTRAP_URL, bootstrapLocation);

      // Set URL
      this.bootstrapUrl = bootstrapLocation;

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#property(java.lang.String, java.lang.String)
    */
   public T property(final String key, final String value) throws IllegalArgumentException
   {
      // Precondition check
      this.checkMutable();

      // Set property
      this.setPropertyForString(key, value);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#freeze()
    */
   public void freeze() throws IllegalStateException
   {
      this.frozen.set(true);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#isFrozen()
    */
   public boolean isFrozen()
   {
      return this.frozen.get();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#bootstrapConfLocation(java.lang.String)
    */
   public T bootstrapConfLocation(String confLocation) throws IllegalArgumentException, IllegalStateException
   {
      // Set
      this.bootstrapConfLocation(this.urlFromString(confLocation));

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#bootstrapHome(java.lang.String)
    */
   public T bootstrapHome(String bootstrapHome) throws IllegalArgumentException, IllegalStateException
   {
      // Set
      this.bootstrapHome(this.urlFromString(bootstrapHome));

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#bootstrapUrl(java.lang.String)
    */
   public T bootstrapUrl(String bootstrapUrl) throws IllegalArgumentException, IllegalStateException
   {
      // Set
      this.bootstrapUrl(this.urlFromString(bootstrapUrl));

      // Return
      return this.covarientReturn();
   }

   //-------------------------------------------------------------------------------------||
   // Helper Methods ---------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Sets both the configuration property and System property with the 
    * specified key and the external form of the specified URL
    */
   protected void setPropertyForUrl(final String propertyName, final URL url)
   {
      String urlString = url != null ? url.toExternalForm() : null;
      this.setPropertyForString(propertyName, urlString);
   }

   /**
    * Sets both the configuration property and System property with the 
    * specified key and specified value
    */
   protected void setPropertyForString(final String propertyName, final String value)
   {
      Map<String, String> properties = this.properties;
      String valueToSet = value;
      if (valueToSet == null)
      {
         valueToSet = "";
      }
      properties.put(propertyName, valueToSet);
      if (log.isTraceEnabled())
      {
         log.trace(("Set property \"" + propertyName + "\" to: ") + (value != null ? "\"" + value + "\"" : "[EMPTY]"));
      }
   }

   /**
    * Throws IllegalStateException if this configuration is frozen
    * 
    * @throws IllegalStateException
    */
   protected void checkMutable() throws IllegalStateException
   {
      if (this.isFrozen())
      {
         throw new IllegalStateException("Cannot mutate state of the configuration after it has been frozen");
      }
   }

   /**
    * If the specified URL denotes a directory (ie. trailing slash), 
    * will return the argument.  Nulls are ignored and returned as-is.
    * 
    * @param url
    * @return
    */
   protected URL adjustToDirectory(final URL url)
   {
      // Do nothing to nulls
      if (url == null)
      {
         return null;
      }

      // Check
      final String externalForm = url.toExternalForm();
      if (externalForm.endsWith(TRAILING_SLASH))
      {
         // Return per normal
         return url;
      }

      // Otherwise make a new URL
      final String newLocation = externalForm + TRAILING_SLASH;
      URL newUrl = null;
      try
      {
         // Construct
         newUrl = new URL(newLocation);
      }
      catch (MalformedURLException e)
      {
         throw new RuntimeException("Could not create new URL to point to directory", e);
      }

      // Log
      if (log.isTraceEnabled())
      {
         log.trace("Appended trailing slash to " + url + " to point to directory: " + newUrl);
      }

      // Return it
      return newUrl;
   }

   /**
    * Returns the actual types of this configuration, used in 
    * covarient return of the mutator methods
    * 
    * @return
    */
   protected Class<T> getActualClass()
   {
      return this.actualClass;
   }

   /**
    * Casts this configuration to the requisite type, using 
    * the actual implementation class.  This is in place to
    * avoid unchecked casting (ie. (T)this)) and the resultant compiler 
    * warnings
    * 
    * @return
    */
   protected final T covarientReturn() throws ClassCastException
   {
      try
      {
         return this.getActualClass().cast(this);
      }
      catch (ClassCastException cce)
      {
         throw new RuntimeException(
               "Could not return the expected type; ensure that this configuration was constructed with the correct actual class",
               cce);
      }
   }

   /**
    * Copies and returns the specified URL.  Used
    * to ensure we don't export mutable URLs (thread safety) 
    * 
    * @param url
    * @return
    */
   protected final URL copyURL(final URL url)
   {
      // If null, return
      if (url == null)
      {
         return url;
      }

      try
      {
         // Copy 
         return new URL(url.toExternalForm());
      }
      catch (MalformedURLException e)
      {
         throw new RuntimeException("Error in copying URL", e);
      }
   }

   /**
    * Constructs a URL from the specified String, throwing
    * {@link IllegalArgumentException} to wrap the
    * {@link MalformedURLException} if unable to do so.  Null arguments 
    * will be ignored.
    *  
    * @param url
    * @return
    * @throws IllegalArgumentException
    */
   protected final URL urlFromString(String url) throws IllegalArgumentException
   {
      try
      {
         return url != null ? new URL(url) : null;
      }
      catch (MalformedURLException e)
      {
         throw new IllegalArgumentException("Could not construct " + URL.class.getSimpleName()
               + " from the supplied argument: " + url, e);
      }
   }
}
