/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.impl.base.server;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;

import org.jboss.logging.Logger;

/**
 * SecurityActions
 * 
 * Utility class for security actions, so as to provide
 * some centralization without leaking privileged actions
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
class SecurityActions
{
   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   private static final Logger log = Logger.getLogger(SecurityActions.class);

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * In place to prevent instanciation 
    */
   private SecurityActions()
   {
   }

   //-------------------------------------------------------------------------------------||
   // Utility Methods --------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Sets a system property with specified key and value
    * 
    * @throws IllegalArgumentException If the key was not specified
    */
   static void setSystemProperty(final String key, final String value) throws IllegalArgumentException
   {
      // Precondition check
      ensureSysPropKeySpecified(key);

      // Set
      AccessController.doPrivileged(new PrivilegedAction<Void>()
      {

         public Void run()
         {
            // Set
            System.setProperty(key, value);

            // Log
            if (log.isTraceEnabled())
            {
               log.trace("Set system property \"" + key + "\" > " + value);
            }

            // Return
            return null;
         }

      });
   }

   /**
    * Clears the specified System Property
    * 
    * @throws IllegalArgumentException If the key was not specified
    */
   static void clearSystemProperty(final String key) throws IllegalArgumentException
   {
      // Precondition check
      ensureSysPropKeySpecified(key);

      // Set
      AccessController.doPrivileged(new PrivilegedAction<Void>()
      {

         public Void run()
         {
            // Clear
            String previousValue = System.clearProperty(key);

            // Log
            if (log.isTraceEnabled())
            {
               log.trace("Removed system property \"" + key + "\" which had previous value: " + previousValue);
            }

            // Return
            return null;
         }

      });
   }

   /**
    * Creates a new instance of the specified class using the default ctor
    * 
    * @param clazz
    * @return
    * @throws IllegalArgumentException If the class is not specified
    * @throws Throwable If some error occured in creating the instance
    */
   static <T> T newInstance(final Class<T> clazz) throws IllegalArgumentException, Throwable
   {
      // Precondition check
      if (clazz == null)
      {
         throw new IllegalArgumentException("Class is a required argument");
      }

      // Create
      Object newInstance = null;
      try
      {
         // Set
         newInstance = AccessController.doPrivileged(new PrivilegedExceptionAction<Object>()
         {

            public Object run() throws Exception
            {
               if (log.isTraceEnabled())
               {
                  log.trace("Attempting to create new instance of " + clazz);
               }
               final Object obj = clazz.newInstance();
               if (log.isTraceEnabled())
               {
                  log.trace("Created " + obj);
               }
               return obj;
            }

         });
      }
      catch (PrivilegedActionException pae)
      {
         // Throw the cause
         throw pae.getCause();
      }

      // Cast
      T returnValue = clazz.cast(newInstance);

      // Return
      return returnValue;
   }

   /**
    * Obtains the Thread Context ClassLoader
    */
   static ClassLoader getThreadContextClassLoader()
   {
      return AccessController.doPrivileged(new PrivilegedAction<ClassLoader>()
      {
         public ClassLoader run()
         {
            return Thread.currentThread().getContextClassLoader();
         }
      });
   }

   /**
    * Sets the specified CL upon the current Thread's Context 
    * 
    * @param cl
    * @throws IllegalArgumentException If the CL was null
    */
   static void setThreadContextClassLoader(final ClassLoader cl) throws IllegalArgumentException
   {
      if (cl == null)
      {
         throw new IllegalArgumentException("ClassLoader was null");
      }

      AccessController.doPrivileged(new PrivilegedAction<Void>()
      {
         public Void run()
         {
            Thread.currentThread().setContextClassLoader(cl);
            return null;
         };
      });
   }

   //-------------------------------------------------------------------------------------||
   // Internal Helper Methods ------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Ensures that the specified key is not null
    * 
    * @throws IllegalArgumentException If the specified key was null
    */
   private static void ensureSysPropKeySpecified(String key) throws IllegalArgumentException
   {
      if (key == null)
      {
         throw new IllegalArgumentException("Key for System Property was not specified");
      }
   }
}
