/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.impl.base.config;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;

import org.jboss.bootstrap.api.config.ServerConfig;
import org.jboss.bootstrap.api.descriptor.BootstrapDescriptor;
import org.jboss.logging.Logger;

/**
 * Base for simple Object-backed implementations of a Server 
 * Configuration.  As this is exported from the Server, this 
 * is Thread-safe.
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public abstract class AbstractBasicServerConfig<T extends ServerConfig<T>> implements ServerConfig<T>
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   private static final Logger log = Logger.getLogger(AbstractBasicServerConfig.class);

   private static final String TRAILING_SLASH = "/";

   //-------------------------------------------------------------------------------------||
   // Instance Members -------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Actual class used in casting for covarient return in
    * method chaining. 
    */
   private volatile Class<T> actualClass;

   /**
    * Properties for this configuration.  Must be backed
    * by a Thread-safe impl.
    */
   private ConcurrentMap<String, String> properties;

   /**
    * Whether or not this configuration is frozen
    */
   private volatile boolean frozen;

   /**
    * Bootstrap descriptors
    */
   private List<BootstrapDescriptor> bootstrapDescriptors;

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Constructor
    * 
    * @param actualClass The actual class of this Server Configuration
    *       to be used in casting for covarient return types
    */
   public AbstractBasicServerConfig(final Class<T> actualClass)
   {
      // Use no override properties
      this(actualClass, null);
   }

   /**
    * Constructor
    * 
    * @param actualClass The actual class of this Server Configuration
    *       to be used in casting for covarient return types
    * @param overrideProperties A Map of properties to override those found 
    *           in System Properties
    */
   public AbstractBasicServerConfig(final Class<T> actualClass, final Map<String, String> overrideProperties)
   {
      /*
       * Initialize properties
       */
      Properties sysProps = System.getProperties();
      ConcurrentMap<String, String> properties = new ConcurrentHashMap<String, String>();
      Set<Object> sysPropKeys = sysProps.keySet();
      for (Object sysPropKey : sysPropKeys)
      {
         // Cast, all Property keys are Strings
         String sysPropKeyString = (String) sysPropKey;
         String value = sysProps.getProperty(sysPropKeyString);
         properties.put(sysPropKeyString, value);
      }

      /*
       * Override properties
       */
      if (overrideProperties != null)
      {
         properties.putAll(overrideProperties);
      }

      /*
       * Set properties
       */
      this.properties = properties;
      this.actualClass = actualClass;
      this.frozen = false;
      this.bootstrapDescriptors = new CopyOnWriteArrayList<BootstrapDescriptor>();
   }

   //-------------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#getProperties()
    */
   public Map<String, String> getProperties()
   {
      return Collections.unmodifiableMap(this.properties);
   }

   /**
    * @see org.jboss.bootstrap.api.config.ServerConfig#getBootstrapDescriptors()
    */
   public List<BootstrapDescriptor> getBootstrapDescriptors()
   {
      // If frozen, don't expose a mutable view
      if (this.isFrozen())
      {
         return Collections.unmodifiableList(this.bootstrapDescriptors);
      }
      else
      {
         return this.bootstrapDescriptors;
      }
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#getProperty(java.lang.String)
    */
   public String getProperty(final String key) throws IllegalArgumentException
   {
      // Precondition check
      if (key == null)
      {
         throw new IllegalArgumentException("key must be specified");
      }

      // Obtain
      final String value = this.properties.get(key);

      // Return 
      return value;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#property(java.lang.String, java.lang.String)
    */
   public T property(final String key, final String value) throws IllegalArgumentException
   {
      // Precondition checks
      this.checkMutable();
      if (key == null)
      {
         throw new IllegalArgumentException("Key may not be null");
      }

      // Set property
      this.setPropertyForString(key, value);

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#properties(java.util.Map)
    */
   public T properties(final Map<String, String> properties) throws IllegalArgumentException, IllegalStateException
   {
      // Precondition checks
      this.checkMutable();
      if (properties == null)
      {
         throw new IllegalArgumentException("properties may not be null");
      }

      // Set properties
      for (final String propName : properties.keySet())
      {
         final String propValue = properties.get(propName);
         this.setPropertyForString(propName, propValue);
      }

      // Return
      return this.covarientReturn();
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#properties(java.util.Properties)
    */
   public T properties(final Properties properties) throws IllegalArgumentException, IllegalStateException
   {
      // Precondition check
      this.checkMutable();
      if (properties == null)
      {
         throw new IllegalArgumentException("properties may not be null");
      }

      // Set properties
      for (final Object propName : properties.keySet())
      {
         // Get the key and cast
         String propNameString = null;
         try
         {
            propNameString = (String) propName;
         }
         catch (final ClassCastException cce)
         {
            throw new IllegalArgumentException("Properties supported must be in String/String pairs, but found: "
                  + propName + " of type: " + propName.getClass().getName(), cce);
         }

         // Get the value and cast
         final Object propValue = properties.get(propNameString);
         String propValueString = null;
         try
         {
            propValueString = (String) propValue;
         }
         catch (final ClassCastException cce)
         {
            throw new IllegalArgumentException("Properties supported must be in String/String pairs, but found: "
                  + propName + " with value: " + propValue + "of type: " + propValue.getClass().getName(), cce);
         }

         // Set
         this.setPropertyForString(propNameString, propValueString);
      }

      // Return
      return this.covarientReturn();
   }

   /**
    * @see org.jboss.bootstrap.api.config.ServerConfig#freeze()
    */
   public void freeze() throws IllegalStateException
   {
      // Precondition checks
      if (this.isFrozen())
      {
         throw new IllegalStateException("Configuration is already frozen");
      }

      // Make a defensive copy of the bootstrap descriptors
      final List<BootstrapDescriptor> copy = new ArrayList<BootstrapDescriptor>(bootstrapDescriptors.size());
      copy.addAll(bootstrapDescriptors);
      // Make immutable
      bootstrapDescriptors = Collections.unmodifiableList(copy);

      // Mark
      this.frozen = true;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ServerConfig#isFrozen()
    */
   public boolean isFrozen()
   {
      return this.frozen;
   }

   //-------------------------------------------------------------------------------------||
   // Helper Methods ---------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Sets both the configuration property and System property with the 
    * specified key and the external form of the specified URL
    */
   protected void setPropertyForUrl(final String propertyName, final URL url)
   {
      final String urlString = url != null ? url.toExternalForm() : null;
      this.setPropertyForString(propertyName, urlString);
   }

   /**
    * Sets both the configuration property and System property with the 
    * specified key and specified value
    */
   protected void setPropertyForString(final String propertyName, final String value)
   {
      final Map<String, String> properties = this.properties;
      String valueToSet = value;
      if (valueToSet == null)
      {
         valueToSet = "";
      }
      properties.put(propertyName, valueToSet);
      if (log.isTraceEnabled())
      {
         log.trace(("Set property \"" + propertyName + "\" to: ") + (value != null ? "\"" + value + "\"" : "[EMPTY]"));
      }
   }

   /**
    * Sets both the configuration property and System property with the 
    * specified key and specified value
    */
   protected void setPropertyForString(final String propertyName, final Integer value)
   {
      final String stringValue = value != null ? value.toString() : null;
      this.setPropertyForString(propertyName, stringValue);
   }

   /**
    * Sets both the configuration property and System property with the 
    * specified key and specified value
    */
   protected void setPropertyForString(final String propertyName, final Boolean value)
   {
      final String stringValue = value != null ? value.toString() : null;
      this.setPropertyForString(propertyName, stringValue);
   }

   /**
    * Throws IllegalStateException if this configuration is frozen
    * 
    * @throws IllegalStateException
    */
   protected void checkMutable() throws IllegalStateException
   {
      if (this.isFrozen())
      {
         throw new IllegalStateException("Cannot mutate state of the configuration after it has been frozen");
      }
   }

   /**
    * If the specified URL denotes a directory (ie. trailing slash), 
    * will return the argument.  Nulls are ignored and returned as-is.
    * 
    * @param url
    * @return
    */
   protected URL adjustToDirectory(final URL url)
   {
      // Do nothing to nulls
      if (url == null)
      {
         return null;
      }

      // Check
      final String externalForm = url.toExternalForm();
      if (externalForm.endsWith(TRAILING_SLASH))
      {
         // Return per normal
         return url;
      }

      // Otherwise make a new URL
      final String newLocation = externalForm + TRAILING_SLASH;
      URL newUrl = null;
      try
      {
         // Construct
         newUrl = new URL(newLocation);
      }
      catch (MalformedURLException e)
      {
         throw new RuntimeException("Could not create new URL to point to directory", e);
      }

      // Log
      log.tracef("Appended trailing slash to %s to point to directory: %s", url, newUrl);

      // Return it
      return newUrl;
   }

   /**
    * Returns the actual types of this configuration, used in 
    * covarient return of the mutator methods
    * 
    * @return
    */
   protected Class<T> getActualClass()
   {
      return this.actualClass;
   }

   /**
    * Casts this configuration to the requisite type, using 
    * the actual implementation class.  This is in place to
    * avoid unchecked casting (ie. (T)this)) and the resultant compiler 
    * warnings
    * 
    * @return
    */
   protected final T covarientReturn() throws ClassCastException
   {
      try
      {
         return this.getActualClass().cast(this);
      }
      catch (ClassCastException cce)
      {
         throw new RuntimeException(
               "Could not return the expected type; ensure that this configuration was constructed with the correct actual class",
               cce);
      }
   }

}
