/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.spi.server;

import org.jboss.bootstrap.spi.Bootstrap;
import org.jboss.bootstrap.spi.config.ConfigurationInitializer;
import org.jboss.bootstrap.spi.config.ConfigurationValidator;
import org.jboss.bootstrap.spi.config.InvalidConfigurationException;
import org.jboss.bootstrap.spi.config.ServerConfig;
import org.jboss.bootstrap.spi.lifecycle.LifecycleState;

/**
 * Server
 * 
 * Contract for generic Server implementations
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public interface Server<K extends Server<K, T>, T extends ServerConfig<T>>
{

   //-------------------------------------------------------------------------------------||
   // Contracts --------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Prepares the server for startup.  If {@link Server#getServerInitializer()} is not 
    * null, will invoke {@link ServerInitializer#initialize(Server)}.
    * Freezes the configuration from further 
    * mutable actions by invoking {@link ServerConfig#freeze()}.  Finally
    * sets the current state to {@link LifecycleState#IDLE}
    * 
    * @throws IllegalStateException  If the server is not in state {@link LifecycleState#PRE_INIT}, 
    *       or if the configuration has not been defined
    * @throws InvalidConfigurationException
    */
   void initialize() throws IllegalStateException, InvalidConfigurationException;

   /**
    * Returns the initializer to be used in initialization, or null
    * if none is defined
    * 
    * @return
    */
   ServerInitializer<K, T> getServerInitializer();

   /**
    * Initializer to use in server initialization.  Pass
    * null to disable further initialization.
    * 
    * @param serverInitializer
    * @return This server
    * @throws IllegalStateException If the server state is anything aside from
    * {@link LifecycleState#PRE_INIT}
    */
   void setServerInitializer(ServerInitializer<K, T> serverInitializer) throws IllegalStateException;

   /**
    * Returns the initializer for the underlying {@link ServerConfig} 
    * 
    * @return
    */
   ConfigurationInitializer<T> getConfigInitializer();

   /**
    * Sets the initializer to be used upon the configuration.  Pass
    * <code>null</code> to remove the initializer.
    * 
    * @param configInitializer
    * @return This server
    * @throws IllegalStateException If the server state is anything aside from
    * {@link LifecycleState#PRE_INIT}
    */
   void setConfigInitializer(ConfigurationInitializer<T> configInitializer) throws IllegalStateException;

   /**
    * Start lifecycle of the Server, optionally invoking upon
    * {@link Server#initialize()} if the state is {@link LifecycleState#PRE_INIT}.
    * During execution the state will be {@link LifecycleState#STARTING}, and upon
    * successful completion will be {@link LifecycleState#STARTED}. 
    * 
    * @throws IllegalStateException If the Server is already started or starting
    * @throws Exception
    */
   void start() throws IllegalStateException, Exception;

   /**
    * Stop lifecycle of the Server.  During execution the state will be 
    * {@link LifecycleState#STOPPING}, and upon
    * successful completion will be {@link LifecycleState#IDLE}.  If a 
    * server initialzer is defined then {@link ServerInitializer#cleanup(Server)}
    * will be invoked during this call.
    * 
    * @throws IllegalStateException If the Server is not started
    * @throws Exception
    */
   void shutdown() throws IllegalStateException, Exception;

   /**
    * Returns the current state of the Server lifecycle
    * 
    * @return The {@link LifecycleState}
    */
   LifecycleState getState();

   /**
    * Returns the underlying server configuration.  If
    * the server state is anything aside from 
    * {@link LifecycleState#PRE_INIT}, the configuration 
    * will be immutable / frozen.
    * 
    * @return
    */
   T getConfiguration();

   /**
    * Sets the configuration
    * 
    * @param config
    * @return This server
    */
   void setConfiguration(T config);

   /**
    * Returns the (possibly null) validator
    * to be used in asserting the validity of a supplied configuration
    * 
    * @return
    */
   ConfigurationValidator<T> getValidator();

   /**
    * Sets the specified validator to be used in 
    * asserting the validity of a supplied configuration. 
    * May be null to remove validation.
    * 
    * @param validator
    * @return This server
    * @throws IllegalStateException If the server state is anything aside from
    * {@link LifecycleState#PRE_INIT}
    */
   void setValidator(ConfigurationValidator<T> validator) throws IllegalStateException;

   /**
    * Adds the specified bootstrap to those to be run on {@link Server#start()}
    * 
    * @param bootstrap
    * @throws IllegalArgumentException If the specified bootstrap is null
    */
   void addBootstrap(Bootstrap bootstrap) throws IllegalArgumentException;

   /**
    * Removes the specified bootstrap from those to be run on {@link Server#start()}
    * 
    * @param bootstrap
    * @throws IllegalArgumentException If the specified bootstrap is not currently
    *       designated as to be run
    */
   void removeBootstrap(Bootstrap bootstrap) throws IllegalArgumentException;
}
