/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.spi.server;

import org.jboss.bootstrap.api.config.ServerConfig;
import org.jboss.bootstrap.api.lifecycle.LifecycleState;
import org.jboss.bootstrap.api.server.Server;
import org.jboss.bootstrap.spi.Bootstrap;
import org.jboss.bootstrap.spi.config.ConfigurationInitializer;
import org.jboss.bootstrap.spi.config.ConfigurationValidator;

/**
 * Server
 * 
 * Contract for generic Server implementations; not intended
 * to be exposed as a client API
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public interface ServerProvider<K extends Server<K, T>, T extends ServerConfig<T>> extends Server<K, T>
{
   //-------------------------------------------------------------------------------------||
   // Constants --------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /** 
    * The JMX notification event type sent on end of server startup 
    */
   public final String START_NOTIFICATION_TYPE = "org.jboss.system.server.started";

   /** 
    * The JMX notification event type sent on begin of the server shutdown  
    */
   public final String STOP_NOTIFICATION_TYPE = "org.jboss.system.server.stopped";

   //-------------------------------------------------------------------------------------||
   // Contracts --------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Returns the initializer for the underlying {@link ServerConfig} 
    * 
    * @return
    */
   ConfigurationInitializer<T> getConfigInitializer();

   /**
    * Sets the initializer to be used upon the configuration.  Pass
    * <code>null</code> to remove the initializer.
    * 
    * @param configInitializer
    * @return This server
    * @throws IllegalStateException If the server state is anything aside from
    * {@link LifecycleState#PRE_INIT}
    */
   void setConfigInitializer(ConfigurationInitializer<T> configInitializer) throws IllegalStateException;

   /**
    * Returns the initializer to be used in initialization, or null
    * if none is defined
    * 
    * @return
    */
   ServerInitializer<K, T> getServerInitializer();

   /**
    * Returns the (possibly null) validator
    * to be used in asserting the validity of a supplied configuration
    * 
    * @return
    */
   ConfigurationValidator<T> getValidator();

   /**
    * Sets the specified validator to be used in 
    * asserting the validity of a supplied configuration. 
    * May be null to remove validation.
    * 
    * @param validator
    * @return This server
    * @throws IllegalStateException If the server state is anything aside from
    * {@link LifecycleState#PRE_INIT}
    */
   void setValidator(ConfigurationValidator<T> validator) throws IllegalStateException;

   /**
    * Initializer to use in server initialization.  Pass
    * null to disable further initialization.
    * 
    * @param serverInitializer
    * @return This server
    * @throws IllegalStateException If the server state is anything aside from
    * {@link LifecycleState#PRE_INIT}
    */
   void setServerInitializer(ServerInitializer<K, T> serverInitializer) throws IllegalStateException;

   /**
    * Adds the specified bootstrap to those to be run on {@link ServerProvider#start()}
    * 
    * @param bootstrap
    * @throws IllegalArgumentException If the specified bootstrap is null
    */
   void addBootstrap(Bootstrap<K, T> bootstrap) throws IllegalArgumentException;

   /**
    * Removes the specified bootstrap from those to be run on {@link ServerProvider#start()}
    * 
    * @param bootstrap
    * @throws IllegalArgumentException If the specified bootstrap is not currently
    *       designated as to be run
    */
   void removeBootstrap(Bootstrap<K, T> bootstrap) throws IllegalArgumentException;
}
