/*
 * Copyright 2016 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.kie.dmn.feel.runtime.functions;

import org.kie.dmn.api.feel.runtime.events.FEELEvent.Severity;
import org.kie.dmn.feel.runtime.events.InvalidParametersEvent;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

public class SplitFunction
        extends BaseFEELFunction {

    public SplitFunction() {
        super( "split" );
    }

    public FEELFnResult<List<String>> invoke(@ParameterName("input") String input, @ParameterName("delimiter") String delimiter) {
        return invoke( input, delimiter, null );
    }

    public FEELFnResult<List<String>> invoke(@ParameterName("input") String input, @ParameterName("delimiter") String delimiter, @ParameterName("flags") String flags) {
        if ( input == null ) {
            return FEELFnResult.ofError( new InvalidParametersEvent( Severity.ERROR, "input", "cannot be null" ) );
        }
        if ( delimiter == null ) {
            return FEELFnResult.ofError( new InvalidParametersEvent( Severity.ERROR, "delimiter", "cannot be null" ) );
        }
        try {
            int f = processFlags( flags );
            Pattern p = Pattern.compile( delimiter, f );
            String[] split = p.split( input );
            return FEELFnResult.ofResult( Arrays.asList( split ) );
        } catch ( PatternSyntaxException e ) {
            return FEELFnResult.ofError( new InvalidParametersEvent( Severity.ERROR, "delimiter", "is invalid and can not be compiled", e ) );
        } catch ( IllegalArgumentException t ) {
            return FEELFnResult.ofError( new InvalidParametersEvent( Severity.ERROR, "flags", "contains unknown flags", t ) );
        } catch ( Throwable t) {
            return FEELFnResult.ofError( new InvalidParametersEvent( Severity.ERROR, "delimiter", "is invalid and can not be compiled", t ) );
        }
    }

    private int processFlags(String flags) {
        int f = 0;
        if( flags != null ) {
            if( flags.contains( "s" ) ) {
                f |= Pattern.DOTALL;
            }
            if( flags.contains( "m" ) ) {
                f |= Pattern.MULTILINE;
            }
            if( flags.contains( "i" ) ) {
                f |= Pattern.CASE_INSENSITIVE;
            }
        }
        return f;
    }

}
