/*
 * Copyright 2011 JBoss Inc
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package org.uberfire.client.tables;

import java.util.List;

import com.github.gwtbootstrap.client.ui.Button;
import com.github.gwtbootstrap.client.ui.DataGrid;
import com.github.gwtbootstrap.client.ui.Label;
import com.google.gwt.core.client.GWT;
import com.google.gwt.dom.client.Style;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.uibinder.client.UiBinder;
import com.google.gwt.uibinder.client.UiField;
import com.google.gwt.user.cellview.client.Column;
import com.google.gwt.user.cellview.client.Header;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.RequiresResize;
import com.google.gwt.user.client.ui.Widget;
import com.google.gwt.view.client.CellPreviewEvent.Handler;
import com.google.gwt.view.client.HasData;
import com.google.gwt.view.client.ProvidesKey;
import com.google.gwt.view.client.Range;
import com.google.gwt.view.client.RangeChangeEvent;
import com.google.gwt.view.client.RowCountChangeEvent;
import com.google.gwt.view.client.SelectionModel;

/**
 * A composite Widget that shows rows of data (not-paged) and a "column picker"
 * to allow columns to be hidden from view. Columns can also be sorted.
 */
public class SimpleTable<T>
        extends Composite
        implements HasData<T>,
                   RequiresResize {

    interface Binder
            extends
            UiBinder<Widget, SimpleTable> {

    }

    private static Binder uiBinder = GWT.create( Binder.class );

    @UiField(provided = true)
    public Button columnPickerButton;

    @UiField(provided = true)
    public DataGrid<T> dataGrid;

    @UiField
    public HorizontalPanel buttonContainer;

    private String emptyTableCaption;

    private ColumnPicker<T> columnPicker;

    public SimpleTable() {
        dataGrid = new DataGrid<T>();
        setupGridTable();
    }

    public SimpleTable( final ProvidesKey<T> providesKey ) {
        this( Integer.MAX_VALUE,
              providesKey );
    }

    public SimpleTable( final int pageSize,
                        final ProvidesKey<T> providesKey ) {
        dataGrid = new DataGrid<T>( pageSize,
                                    providesKey );
        setupGridTable();
    }

    private void setupGridTable() {
        dataGrid.setStriped( true );
        dataGrid.setBordered( true );
        dataGrid.setSkipRowHoverCheck( false );
        dataGrid.setSkipRowHoverStyleUpdate( false );
        dataGrid.setWidth( "100%" );
        dataGrid.setHeight( "300px" );
        setEmptyTableWidget();

        columnPicker = new ColumnPicker<T>( dataGrid );
        columnPickerButton = columnPicker.createToggleButton();

        initWidget( makeWidget() );
    }

    protected Widget makeWidget() {
        return uiBinder.createAndBindUi( this );
    }

    public void setEmptyTableCaption( final String emptyTableCaption ) {
        this.emptyTableCaption = emptyTableCaption;
        setEmptyTableWidget();
    }

    private void setEmptyTableWidget() {
        String caption = "-----";
        if ( !( emptyTableCaption == null || emptyTableCaption.trim().isEmpty() ) ) {
            caption = emptyTableCaption;
        }
        dataGrid.setEmptyTableWidget( new Label( caption ) );
    }

    public void redraw() {
        dataGrid.redraw();
    }

    public void refresh() {
        dataGrid.setVisibleRangeAndClearData( dataGrid.getVisibleRange(),
                                              true );
    }

    public HandlerRegistration addCellPreviewHandler( final Handler<T> handler ) {
        return dataGrid.addCellPreviewHandler( handler );
    }

    public HandlerRegistration addRangeChangeHandler( final RangeChangeEvent.Handler handler ) {
        return dataGrid.addRangeChangeHandler( handler );
    }

    public HandlerRegistration addRowCountChangeHandler( final RowCountChangeEvent.Handler handler ) {
        return dataGrid.addRowCountChangeHandler( handler );
    }

    public int getRowCount() {
        return dataGrid.getRowCount();
    }

    public Range getVisibleRange() {
        return dataGrid.getVisibleRange();
    }

    public boolean isRowCountExact() {
        return dataGrid.isRowCountExact();
    }

    public void setRowCount( final int count ) {
        dataGrid.setRowCount( count );
    }

    public void setRowCount( final int count,
                             final boolean isExact ) {
        dataGrid.setRowCount( count,
                              isExact );
    }

    public void setVisibleRange( final int start,
                                 final int length ) {
        dataGrid.setVisibleRange( start,
                                  length );
    }

    public void setVisibleRange( final Range range ) {
        dataGrid.setVisibleRange( range );
    }

    public SelectionModel<? super T> getSelectionModel() {
        return dataGrid.getSelectionModel();
    }

    public T getVisibleItem( final int indexOnPage ) {
        return dataGrid.getVisibleItem( indexOnPage );
    }

    public int getVisibleItemCount() {
        return dataGrid.getVisibleItemCount();
    }

    public Iterable<T> getVisibleItems() {
        return dataGrid.getVisibleItems();
    }

    public void setRowData( final int start,
                            final List<? extends T> values ) {
        dataGrid.setRowData( start,
                             values );
    }

    public void setRowData( final List<? extends T> values ) {
        dataGrid.setRowData( values );
    }

    public void setSelectionModel( final SelectionModel<? super T> selectionModel ) {
        dataGrid.setSelectionModel( selectionModel );
    }

    public void setVisibleRangeAndClearData( final Range range,
                                             final boolean forceRangeChangeEvent ) {
        dataGrid.setVisibleRangeAndClearData( range,
                                              forceRangeChangeEvent );
    }

    public void addColumn( final Column<T, ?> column,
                           final String caption ) {
        final Header header = new ResizableMovableHeader<T>( caption,
                                                             dataGrid,
                                                             columnPicker,
                                                             column ) {
            @Override
            protected int getTableBodyHeight() {
                return dataGrid.getOffsetHeight();
            }
        };
        columnPicker.addColumn( column,
                                header,
                                true );
    }

    public void addColumn( final Column<T, ?> column,
                           final String caption,
                           final boolean visible ) {
        final Header header = new ResizableHeader<T>( caption,
                                                      dataGrid,
                                                      column );
        columnPicker.addColumn( column,
                                header,
                                visible );
    }

    public void setColumnWidth( final Column<T, ?> column,
                                final double width,
                                final Style.Unit unit ) {
        dataGrid.setColumnWidth( column,
                                 width,
                                 unit );
    }

    @Override
    public void onResize() {
        final Widget parent = getParent();
        if ( parent == null ) {
            return;
        }
        final int width = parent.getOffsetWidth();
        final int height = parent.getOffsetHeight();
        if ( width < 0 ) {
            return;
        }
        if ( height < 0 ) {
            return;
        }
        if ( height - 32 >= 0 ) {
//            dataGrid.setPixelSize( width,
//                                   height - 32 );
        }
    }

    @Override
    public void setHeight( String height ) {
        dataGrid.setHeight( height );
    }

    @Override
    public void setPixelSize( int width,
                              int height ) {
        dataGrid.setPixelSize( width, height );
    }

    @Override
    public void setSize( String width,
                         String height ) {
        dataGrid.setSize( width, height );
    }

    @Override
    public void setWidth( String width ) {
        dataGrid.setWidth( width );
    }

    public void addButton( final Button button ) {
        buttonContainer.add( button );
    }

}
