/*
 * Copyright The WildFly Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package org.wildfly.clustering.server.util;

import java.util.Map;
import java.util.Objects;
import java.util.function.Function;

/**
 * An immutable map entry.  This implementation differs from {@link Map#entry(Object, Object)} in the following ways:
 * <ul>
 * <li>Supports null key or value</li>
 * <li>Supports mapping operation</li>
 * </ul>
 * @author Paul Ferraro
 * @param <K> key type
 * @param <V> value type
 */
public interface MapEntry<K, V> extends Map.Entry<K, V> {

	/**
	 * Returns a new map entry whose key/value are generated by applying the specified mapping functions.
	 * @param <KK> the mapped key type
	 * @param <VV> the mapped value type
	 * @param keyMapper a key mapper
	 * @param valueMapper a value mapper
	 * @return a new map entry
	 */
	<KK, VV> MapEntry<KK, VV> map(Function<K, KK> keyMapper, Function<V, VV> valueMapper);

	@Override
	default V setValue(V value) {
		throw new UnsupportedOperationException();
	}

	/**
	 * Returns a new immutable map entry from the specified key and value.
	 * @param <K> the key type
	 * @param <V> the value type
	 * @param key the map entry key
	 * @param value the map entry value
	 * @return a new map entry
	 */
	static <K, V> MapEntry<K, V> of(K key, V value) {
		return new MapEntry<>() {
			@Override
			public K getKey() {
				return key;
			}

			@Override
			public V getValue() {
				return value;
			}

			@Override
			public <KK, VV> MapEntry<KK, VV> map(Function<K, KK> keyMapper, Function<V, VV> valueMapper) {
				return of(keyMapper.apply(key), valueMapper.apply(value));
			}

			@Override
			public int hashCode() {
				return (key != null) ? key.hashCode() : 0;
			}

			@Override
			public boolean equals(Object object) {
				if (!(object instanceof Map.Entry)) return false;
				@SuppressWarnings("unchecked")
				Map.Entry<Object, Object> entry = (Map.Entry<Object, Object>) object;
				return Objects.equals(key, entry.getKey()) && Objects.equals(value, entry.getValue());
			}

			@Override
			public String toString() {
				return key + "=" + value;
			}
		};
	}
}
