/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.okhttp;

import com.google.common.base.Preconditions;
import io.grpc.okhttp.OkHttpClientStream;
import io.grpc.okhttp.OkHttpClientTransport;
import io.grpc.okhttp.internal.framed.FrameWriter;
import java.io.IOException;
import javax.annotation.Nullable;
import okio.Buffer;

class OutboundFlowController {
    private final OkHttpClientTransport transport;
    private final FrameWriter frameWriter;
    private int initialWindowSize;
    private final OutboundFlowState connectionState;

    OutboundFlowController(OkHttpClientTransport transport, FrameWriter frameWriter) {
        this.transport = (OkHttpClientTransport)Preconditions.checkNotNull((Object)transport, (Object)"transport");
        this.frameWriter = (FrameWriter)Preconditions.checkNotNull((Object)frameWriter, (Object)"frameWriter");
        this.initialWindowSize = 65535;
        this.connectionState = new OutboundFlowState(0, 65535);
    }

    boolean initialOutboundWindowSize(int newWindowSize) {
        if (newWindowSize < 0) {
            throw new IllegalArgumentException("Invalid initial window size: " + newWindowSize);
        }
        int delta = newWindowSize - this.initialWindowSize;
        this.initialWindowSize = newWindowSize;
        for (OkHttpClientStream stream : this.transport.getActiveStreams()) {
            OutboundFlowState state = (OutboundFlowState)stream.getOutboundFlowState();
            if (state == null) {
                state = new OutboundFlowState(stream, this.initialWindowSize);
                stream.setOutboundFlowState(state);
                continue;
            }
            state.incrementStreamWindow(delta);
        }
        return delta > 0;
    }

    int windowUpdate(@Nullable OkHttpClientStream stream, int delta) {
        int updatedWindow;
        if (stream == null) {
            updatedWindow = this.connectionState.incrementStreamWindow(delta);
            this.writeStreams();
        } else {
            OutboundFlowState state = this.state(stream);
            updatedWindow = state.incrementStreamWindow(delta);
            WriteStatus writeStatus = new WriteStatus();
            state.writeBytes(state.writableWindow(), writeStatus);
            if (writeStatus.hasWritten()) {
                this.flush();
            }
        }
        return updatedWindow;
    }

    void data(boolean outFinished, int streamId, Buffer source, boolean flush) {
        Preconditions.checkNotNull((Object)source, (Object)"source");
        OkHttpClientStream stream = this.transport.getStream(streamId);
        if (stream == null) {
            return;
        }
        OutboundFlowState state = this.state(stream);
        int window = state.writableWindow();
        boolean framesAlreadyQueued = state.hasPendingData();
        int size = (int)source.size();
        if (!framesAlreadyQueued && window >= size) {
            state.write(source, size, outFinished);
        } else {
            if (!framesAlreadyQueued && window > 0) {
                state.write(source, window, false);
            }
            state.enqueue(source, (int)source.size(), outFinished);
        }
        if (flush) {
            this.flush();
        }
    }

    void flush() {
        try {
            this.frameWriter.flush();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private OutboundFlowState state(OkHttpClientStream stream) {
        OutboundFlowState state = (OutboundFlowState)stream.getOutboundFlowState();
        if (state == null) {
            state = new OutboundFlowState(stream, this.initialWindowSize);
            stream.setOutboundFlowState(state);
        }
        return state;
    }

    void writeStreams() {
        OutboundFlowState state;
        OkHttpClientStream[] streams = this.transport.getActiveStreams();
        int connectionWindow = this.connectionState.window();
        int numStreams = streams.length;
        while (numStreams > 0 && connectionWindow > 0) {
            int nextNumStreams = 0;
            int windowSlice = (int)Math.ceil((float)connectionWindow / (float)numStreams);
            for (int index = 0; index < numStreams && connectionWindow > 0; ++index) {
                OkHttpClientStream stream = streams[index];
                state = this.state(stream);
                int bytesForStream = Math.min(connectionWindow, Math.min(state.unallocatedBytes(), windowSlice));
                if (bytesForStream > 0) {
                    state.allocateBytes(bytesForStream);
                    connectionWindow -= bytesForStream;
                }
                if (state.unallocatedBytes() <= 0) continue;
                streams[nextNumStreams++] = stream;
            }
            numStreams = nextNumStreams;
        }
        WriteStatus writeStatus = new WriteStatus();
        for (OkHttpClientStream stream : this.transport.getActiveStreams()) {
            state = this.state(stream);
            state.writeBytes(state.allocatedBytes(), writeStatus);
            state.clearAllocatedBytes();
        }
        if (writeStatus.hasWritten()) {
            this.flush();
        }
    }

    private final class OutboundFlowState {
        final Buffer pendingWriteBuffer;
        final int streamId;
        int window;
        int allocatedBytes;
        OkHttpClientStream stream;
        boolean pendingBufferHasEndOfStream = false;

        OutboundFlowState(int streamId, int initialWindowSize) {
            this.streamId = streamId;
            this.window = initialWindowSize;
            this.pendingWriteBuffer = new Buffer();
        }

        OutboundFlowState(OkHttpClientStream stream, int initialWindowSize) {
            this(stream.id(), initialWindowSize);
            this.stream = stream;
        }

        int window() {
            return this.window;
        }

        void allocateBytes(int bytes) {
            this.allocatedBytes += bytes;
        }

        int allocatedBytes() {
            return this.allocatedBytes;
        }

        int unallocatedBytes() {
            return this.streamableBytes() - this.allocatedBytes;
        }

        void clearAllocatedBytes() {
            this.allocatedBytes = 0;
        }

        int incrementStreamWindow(int delta) {
            if (delta > 0 && Integer.MAX_VALUE - delta < this.window) {
                throw new IllegalArgumentException("Window size overflow for stream: " + this.streamId);
            }
            this.window += delta;
            return this.window;
        }

        int writableWindow() {
            return Math.min(this.window, OutboundFlowController.this.connectionState.window());
        }

        int streamableBytes() {
            return Math.max(0, Math.min(this.window, (int)this.pendingWriteBuffer.size()));
        }

        boolean hasPendingData() {
            return this.pendingWriteBuffer.size() > 0L;
        }

        int writeBytes(int bytes, WriteStatus writeStatus) {
            int bytesAttempted = 0;
            int maxBytes = Math.min(bytes, this.writableWindow());
            while (this.hasPendingData() && maxBytes > 0) {
                if ((long)maxBytes >= this.pendingWriteBuffer.size()) {
                    bytesAttempted += (int)this.pendingWriteBuffer.size();
                    this.write(this.pendingWriteBuffer, (int)this.pendingWriteBuffer.size(), this.pendingBufferHasEndOfStream);
                } else {
                    bytesAttempted += maxBytes;
                    this.write(this.pendingWriteBuffer, maxBytes, false);
                }
                writeStatus.incrementNumWrites();
                maxBytes = Math.min(bytes - bytesAttempted, this.writableWindow());
            }
            return bytesAttempted;
        }

        void write(Buffer buffer, int bytesToSend, boolean endOfStream) {
            int frameBytes;
            int bytesToWrite = bytesToSend;
            do {
                frameBytes = Math.min(bytesToWrite, OutboundFlowController.this.frameWriter.maxDataLength());
                OutboundFlowController.this.connectionState.incrementStreamWindow(-frameBytes);
                this.incrementStreamWindow(-frameBytes);
                try {
                    boolean isEndOfStream = buffer.size() == (long)frameBytes && endOfStream;
                    OutboundFlowController.this.frameWriter.data(isEndOfStream, this.streamId, buffer, frameBytes);
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
                this.stream.transportState().onSentBytes(frameBytes);
            } while ((bytesToWrite -= frameBytes) > 0);
        }

        void enqueue(Buffer buffer, int size, boolean endOfStream) {
            this.pendingWriteBuffer.write(buffer, (long)size);
            this.pendingBufferHasEndOfStream |= endOfStream;
        }
    }

    private static final class WriteStatus {
        int numWrites;

        private WriteStatus() {
        }

        void incrementNumWrites() {
            ++this.numWrites;
        }

        boolean hasWritten() {
            return this.numWrites > 0;
        }
    }
}

