/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.core.postoffice.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.IntFunction;
import org.apache.activemq.artemis.api.core.ActiveMQDuplicateIdException;
import org.apache.activemq.artemis.api.core.ActiveMQException;
import org.apache.activemq.artemis.api.core.ObjLongPair;
import org.apache.activemq.artemis.api.core.Pair;
import org.apache.activemq.artemis.api.core.SimpleString;
import org.apache.activemq.artemis.core.persistence.StorageManager;
import org.apache.activemq.artemis.core.postoffice.DuplicateIDCache;
import org.apache.activemq.artemis.core.postoffice.impl.ByteArray;
import org.apache.activemq.artemis.core.postoffice.impl.IntegerCache;
import org.apache.activemq.artemis.core.server.ActiveMQServerLogger;
import org.apache.activemq.artemis.core.server.MessageReference;
import org.apache.activemq.artemis.core.transaction.Transaction;
import org.apache.activemq.artemis.core.transaction.TransactionOperationAbstract;
import org.apache.activemq.artemis.utils.ByteUtil;
import org.jboss.logging.Logger;

final class PersistentDuplicateIDCache
implements DuplicateIDCache {
    private static final Logger LOGGER = Logger.getLogger(PersistentDuplicateIDCache.class);
    private final Map<ByteArray, Integer> cache = new ConcurrentHashMap<ByteArray, Integer>();
    private final SimpleString address;
    private final ArrayList<ObjLongPair<ByteArray>> ids;
    private final IntFunction<Integer> cachedBoxedInts;
    private int pos;
    private final int cacheSize;
    private final StorageManager storageManager;

    PersistentDuplicateIDCache(SimpleString address, int size, StorageManager storageManager) {
        this.address = address;
        this.cacheSize = size;
        this.ids = new ArrayList(size);
        this.cachedBoxedInts = IntegerCache.boxedInts(size);
        this.storageManager = storageManager;
    }

    @Override
    public synchronized void load(List<Pair<byte[], Long>> ids) throws Exception {
        if (!this.cache.isEmpty()) {
            throw new IllegalStateException("load is valid only on empty cache");
        }
        long txID = -1L;
        int toNotBeAdded = ids.size() - this.cacheSize;
        if (toNotBeAdded < 0) {
            toNotBeAdded = 0;
        }
        for (Pair<byte[], Long> id : ids) {
            if (id.getB() == null) {
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.tracef("ignoring id = %s because without record ID", (Object)PersistentDuplicateIDCache.describeID((byte[])id.getA()));
                }
                if (toNotBeAdded <= 0) continue;
                --toNotBeAdded;
                continue;
            }
            assert (id.getB() != null && (Long)id.getB() != -1L);
            if (toNotBeAdded > 0) {
                if (txID == -1L) {
                    txID = this.storageManager.generateID();
                }
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.tracef("deleting id = %s", (Object)PersistentDuplicateIDCache.describeID((byte[])id.getA(), (Long)id.getB()));
                }
                this.storageManager.deleteDuplicateIDTransactional(txID, (Long)id.getB());
                --toNotBeAdded;
                continue;
            }
            ByteArray bah = new ByteArray((byte[])id.getA());
            ObjLongPair pair = new ObjLongPair((Object)bah, ((Long)id.getB()).longValue());
            this.cache.put(bah, this.cachedBoxedInts.apply(this.ids.size()));
            this.ids.add((ObjLongPair<ByteArray>)pair);
            if (!LOGGER.isTraceEnabled()) continue;
            LOGGER.tracef("loading id = %s", (Object)PersistentDuplicateIDCache.describeID((byte[])id.getA(), (Long)id.getB()));
        }
        if (txID != -1L) {
            this.storageManager.commit(txID);
        }
        this.pos = this.ids.size();
        if (this.pos == this.cacheSize) {
            this.pos = 0;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void deleteFromCache(byte[] duplicateID) throws Exception {
        ByteArray bah;
        Integer posUsed;
        if (LOGGER.isTraceEnabled()) {
            LOGGER.tracef("deleting id = %s", (Object)PersistentDuplicateIDCache.describeID(duplicateID));
        }
        if ((posUsed = this.cache.remove(bah = new ByteArray(duplicateID))) != null) {
            PersistentDuplicateIDCache persistentDuplicateIDCache = this;
            synchronized (persistentDuplicateIDCache) {
                ObjLongPair<ByteArray> id = this.ids.get(posUsed);
                if (((ByteArray)id.getA()).equals(bah)) {
                    long recordID = id.getB();
                    id.setA(null);
                    id.setB(-1L);
                    if (LOGGER.isTraceEnabled()) {
                        LOGGER.tracef("address = %s deleting id = %s", (Object)this.address, (Object)PersistentDuplicateIDCache.describeID(duplicateID, id.getB()));
                    }
                    this.storageManager.deleteDuplicateID(recordID);
                }
            }
        }
    }

    private static String describeID(byte[] duplicateID) {
        return ByteUtil.bytesToHex((byte[])duplicateID, (int)4) + ", simpleString=" + ByteUtil.toSimpleString((byte[])duplicateID);
    }

    private static String describeID(byte[] duplicateID, long id) {
        return ByteUtil.bytesToHex((byte[])duplicateID, (int)4) + ", simpleString=" + ByteUtil.toSimpleString((byte[])duplicateID) + ", id=" + id;
    }

    @Override
    public boolean contains(byte[] duplID) {
        return this.contains(new ByteArray(duplID));
    }

    private boolean contains(ByteArray duplID) {
        boolean contains = this.cache.containsKey(duplID);
        if (LOGGER.isTraceEnabled() && contains) {
            LOGGER.tracef("address = %s found a duplicate %s", (Object)this.address, (Object)PersistentDuplicateIDCache.describeID(duplID.bytes));
        }
        return contains;
    }

    @Override
    public void addToCache(byte[] duplID) throws Exception {
        this.addToCache(duplID, null, false);
    }

    @Override
    public void addToCache(byte[] duplID, Transaction tx) throws Exception {
        this.addToCache(duplID, tx, false);
    }

    @Override
    public synchronized boolean atomicVerify(byte[] duplID, Transaction tx) throws Exception {
        ByteArray holder = new ByteArray(duplID);
        if (this.contains(holder)) {
            if (tx != null) {
                tx.markAsRollbackOnly((ActiveMQException)new ActiveMQDuplicateIdException());
            }
            return false;
        }
        this.addToCache(holder, tx, true);
        return true;
    }

    @Override
    public synchronized void addToCache(byte[] duplID, Transaction tx, boolean instantAdd) throws Exception {
        this.addToCache(new ByteArray(duplID), tx, instantAdd);
    }

    private synchronized void addToCache(ByteArray holder, Transaction tx, boolean instantAdd) throws Exception {
        long recordID = this.storageManager.generateID();
        if (tx == null) {
            this.storageManager.storeDuplicateID(this.address, holder.bytes, recordID);
            this.addToCacheInMemory(holder, recordID);
        } else {
            this.storageManager.storeDuplicateIDTransactional(tx.getID(), this.address, holder.bytes, recordID);
            tx.setContainsPersistent();
            if (LOGGER.isTraceEnabled()) {
                LOGGER.tracef("address = %s adding duplicateID TX operation for %s, tx = %s", (Object)this.address, (Object)PersistentDuplicateIDCache.describeID(holder.bytes, recordID), (Object)tx);
            }
            if (instantAdd) {
                tx.addOperation(new AddDuplicateIDOperation(holder, recordID, false));
            } else {
                tx.afterStore(new AddDuplicateIDOperation(holder, recordID, true));
            }
        }
    }

    @Override
    public void load(Transaction tx, byte[] duplID) {
        tx.addOperation(new AddDuplicateIDOperation(new ByteArray(duplID), tx.getID(), true));
    }

    private synchronized void addToCacheInMemory(ByteArray holder, long recordID) {
        Objects.requireNonNull(holder, "holder must be not null");
        if (recordID < 0L) {
            throw new IllegalArgumentException("recordID must be >= 0");
        }
        if (LOGGER.isTraceEnabled()) {
            LOGGER.tracef("address = %s adding %s", (Object)this.address, (Object)PersistentDuplicateIDCache.describeID(holder.bytes, recordID));
        }
        this.cache.put(holder, this.cachedBoxedInts.apply(this.pos));
        if (this.pos < this.ids.size()) {
            ObjLongPair<ByteArray> id = this.ids.get(this.pos);
            if (id.getA() != null) {
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.tracef("address = %s removing excess duplicateDetection %s", (Object)this.address, (Object)PersistentDuplicateIDCache.describeID(((ByteArray)id.getA()).bytes, id.getB()));
                }
                this.cache.remove(id.getA());
                assert (id.getB() != -1L);
                try {
                    this.storageManager.deleteDuplicateID(id.getB());
                }
                catch (Exception e) {
                    ActiveMQServerLogger.LOGGER.errorDeletingDuplicateCache(e);
                }
            }
            id.setA((Object)holder);
            id.setB(recordID);
            if (LOGGER.isTraceEnabled()) {
                LOGGER.tracef("address = %s replacing old duplicateID by %s", (Object)this.address, (Object)PersistentDuplicateIDCache.describeID(((ByteArray)id.getA()).bytes, id.getB()));
            }
        } else {
            ObjLongPair id = new ObjLongPair((Object)holder, recordID);
            if (LOGGER.isTraceEnabled()) {
                LOGGER.tracef("address = %s adding new duplicateID %s", (Object)this.address, (Object)PersistentDuplicateIDCache.describeID(((ByteArray)id.getA()).bytes, id.getB()));
            }
            this.ids.add((ObjLongPair<ByteArray>)id);
        }
        if (this.pos++ == this.cacheSize - 1) {
            this.pos = 0;
        }
    }

    @Override
    public synchronized void clear() throws Exception {
        LOGGER.debugf("address = %s removing duplicate ID data", (Object)this.address);
        int idsSize = this.ids.size();
        if (idsSize > 0) {
            long tx = this.storageManager.generateID();
            for (int i = 0; i < idsSize; ++i) {
                ObjLongPair<ByteArray> id = this.ids.get(i);
                if (id.getA() == null) continue;
                assert (id.getB() != -1L);
                this.storageManager.deleteDuplicateIDTransactional(tx, id.getB());
            }
            this.storageManager.commit(tx);
        }
        this.ids.clear();
        this.cache.clear();
        this.pos = 0;
    }

    @Override
    public synchronized List<Pair<byte[], Long>> getMap() {
        int idsSize = this.ids.size();
        ArrayList<Pair<byte[], Long>> copy = new ArrayList<Pair<byte[], Long>>(idsSize);
        for (int i = 0; i < idsSize; ++i) {
            ObjLongPair<ByteArray> id = this.ids.get(i);
            if (id.getA() == null) continue;
            assert (id.getB() != -1L);
            copy.add((Pair<byte[], Long>)new Pair((Object)((ByteArray)id.getA()).bytes, (Object)id.getB()));
        }
        return copy;
    }

    private final class AddDuplicateIDOperation
    extends TransactionOperationAbstract {
        final ByteArray holder;
        final long recordID;
        volatile boolean done;
        private final boolean afterCommit;

        AddDuplicateIDOperation(ByteArray holder, long recordID, boolean afterCommit) {
            this.holder = holder;
            this.recordID = recordID;
            this.afterCommit = afterCommit;
        }

        private void process() {
            if (!this.done) {
                PersistentDuplicateIDCache.this.addToCacheInMemory(this.holder, this.recordID);
                this.done = true;
            }
        }

        @Override
        public void afterCommit(Transaction tx) {
            if (this.afterCommit) {
                this.process();
            }
        }

        @Override
        public void beforeCommit(Transaction tx) throws Exception {
            if (!this.afterCommit) {
                this.process();
            }
        }

        @Override
        public List<MessageReference> getRelatedMessageReferences() {
            return null;
        }
    }
}

