/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.tests.integration.stomp.v12;

import jakarta.jms.BytesMessage;
import jakarta.jms.Destination;
import jakarta.jms.JMSException;
import jakarta.jms.Message;
import jakarta.jms.MessageConsumer;
import jakarta.jms.MessageListener;
import jakarta.jms.MessageProducer;
import jakarta.jms.TextMessage;
import java.io.IOException;
import java.net.URI;
import java.nio.channels.ClosedChannelException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collection;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.regex.Pattern;
import org.apache.activemq.artemis.api.core.RoutingType;
import org.apache.activemq.artemis.api.core.SimpleString;
import org.apache.activemq.artemis.core.protocol.stomp.StompConnection;
import org.apache.activemq.artemis.core.settings.impl.AddressSettings;
import org.apache.activemq.artemis.tests.integration.stomp.StompTestBase;
import org.apache.activemq.artemis.tests.integration.stomp.util.ClientStompFrame;
import org.apache.activemq.artemis.tests.integration.stomp.util.StompClientConnection;
import org.apache.activemq.artemis.tests.integration.stomp.util.StompClientConnectionFactory;
import org.apache.activemq.artemis.tests.integration.stomp.util.StompClientConnectionV11;
import org.apache.activemq.artemis.tests.integration.stomp.util.StompClientConnectionV12;
import org.apache.activemq.artemis.utils.Wait;
import org.jboss.logmanager.Logger;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
public class StompV12Test
extends StompTestBase {
    private static final org.jboss.logging.Logger log = org.jboss.logging.Logger.getLogger(StompV12Test.class);
    public static final String CLIENT_ID = "myclientid";
    private StompClientConnectionV12 conn;
    private URI v10Uri;
    private URI v11Uri;

    @Parameterized.Parameters(name="{0}")
    public static Collection<Object[]> data() {
        return Arrays.asList({"ws+v12.stomp"}, {"tcp+v12.stomp"});
    }

    @Override
    @Before
    public void setUp() throws Exception {
        super.setUp();
        this.v10Uri = new URI(this.uri.toString().replace("v12", "v10"));
        this.v11Uri = new URI(this.uri.toString().replace("v12", "v11"));
        this.conn = (StompClientConnectionV12)StompClientConnectionFactory.createClientConnection(this.uri);
    }

    @Override
    @After
    public void tearDown() throws Exception {
        try {
            boolean connected = this.conn != null && this.conn.isConnected();
            log.debug((Object)("Connection 1.2 : " + connected));
            if (connected) {
                this.conn.disconnect();
            }
        }
        finally {
            super.tearDown();
            this.conn.closeTransport();
        }
    }

    @Test
    public void testSubscribeWithReceipt() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        Pattern p = Pattern.compile("receipt-id:.*\\nreceipt-id");
        StompV12Test.assertFalse((boolean)p.matcher(StompV12Test.subscribe(this.conn, null).toString()).find());
        this.conn.disconnect();
    }

    @Test
    public void testConnection() throws Exception {
        this.server.getSecurityStore().setSecurityEnabled(true);
        StompClientConnection connection = StompClientConnectionFactory.createClientConnection(this.v10Uri);
        connection.connect(this.defUser, this.defPass);
        Assert.assertTrue((boolean)connection.isConnected());
        Assert.assertEquals((Object)"1.0", (Object)connection.getVersion());
        connection.disconnect();
        connection = StompClientConnectionFactory.createClientConnection(this.uri);
        connection.connect(this.defUser, this.defPass);
        Assert.assertTrue((boolean)connection.isConnected());
        Assert.assertEquals((Object)"1.2", (Object)connection.getVersion());
        connection.disconnect();
        connection = StompClientConnectionFactory.createClientConnection(this.uri);
        connection.connect();
        Assert.assertFalse((boolean)connection.isConnected());
        StompClientConnectionV11 conn = (StompClientConnectionV11)StompClientConnectionFactory.createClientConnection(this.v11Uri);
        conn.connect1(this.defUser, this.defPass);
        Assert.assertTrue((boolean)conn.isConnected());
        conn.disconnect();
    }

    @Test
    public void testConnectionAsInSpec() throws Exception {
        StompClientConnection conn = StompClientConnectionFactory.createClientConnection(this.v10Uri);
        ClientStompFrame frame = conn.createFrame("CONNECT");
        frame.addHeader("login", this.defUser);
        frame.addHeader("passcode", this.defPass);
        frame.addHeader("accept-version", "1.2");
        frame.addHeader("host", "127.0.0.1");
        ClientStompFrame reply = conn.sendFrame(frame);
        Assert.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assert.assertEquals((Object)"1.2", (Object)reply.getHeader("version"));
        conn.disconnect();
        conn = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = conn.createFrame("STOMP");
        frame.addHeader("login", this.defUser);
        frame.addHeader("passcode", this.defPass);
        frame.addHeader("accept-version", "1.2");
        frame.addHeader("host", "127.0.0.1");
        reply = conn.sendFrame(frame);
        Assert.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assert.assertEquals((Object)"1.2", (Object)reply.getHeader("version"));
        conn.disconnect();
    }

    @Test
    public void testNegotiation() throws Exception {
        StompClientConnection conn = StompClientConnectionFactory.createClientConnection(this.v10Uri);
        ClientStompFrame frame = conn.createFrame("CONNECT");
        frame.addHeader("host", "127.0.0.1");
        frame.addHeader("login", this.defUser);
        frame.addHeader("passcode", this.defPass);
        ClientStompFrame reply = conn.sendFrame(frame);
        Assert.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assert.assertEquals(null, (Object)reply.getHeader("version"));
        conn.disconnect();
        conn = StompClientConnectionFactory.createClientConnection(this.v11Uri);
        frame = conn.createFrame("CONNECT");
        frame.addHeader("accept-version", "1.0");
        frame.addHeader("host", "127.0.0.1");
        frame.addHeader("login", this.defUser);
        frame.addHeader("passcode", this.defPass);
        reply = conn.sendFrame(frame);
        Assert.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assert.assertEquals((Object)"1.0", (Object)reply.getHeader("version"));
        conn.disconnect();
        conn = StompClientConnectionFactory.createClientConnection(this.v11Uri);
        frame = conn.createFrame("CONNECT");
        frame.addHeader("accept-version", "1.1");
        frame.addHeader("host", "127.0.0.1");
        frame.addHeader("login", this.defUser);
        frame.addHeader("passcode", this.defPass);
        reply = conn.sendFrame(frame);
        Assert.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assert.assertEquals((Object)"1.1", (Object)reply.getHeader("version"));
        conn.disconnect();
        conn = StompClientConnectionFactory.createClientConnection(this.v11Uri);
        frame = conn.createFrame("CONNECT");
        frame.addHeader("accept-version", "1.0,1.1,1.3");
        frame.addHeader("host", "127.0.0.1");
        frame.addHeader("login", this.defUser);
        frame.addHeader("passcode", this.defPass);
        reply = conn.sendFrame(frame);
        Assert.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assert.assertEquals((Object)"1.1", (Object)reply.getHeader("version"));
        conn.disconnect();
        conn = StompClientConnectionFactory.createClientConnection(this.v11Uri);
        frame = conn.createFrame("CONNECT");
        frame.addHeader("accept-version", "1.3");
        frame.addHeader("host", "127.0.0.1");
        frame.addHeader("login", this.defUser);
        frame.addHeader("passcode", this.defPass);
        reply = conn.sendFrame(frame);
        Assert.assertEquals((Object)"ERROR", (Object)reply.getCommand());
        conn.disconnect();
        this.instanceLog.debug((Object)("Got error frame " + reply));
    }

    @Test
    public void testSendAndReceive() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame response = StompV12Test.send(this.conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), "text/plain", "Hello World 1!");
        Assert.assertNull((Object)response);
        StompV12Test.send(this.conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), "text/plain", "Hello World 2!", true);
        StompClientConnection newConn = StompClientConnectionFactory.createClientConnection(this.uri);
        newConn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(newConn, "a-sub");
        ClientStompFrame frame = newConn.receiveFrame();
        this.instanceLog.debug((Object)("received " + frame));
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)"a-sub", (Object)frame.getHeader("subscription"));
        Assert.assertNull((Object)frame.getHeader("ack"));
        Assert.assertEquals((Object)(StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()), (Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)"Hello World 1!", (Object)frame.getBody());
        frame = newConn.receiveFrame();
        this.instanceLog.debug((Object)("received " + frame));
        StompV12Test.unsubscribe(newConn, "a-sub");
        newConn.disconnect();
    }

    @Test
    public void testHeaderContentType() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.send(this.conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), "application/xml", "Hello World 1!");
        StompClientConnection newConn = StompClientConnectionFactory.createClientConnection(this.v11Uri);
        newConn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(newConn, "a-sub");
        ClientStompFrame frame = newConn.receiveFrame();
        this.instanceLog.debug((Object)("received " + frame));
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)"application/xml", (Object)frame.getHeader("content-type"));
        StompV12Test.unsubscribe(newConn, "a-sub");
        newConn.disconnect();
    }

    @Test
    public void testHeaderContentLength() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String body = "Hello World 1!";
        String cLen = String.valueOf(body.getBytes(StandardCharsets.UTF_8).length);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).addHeader("content-type", "application/xml").addHeader("content-length", cLen).setBody(body + "extra");
        this.conn.sendFrame(frame);
        StompClientConnection newConn = StompClientConnectionFactory.createClientConnection(this.uri);
        newConn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(newConn, "a-sub");
        frame = newConn.receiveFrame();
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)body, (Object)frame.getBody());
        Assert.assertEquals((Object)cLen, (Object)frame.getHeader("content-length"));
        frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).addHeader("content-type", "application/xml").setBody(body + "extra");
        this.conn.sendFrame(frame);
        frame = newConn.receiveFrame();
        Assert.assertEquals((Object)(body + "extra"), (Object)frame.getBody());
        Assert.assertEquals((long)(body + "extra").getBytes(StandardCharsets.UTF_8).length, (long)Integer.valueOf(frame.getHeader("content-length")).intValue());
        StompV12Test.unsubscribe(newConn, "a-sub");
        newConn.disconnect();
    }

    @Test
    public void testHeaderRepetitive() throws Exception {
        AddressSettings addressSettings = new AddressSettings();
        addressSettings.setAutoCreateQueues(Boolean.valueOf(false));
        addressSettings.setAutoCreateAddresses(Boolean.valueOf(false));
        this.server.getAddressSettingsRepository().addMatch("#", (Object)addressSettings);
        this.conn.connect(this.defUser, this.defPass);
        String body = "Hello World!";
        String cLen = String.valueOf(body.getBytes(StandardCharsets.UTF_8).length);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).addHeader("destination", "aNonexistentQueue").addHeader("content-type", "application/xml").addHeader("content-length", cLen).addHeader("foo", "value1").addHeader("foo", "value2").addHeader("foo", "value3");
        frame.setBody(body);
        this.conn.sendFrame(frame);
        StompClientConnection newConn = StompClientConnectionFactory.createClientConnection(this.uri);
        newConn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(newConn, "a-sub", null, null, true);
        frame = newConn.receiveFrame();
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)body, (Object)frame.getBody());
        this.instanceLog.debug((Object)("received: " + frame));
        Assert.assertEquals((Object)"value1", (Object)frame.getHeader("foo"));
        StompV12Test.unsubscribe(newConn, "a-sub", true);
        newConn.disconnect();
        body = "Hello World!";
        cLen = String.valueOf(body.getBytes(StandardCharsets.UTF_8).length);
        frame = this.conn.createFrame("SEND").addHeader("destination", "aNonexistentQueue").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).addHeader("content-type", "application/xml").addHeader("content-length", cLen).addHeader("receipt", "1234").setBody(body);
        ClientStompFrame reply = this.conn.sendFrame(frame);
        Assert.assertEquals((Object)"ERROR", (Object)reply.getCommand());
    }

    @Test
    public void testHeadersPadding() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String body = "<p>Hello World!</p>";
        String cLen = String.valueOf(body.getBytes(StandardCharsets.UTF_8).length);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).addHeader("content-type", "application/xml").addHeader("content-length", cLen).addHeader(" header1", "value1 ").addHeader("  header2", "value2   ").addHeader("header3 ", " value3").addHeader(" header4 ", " value4 ").addHeader(" header 5  ", " value 5 ").addHeader("header6", "\t value\t 6 \t").setBody(body);
        this.conn.sendFrame(frame);
        StompClientConnection newConn = StompClientConnectionFactory.createClientConnection(this.uri);
        newConn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(newConn, "a-sub");
        frame = newConn.receiveFrame();
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)body, (Object)frame.getBody());
        this.instanceLog.debug((Object)("received: " + frame));
        Assert.assertEquals(null, (Object)frame.getHeader("header1"));
        Assert.assertEquals((Object)"value1 ", (Object)frame.getHeader(" header1"));
        Assert.assertEquals((Object)"value2   ", (Object)frame.getHeader("  header2"));
        Assert.assertEquals((Object)" value3", (Object)frame.getHeader("header3 "));
        Assert.assertEquals((Object)" value4 ", (Object)frame.getHeader(" header4 "));
        Assert.assertEquals((Object)" value 5 ", (Object)frame.getHeader(" header 5  "));
        Assert.assertEquals((Object)"\t value\t 6 \t", (Object)frame.getHeader("header6"));
        StompV12Test.unsubscribe(newConn, "a-sub");
        newConn.disconnect();
    }

    @Test
    public void testHeaderEncoding() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String body = "Hello World 1!";
        String cLen = String.valueOf(body.getBytes(StandardCharsets.UTF_8).length);
        String hKey = "\\rspecial-header\\\\\\n\\c\\r\\n";
        String hVal = "\\c\\\\\\ngood\\n\\r";
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).addHeader("content-type", "application/xml").addHeader("content-length", cLen).addHeader(hKey, hVal).setBody(body);
        this.instanceLog.debug((Object)("key: |" + hKey + "| val: |" + hVal + "|"));
        this.conn.sendFrame(frame);
        StompClientConnection newConn = StompClientConnectionFactory.createClientConnection(this.uri);
        newConn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(newConn, "a-sub");
        frame = newConn.receiveFrame();
        this.instanceLog.debug((Object)("received " + frame));
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        String value = frame.getHeader("\rspecial-header\\\n:\r\n");
        Assert.assertEquals((Object)":\\\ngood\n\r", (Object)value);
        StompV12Test.unsubscribe(newConn, "a-sub");
        newConn.disconnect();
    }

    @Test
    public void testHeaderUndefinedEscape() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND");
        String body = "Hello World 1!";
        String cLen = String.valueOf(body.getBytes(StandardCharsets.UTF_8).length);
        frame.addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName());
        frame.addHeader("content-type", "text/plain");
        frame.addHeader("content-length", cLen);
        String hKey = "undefined-escape";
        String hVal = "is\\ttab";
        frame.addHeader(hKey, hVal);
        this.instanceLog.debug((Object)("key: |" + hKey + "| val: |" + hVal + "|"));
        frame.setBody(body);
        this.conn.sendFrame(frame);
        ClientStompFrame error = this.conn.receiveFrame();
        this.instanceLog.debug((Object)("received " + error));
        String desc = "Should have received an ERROR for undefined escape sequence";
        Assert.assertNotNull((String)desc, (Object)error);
        Assert.assertEquals((String)desc, (Object)"ERROR", (Object)error.getCommand());
        this.waitDisconnect(this.conn);
        Assert.assertFalse((String)"Should be disconnected in STOMP 1.2 after ERROR", (boolean)this.conn.isConnected());
    }

    @Test
    public void testHeartBeat() throws Exception {
        StompClientConnection conn = StompClientConnectionFactory.createClientConnection(this.uri);
        ClientStompFrame frame = conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass);
        ClientStompFrame reply = conn.sendFrame(frame);
        Assert.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Thread.sleep(5000L);
        Assert.assertEquals((long)0L, (long)conn.getFrameQueueSize());
        conn.disconnect();
        conn = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "0,0").addHeader("accept-version", "1.0,1.1,1.2");
        reply = conn.sendFrame(frame);
        Assert.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assert.assertEquals((Object)"0,30000", (Object)reply.getHeader("heart-beat"));
        Thread.sleep(5000L);
        Assert.assertEquals((long)0L, (long)conn.getFrameQueueSize());
        conn.disconnect();
        conn = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "1,0").addHeader("accept-version", "1.0,1.2");
        reply = conn.sendFrame(frame);
        Assert.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assert.assertEquals((Object)"0,500", (Object)reply.getHeader("heart-beat"));
        Thread.sleep(2000L);
        try {
            StompV12Test.send(conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), "text/plain", "Hello World");
            Assert.fail((String)"connection should have been destroyed by now");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        conn = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "1,0").addHeader("accept-version", "1.0,1.1,1.2");
        reply = conn.sendFrame(frame);
        Assert.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assert.assertEquals((Object)"0,500", (Object)reply.getHeader("heart-beat"));
        conn.startPinger(500L);
        Thread.sleep(2000L);
        StompV12Test.send(conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), "text/plain", "Hello World");
        conn.stopPinger();
        conn.disconnect();
    }

    @Test
    public void testHeartBeat2() throws Exception {
        ClientStompFrame frame = this.conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "1,1").addHeader("accept-version", "1.0,1.1,1.2");
        ClientStompFrame reply = this.conn.sendFrame(frame);
        Assert.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assert.assertEquals((Object)"500,500", (Object)reply.getHeader("heart-beat"));
        this.conn.disconnect();
        this.conn = (StompClientConnectionV12)StompClientConnectionFactory.createClientConnection(this.uri);
        frame = this.conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "500,1000").addHeader("accept-version", "1.0,1.1,1.2");
        reply = this.conn.sendFrame(frame);
        Assert.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assert.assertEquals((Object)"1000,500", (Object)reply.getHeader("heart-beat"));
        this.conn.startPinger(500L);
        Thread.sleep(10000L);
        int size = this.conn.getServerPingNumber();
        this.instanceLog.debug((Object)("ping received: " + size));
        Assert.assertTrue((String)("size: " + size), (size > 5 ? (byte)1 : 0) != 0);
        StompV12Test.send(this.conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), "text/plain", "Hello World");
        this.conn.disconnect();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testSendWithHeartBeatsAndReceive() throws Exception {
        StompClientConnection newConn = null;
        try {
            ClientStompFrame frame = this.conn.createFrame("CONNECT");
            frame.addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "500,1000").addHeader("accept-version", "1.0,1.1,1.2");
            this.conn.sendFrame(frame);
            this.conn.startPinger(500L);
            for (int i = 0; i < 10; ++i) {
                StompV12Test.send(this.conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), "text/plain", "Hello World " + i + "!");
                Thread.sleep(500L);
            }
            newConn = StompClientConnectionFactory.createClientConnection(this.uri);
            newConn.connect(this.defUser, this.defPass);
            StompV12Test.subscribe(newConn, "a-sub");
            int cnt = 0;
            frame = newConn.receiveFrame();
            while (frame != null) {
                ++cnt;
                Thread.sleep(500L);
                frame = newConn.receiveFrame(5000L);
            }
            Assert.assertEquals((long)10L, (long)cnt);
            StompV12Test.unsubscribe(newConn, "a-sub");
        }
        finally {
            if (newConn != null) {
                newConn.disconnect();
            }
            this.conn.disconnect();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testSendAndReceiveWithHeartBeats() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        for (int i = 0; i < 10; ++i) {
            StompV12Test.send(this.conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), "text/plain", "Hello World " + i + "!");
            Thread.sleep(500L);
        }
        StompClientConnection newConn = StompClientConnectionFactory.createClientConnection(this.v11Uri);
        try {
            ClientStompFrame frame = newConn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "500,1000").addHeader("accept-version", "1.0,1.1");
            newConn.sendFrame(frame);
            newConn.startPinger(500L);
            Thread.sleep(500L);
            StompV12Test.subscribe(newConn, "a-sub");
            int cnt = 0;
            frame = newConn.receiveFrame();
            while (frame != null) {
                ++cnt;
                Thread.sleep(500L);
                frame = newConn.receiveFrame(5000L);
            }
            Assert.assertEquals((long)10L, (long)cnt);
            StompV12Test.unsubscribe(newConn, "a-sub");
        }
        finally {
            newConn.disconnect();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testSendWithHeartBeatsAndReceiveWithHeartBeats() throws Exception {
        StompClientConnection newConn = null;
        try {
            ClientStompFrame frame = this.conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "500,1000").addHeader("accept-version", "1.0,1.1,1.2");
            this.conn.sendFrame(frame);
            this.conn.startPinger(500L);
            for (int i = 0; i < 10; ++i) {
                StompV12Test.send(this.conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), "text/plain", "Hello World " + i + "!");
                Thread.sleep(500L);
            }
            newConn = StompClientConnectionFactory.createClientConnection(this.uri);
            frame = newConn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "500,1000").addHeader("accept-version", "1.0,1.1,1.2");
            newConn.sendFrame(frame);
            newConn.startPinger(500L);
            Thread.sleep(500L);
            StompV12Test.subscribe(newConn, "a-sub");
            int cnt = 0;
            frame = newConn.receiveFrame();
            while (frame != null) {
                ++cnt;
                Thread.sleep(500L);
                frame = newConn.receiveFrame(5000L);
            }
            Assert.assertEquals((long)10L, (long)cnt);
            StompV12Test.unsubscribe(newConn, "a-sub");
        }
        finally {
            if (newConn != null) {
                newConn.disconnect();
            }
            this.conn.disconnect();
        }
    }

    @Test
    public void testNack() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        String messageID = frame.getHeader("message-id");
        this.nack(this.conn, messageID);
        StompV12Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receiveNoWait();
        Assert.assertNull((Object)message);
    }

    @Test
    public void testNackWithWrongSubId() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        String messageID = frame.getHeader("ack");
        this.nack(this.conn, messageID + "0");
        ClientStompFrame error = this.conn.receiveFrame();
        Assert.assertEquals((Object)"ERROR", (Object)error.getCommand());
        this.waitDisconnect(this.conn);
        Assert.assertFalse((String)"Should be disconnected in STOMP 1.2 after ERROR", (boolean)this.conn.isConnected());
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
    }

    @Test
    public void testNackWithWrongMessageId() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        Assert.assertNotNull((Object)frame);
        Assert.assertNotNull((Object)frame.getHeader("ack"));
        this.nack(this.conn, "someother");
        ClientStompFrame error = this.conn.receiveFrame();
        this.instanceLog.debug((Object)("Receiver error: " + error));
        this.waitDisconnect(this.conn);
        Assert.assertFalse((String)"Should be disconnected in STOMP 1.2 after ERROR", (boolean)this.conn.isConnected());
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
    }

    @Test
    public void testAck() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        String messageID = frame.getHeader("ack");
        Assert.assertNotNull((Object)messageID);
        this.ack((StompClientConnection)this.conn, messageID);
        StompV12Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receiveNoWait();
        Assert.assertNull((Object)message);
    }

    @Test
    public void testAckNoIDHeader() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "client-individual");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        String messageID = frame.getHeader("ack");
        Assert.assertNotNull((Object)messageID);
        ClientStompFrame ackFrame = this.conn.createFrame("ACK");
        this.conn.sendFrame(ackFrame);
        frame = this.conn.receiveFrame();
        Assert.assertEquals((Object)"ERROR", (Object)frame.getCommand());
        this.waitDisconnect(this.conn);
        Assert.assertFalse((String)"Should be disconnected in STOMP 1.2 after ERROR", (boolean)this.conn.isConnected());
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
    }

    @Test
    public void testAckWithWrongMessageId() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        Assert.assertNotNull((Object)frame);
        this.ack((StompClientConnection)this.conn, "someother");
        ClientStompFrame error = this.conn.receiveFrame();
        this.instanceLog.debug((Object)("Receiver error: " + error));
        this.waitDisconnect(this.conn);
        Assert.assertFalse((String)"Should be disconnected in STOMP 1.2 after ERROR", (boolean)this.conn.isConnected());
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
    }

    @Test
    public void testErrorWithReceipt() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        String messageID = frame.getHeader("message-id");
        ClientStompFrame ackFrame = this.conn.createFrame("ACK");
        ackFrame.addHeader("subscription", "sub2");
        ackFrame.addHeader("message-id", messageID);
        ackFrame.addHeader("receipt", "answer-me");
        ClientStompFrame error = this.conn.sendFrame(ackFrame);
        this.instanceLog.debug((Object)("Receiver error: " + error));
        Assert.assertEquals((Object)"ERROR", (Object)error.getCommand());
        Assert.assertEquals((Object)"answer-me", (Object)error.getHeader("receipt-id"));
        this.waitDisconnect(this.conn);
        Assert.assertFalse((String)"Should be disconnected in STOMP 1.2 after ERROR", (boolean)this.conn.isConnected());
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
    }

    @Test
    public void testErrorWithReceipt2() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        String messageID = frame.getHeader("message-id");
        ClientStompFrame ackFrame = this.conn.createFrame("ACK");
        ackFrame.addHeader("subscription", "sub1");
        ackFrame.addHeader("message-id", String.valueOf(Long.valueOf(messageID) + 1L));
        ackFrame.addHeader("receipt", "answer-me");
        ClientStompFrame error = this.conn.sendFrame(ackFrame);
        this.instanceLog.debug((Object)("Receiver error: " + error));
        Assert.assertEquals((Object)"ERROR", (Object)error.getCommand());
        Assert.assertEquals((Object)"answer-me", (Object)error.getHeader("receipt-id"));
        this.waitDisconnect(this.conn);
        Assert.assertFalse((String)"Should be disconnected in STOMP 1.2 after ERROR", (boolean)this.conn.isConnected());
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
    }

    protected void waitDisconnect(StompClientConnectionV12 connection) throws Exception {
        long timeout = System.currentTimeMillis() + 10000L;
        while (timeout > System.currentTimeMillis() && connection.isConnected()) {
            Thread.sleep(10L);
        }
    }

    @Test
    public void testAckModeClient() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "client");
        int num = 50;
        for (int i = 0; i < num; ++i) {
            this.sendJmsMessage("client-ack" + i);
        }
        ClientStompFrame frame = null;
        for (int i = 0; i < num; ++i) {
            frame = this.conn.receiveFrame();
            Assert.assertNotNull((Object)frame);
        }
        this.ack((StompClientConnection)this.conn, frame);
        StompV12Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receiveNoWait();
        Assert.assertNull((Object)message);
    }

    @Test
    public void testAckModeClient2() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "client");
        int num = 50;
        for (int i = 0; i < num; ++i) {
            this.sendJmsMessage("client-ack" + i);
        }
        ClientStompFrame frame = null;
        for (int i = 0; i < num; ++i) {
            frame = this.conn.receiveFrame();
            Assert.assertNotNull((Object)frame);
            if (i != num - 2) continue;
            this.ack((StompClientConnection)this.conn, frame);
        }
        StompV12Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        message = consumer.receiveNoWait();
        Assert.assertNull((Object)message);
    }

    @Test
    public void testAckModeDefault() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", null);
        int num = 50;
        for (int i = 0; i < num; ++i) {
            this.sendJmsMessage("auto-ack" + i);
        }
        ClientStompFrame frame = null;
        for (int i = 0; i < num; ++i) {
            frame = this.conn.receiveFrame();
            Assert.assertNotNull((Object)frame);
        }
        StompV12Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receiveNoWait();
        Assert.assertNull((Object)message);
    }

    @Test
    public void testAckModeAuto() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "auto");
        int num = 50;
        for (int i = 0; i < num; ++i) {
            this.sendJmsMessage("auto-ack" + i);
        }
        ClientStompFrame frame = null;
        for (int i = 0; i < num; ++i) {
            frame = this.conn.receiveFrame();
            Assert.assertNotNull((Object)frame);
        }
        StompV12Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receiveNoWait();
        Assert.assertNull((Object)message);
    }

    @Test
    public void testAckModeClientIndividual() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "client-individual");
        int num = 50;
        for (int i = 0; i < num; ++i) {
            this.sendJmsMessage("client-individual-ack" + i);
        }
        ClientStompFrame frame = null;
        for (int i = 0; i < num; ++i) {
            frame = this.conn.receiveFrame();
            Assert.assertNotNull((Object)frame);
            this.instanceLog.debug((Object)(i + " == received: " + frame));
            if (i % 2 != 0) continue;
            this.ack((StompClientConnection)this.conn, frame);
        }
        StompV12Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        TextMessage message = null;
        for (int i = 0; i < num / 2; ++i) {
            message = (TextMessage)consumer.receive(1000L);
            Assert.assertNotNull((Object)message);
            this.instanceLog.debug((Object)("Legal: " + message.getText()));
        }
        message = (TextMessage)consumer.receiveNoWait();
        Assert.assertNull((Object)message);
    }

    @Test
    public void testTwoSubscribers() throws Exception {
        this.conn.connect(this.defUser, this.defPass, CLIENT_ID);
        StompV12Test.subscribeTopic(this.conn, "sub1", "auto", null);
        StompClientConnection newConn = StompClientConnectionFactory.createClientConnection(this.v11Uri);
        newConn.connect(this.defUser, this.defPass, "myclientid2");
        StompV12Test.subscribeTopic(newConn, "sub2", "auto", null);
        StompV12Test.send(this.conn, StompV12Test.getTopicPrefix() + StompV12Test.getTopicName(), null, "Hello World");
        ClientStompFrame frame = this.conn.receiveFrame(1000L);
        this.instanceLog.debug((Object)("received frame : " + frame));
        Assert.assertEquals((Object)"Hello World", (Object)frame.getBody());
        Assert.assertEquals((Object)"sub1", (Object)frame.getHeader("subscription"));
        frame = newConn.receiveFrame(1000L);
        this.instanceLog.debug((Object)("received 2 frame : " + frame));
        Assert.assertEquals((Object)"Hello World", (Object)frame.getBody());
        Assert.assertEquals((Object)"sub2", (Object)frame.getHeader("subscription"));
        StompV12Test.unsubscribe(this.conn, "sub1", true);
        StompV12Test.unsubscribe(newConn, "sub2", true);
        this.conn.disconnect();
        newConn.disconnect();
    }

    @Test
    public void testSendAndReceiveOnDifferentConnections() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.send(this.conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), null, "Hello World");
        StompClientConnection connV12_2 = StompClientConnectionFactory.createClientConnection(this.v11Uri);
        connV12_2.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(connV12_2, "sub1", "auto");
        ClientStompFrame frame = connV12_2.receiveFrame(2000L);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)"Hello World", (Object)frame.getBody());
        this.conn.disconnect();
        connV12_2.disconnect();
    }

    @Test
    public void testBeginSameTransactionTwice() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.beginTransaction(this.conn, "tx1");
        StompV12Test.beginTransaction(this.conn, "tx1");
        ClientStompFrame f = this.conn.receiveFrame();
        Assert.assertTrue((boolean)f.getCommand().equals("ERROR"));
    }

    @Test
    public void testBodyWithUTF8() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, this.getName(), "auto");
        String text = "A\u00ea\u00f1\u00fcC";
        this.instanceLog.debug((Object)text);
        this.sendJmsMessage(text);
        ClientStompFrame frame = this.conn.receiveFrame();
        this.instanceLog.debug((Object)frame);
        Assert.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        Assert.assertNotNull((Object)frame.getHeader("destination"));
        Assert.assertTrue((boolean)frame.getBody().equals(text));
        this.conn.disconnect();
    }

    @Test
    public void testClientAckNotPartOfTransaction() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, this.getName(), "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        Assert.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        Assert.assertNotNull((Object)frame.getHeader("destination"));
        Assert.assertTrue((boolean)frame.getBody().equals(this.getName()));
        Assert.assertNotNull((Object)frame.getHeader("message-id"));
        Assert.assertNotNull((Object)frame.getHeader("ack"));
        String messageID = frame.getHeader("ack");
        StompV12Test.beginTransaction(this.conn, "tx1");
        this.ack(this.conn, messageID, "tx1");
        StompV12Test.abortTransaction(this.conn, "tx1");
        frame = this.conn.receiveFrame(100L);
        Assert.assertNull((Object)frame);
        StompV12Test.unsubscribe(this.conn, this.getName());
        this.conn.disconnect();
    }

    @Test
    public void testDisconnectAndError() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, this.getName(), "client");
        ClientStompFrame frame = this.conn.createFrame("DISCONNECT");
        frame.addHeader("receipt", "1");
        ClientStompFrame result = this.conn.sendFrame(frame);
        if (result == null || !"RECEIPT".equals(result.getCommand()) || !"1".equals(result.getHeader("receipt-id"))) {
            Assert.fail((String)("Disconnect failed! " + result));
        }
        final CountDownLatch latch = new CountDownLatch(1);
        Thread thr = new Thread(){

            /*
             * Enabled aggressive block sorting
             * Enabled unnecessary exception pruning
             * Enabled aggressive exception aggregation
             */
            @Override
            public void run() {
                while (latch.getCount() != 0L) {
                    try {
                        StompTestBase.send(StompV12Test.this.conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), null, "Hello World");
                        Thread.sleep(500L);
                    }
                    catch (InterruptedException interruptedException) {
                    }
                    catch (ClosedChannelException e) {
                        latch.countDown();
                        return;
                    }
                    catch (IOException e) {
                        latch.countDown();
                        return;
                    }
                    finally {
                        StompV12Test.this.conn.destroy();
                    }
                }
            }
        };
        thr.start();
        latch.await(10L, TimeUnit.SECONDS);
        long count = latch.getCount();
        if (count != 0L) {
            latch.countDown();
        }
        thr.join();
        Assert.assertTrue((String)"Server failed to disconnect.", (count == 0L ? (byte)1 : 0) != 0);
    }

    @Test
    public void testDurableSubscriber() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "client", this.getName());
        ClientStompFrame frame = StompV12Test.subscribe(this.conn, "sub1", "client", this.getName());
        Assert.assertTrue((boolean)frame.getCommand().equals("ERROR"));
        this.waitDisconnect(this.conn);
        Assert.assertFalse((String)"Should be disconnected in STOMP 1.2 after ERROR", (boolean)this.conn.isConnected());
    }

    @Test
    public void testMultipleDurableSubscribers() throws Exception {
        this.conn.connect(this.defUser, this.defPass, "myClientID");
        StompClientConnectionV12 conn2 = (StompClientConnectionV12)StompClientConnectionFactory.createClientConnection(this.uri);
        conn2.connect(this.defUser, this.defPass, "myClientID");
        StompV12Test.subscribe(this.conn, UUID.randomUUID().toString(), "client-individual", this.getName());
        StompV12Test.subscribe(conn2, UUID.randomUUID().toString(), "clientindividual", this.getName());
        this.conn.closeTransport();
        this.waitDisconnect(this.conn);
        conn2.closeTransport();
        this.waitDisconnect(conn2);
    }

    @Test
    public void testMultipleConcurrentDurableSubscribers() throws Exception {
        int NUMBER_OF_THREADS = 25;
        SubscriberThread[] threads = new SubscriberThread[NUMBER_OF_THREADS];
        CountDownLatch startFlag = new CountDownLatch(1);
        CountDownLatch alignFlag = new CountDownLatch(NUMBER_OF_THREADS);
        for (int i = 0; i < threads.length; ++i) {
            threads[i] = new SubscriberThread("subscriber::" + i, StompClientConnectionFactory.createClientConnection(this.uri), startFlag, alignFlag);
        }
        for (SubscriberThread t : threads) {
            t.start();
        }
        alignFlag.await();
        startFlag.countDown();
        for (SubscriberThread t : threads) {
            t.join();
            Assert.assertEquals((long)0L, (long)t.errors.get());
        }
    }

    @Test
    public void testDurableSubscriberWithReconnection() throws Exception {
        this.conn.connect(this.defUser, this.defPass, CLIENT_ID);
        StompV12Test.subscribeTopic(this.conn, "sub1", "auto", this.getName());
        ClientStompFrame frame = this.conn.createFrame("DISCONNECT");
        frame.addHeader("receipt", "1");
        ClientStompFrame result = this.conn.sendFrame(frame);
        if (result == null || !"RECEIPT".equals(result.getCommand()) || !"1".equals(result.getHeader("receipt-id"))) {
            Assert.fail((String)("Disconnect failed! " + result));
        }
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        this.conn.destroy();
        this.conn = (StompClientConnectionV12)StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass, CLIENT_ID);
        StompV12Test.subscribeTopic(this.conn, "sub1", "auto", this.getName());
        frame = this.conn.receiveFrame();
        Assert.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        Assert.assertNotNull((Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)this.getName(), (Object)frame.getBody());
        StompV12Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
    }

    @Test
    public void testDurableUnSubscribe() throws Exception {
        this.conn.connect(this.defUser, this.defPass, CLIENT_ID);
        StompV12Test.subscribeTopic(this.conn, null, "auto", this.getName());
        this.conn.disconnect();
        this.conn.destroy();
        this.conn = (StompClientConnectionV12)StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass, CLIENT_ID);
        StompV12Test.unsubscribe(this.conn, this.getName(), null, false, true);
        SimpleString queueName = SimpleString.toSimpleString((String)("myclientid." + this.getName()));
        Wait.assertTrue(() -> this.server.locateQueue(queueName) == null);
        this.conn.disconnect();
    }

    @Test
    public void testJMSXGroupIdCanBeSet() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND");
        frame.addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName());
        frame.addHeader("JMSXGroupID", "TEST");
        frame.setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((Object)"TEST", (Object)message.getStringProperty("JMSXGroupID"));
    }

    @Test
    public void testMessagesAreInOrder() throws Exception {
        int i;
        int ctr = 10;
        String[] data = new String[ctr];
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "auto");
        for (int i2 = 0; i2 < ctr; ++i2) {
            data[i2] = this.getName() + i2;
            this.sendJmsMessage(data[i2]);
        }
        ClientStompFrame frame = null;
        for (i = 0; i < ctr; ++i) {
            frame = this.conn.receiveFrame();
            Assert.assertTrue((String)"Message not in order", (boolean)frame.getBody().equals(data[i]));
        }
        for (i = 0; i < ctr; ++i) {
            data[i] = this.getName() + ":second:" + i;
            this.sendJmsMessage(data[i]);
        }
        for (i = 0; i < ctr; ++i) {
            frame = this.conn.receiveFrame();
            Assert.assertTrue((String)"Message not in order", (boolean)frame.getBody().equals(data[i]));
        }
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeWithAutoAckAndSelector() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe((StompClientConnection)this.conn, "sub1", "auto", null, "foo = 'zzz'");
        this.sendJmsMessage("Ignored message", "foo", "1234");
        this.sendJmsMessage("Real message", "foo", "zzz");
        ClientStompFrame frame = this.conn.receiveFrame();
        Assert.assertTrue((String)("Should have received the real message but got: " + frame), (boolean)frame.getBody().equals("Real message"));
        this.conn.disconnect();
    }

    @Test
    public void testRedeliveryWithClientAck() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "subId", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        Assert.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertTrue((boolean)message.getJMSRedelivered());
    }

    @Test
    public void testSendManyMessages() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        int count = 1000;
        final CountDownLatch latch = new CountDownLatch(count);
        consumer.setMessageListener(new MessageListener(){

            public void onMessage(Message arg0) {
                TextMessage m = (TextMessage)arg0;
                latch.countDown();
                try {
                    StompV12Test.this.instanceLog.debug((Object)("___> latch now: " + latch.getCount() + " m: " + m.getText()));
                }
                catch (JMSException e) {
                    Assert.fail((String)"here failed");
                    e.printStackTrace();
                }
            }
        });
        for (int i = 1; i <= count; ++i) {
            StompV12Test.send(this.conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), null, "Hello World");
        }
        Assert.assertTrue((boolean)latch.await(60L, TimeUnit.SECONDS));
        this.conn.disconnect();
    }

    @Test
    public void testSendMessage() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.send(this.conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), null, "Hello World");
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((String)"getJMSPriority", (long)4L, (long)message.getJMSPriority());
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assert.assertTrue((Math.abs(tnow - tmsg) < 1000L ? (byte)1 : 0) != 0);
    }

    @Test
    public void testSendMessageWithContentLength() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        byte[] data = new byte[]{1, 0, 0, 4};
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).setBody(new String(data, StandardCharsets.UTF_8)).addHeader("content-length", String.valueOf(data.length));
        this.conn.sendFrame(frame);
        BytesMessage message = (BytesMessage)consumer.receive(10000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((long)data.length, (long)message.getBodyLength());
        Assert.assertEquals((long)data[0], (long)message.readByte());
        Assert.assertEquals((long)data[1], (long)message.readByte());
        Assert.assertEquals((long)data[2], (long)message.readByte());
        Assert.assertEquals((long)data[3], (long)message.readByte());
    }

    @Test
    public void testSendMessageWithCustomHeadersAndSelector() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue, "foo = 'abc'");
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).addHeader("foo", "abc").addHeader("bar", "123").setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((String)"foo", (Object)"abc", (Object)message.getStringProperty("foo"));
        Assert.assertEquals((String)"bar", (Object)"123", (Object)message.getStringProperty("bar"));
    }

    @Test
    public void testSendMessageWithLeadingNewLine() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).setBody("Hello World");
        this.conn.sendWickedFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assert.assertTrue((Math.abs(tnow - tmsg) < 1000L ? (byte)1 : 0) != 0);
        Assert.assertNull((Object)consumer.receiveNoWait());
        this.conn.disconnect();
    }

    @Test
    public void testSendMessageWithReceipt() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.send(this.conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), null, "Hello World", true);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assert.assertTrue((Math.abs(tnow - tmsg) < 1000L ? (byte)1 : 0) != 0);
        this.conn.disconnect();
    }

    @Test
    public void testSendMessageWithStandardHeaders() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).addHeader("correlation-id", "c123").addHeader("persistent", "true").addHeader("priority", "3").addHeader("type", "t345").addHeader("JMSXGroupID", "abc").addHeader("foo", "abc").addHeader("bar", "123").setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((String)"JMSCorrelationID", (Object)"c123", (Object)message.getJMSCorrelationID());
        Assert.assertEquals((String)"getJMSType", (Object)"t345", (Object)message.getJMSType());
        Assert.assertEquals((String)"getJMSPriority", (long)3L, (long)message.getJMSPriority());
        Assert.assertEquals((long)2L, (long)message.getJMSDeliveryMode());
        Assert.assertEquals((String)"foo", (Object)"abc", (Object)message.getStringProperty("foo"));
        Assert.assertEquals((String)"bar", (Object)"123", (Object)message.getStringProperty("bar"));
        Assert.assertEquals((String)"JMSXGroupID", (Object)"abc", (Object)message.getStringProperty("JMSXGroupID"));
        this.conn.disconnect();
    }

    @Test
    public void testSendMessageWithLongHeaders() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StringBuffer buffer = new StringBuffer();
        for (int i = 0; i < 2048; ++i) {
            buffer.append("a");
        }
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).addHeader("correlation-id", "c123").addHeader("persistent", "true").addHeader("priority", "3").addHeader("type", "t345").addHeader("JMSXGroupID", "abc").addHeader("foo", "abc").addHeader("very-very-long-stomp-message-header", buffer.toString()).setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"Hello World", (Object)message.getText());
        Assert.assertEquals((String)"JMSCorrelationID", (Object)"c123", (Object)message.getJMSCorrelationID());
        Assert.assertEquals((String)"getJMSType", (Object)"t345", (Object)message.getJMSType());
        Assert.assertEquals((String)"getJMSPriority", (long)3L, (long)message.getJMSPriority());
        Assert.assertEquals((long)2L, (long)message.getJMSDeliveryMode());
        Assert.assertEquals((String)"foo", (Object)"abc", (Object)message.getStringProperty("foo"));
        Assert.assertEquals((String)"very-very-long-stomp-message-header", (long)2048L, (long)message.getStringProperty("very-very-long-stomp-message-header").length());
        Assert.assertEquals((String)"JMSXGroupID", (Object)"abc", (Object)message.getStringProperty("JMSXGroupID"));
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeToTopic() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribeTopic(this.conn, "sub1", null, null, true);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        ClientStompFrame frame = this.conn.receiveFrame();
        Assert.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        Assert.assertTrue((boolean)frame.getHeader("destination").equals(StompV12Test.getTopicPrefix() + StompV12Test.getTopicName()));
        Assert.assertTrue((boolean)frame.getBody().equals(this.getName()));
        StompV12Test.unsubscribe(this.conn, "sub1", true);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        frame = this.conn.receiveFrame(100L);
        Assert.assertNull((Object)frame);
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeToTopicWithNoLocal() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribeTopic(this.conn, "sub1", null, null, true, true);
        StompV12Test.send(this.conn, StompV12Test.getTopicPrefix() + StompV12Test.getTopicName(), null, "Hello World");
        ClientStompFrame frame = this.conn.receiveFrame(100L);
        Assert.assertNull((Object)frame);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        frame = this.conn.receiveFrame();
        Assert.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        Assert.assertTrue((boolean)frame.getHeader("destination").equals(StompV12Test.getTopicPrefix() + StompV12Test.getTopicName()));
        Assert.assertTrue((boolean)frame.getBody().equals(this.getName()));
        StompV12Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeWithAutoAck() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "auto");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertNotNull((Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)this.getName(), (Object)frame.getBody());
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receiveNoWait();
        Assert.assertNull((Object)message);
    }

    @Test
    public void testSubscribeWithAutoAckAndBytesMessage() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "auto");
        byte[] payload = new byte[]{1, 2, 3, 4, 5};
        this.sendJmsMessage(payload, (Destination)this.queue);
        ClientStompFrame frame = this.conn.receiveFrame();
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        this.instanceLog.debug((Object)("Message: " + frame));
        Assert.assertEquals((Object)"5", (Object)frame.getHeader("content-length"));
        Assert.assertEquals(null, (Object)frame.getHeader("type"));
        Assert.assertEquals((Object)frame.getBody(), (Object)new String(payload, StandardCharsets.UTF_8));
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeWithClientAck() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        this.ack((StompClientConnection)this.conn, frame);
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receiveNoWait();
        Assert.assertNull((Object)message);
    }

    @Test
    public void testSubscribeWithClientAckThenConsumingAgainWithAutoAckWithExplicitDisconnect() throws Exception {
        this.assertSubscribeWithClientAckThenConsumeWithAutoAck(true);
    }

    @Test
    public void testSubscribeWithClientAckThenConsumingAgainWithAutoAckWithNoDisconnectFrame() throws Exception {
        this.assertSubscribeWithClientAckThenConsumeWithAutoAck(false);
    }

    @Test
    public void testSubscribeWithID() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "mysubid", "auto");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        Assert.assertTrue((frame.getHeader("subscription") != null ? (byte)1 : 0) != 0);
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeWithMessageSentWithProperties() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "auto");
        MessageProducer producer = this.session.createProducer((Destination)this.queue);
        BytesMessage message = this.session.createBytesMessage();
        message.setStringProperty("S", "value");
        message.setBooleanProperty("n", false);
        message.setByteProperty("byte", (byte)9);
        message.setDoubleProperty("d", 2.0);
        message.setFloatProperty("f", 6.0f);
        message.setIntProperty("i", 10);
        message.setLongProperty("l", 121L);
        message.setShortProperty("s", (short)12);
        message.writeBytes("Hello World".getBytes(StandardCharsets.UTF_8));
        producer.send((Message)message);
        ClientStompFrame frame = this.conn.receiveFrame();
        Assert.assertNotNull((Object)frame);
        Assert.assertTrue((frame.getHeader("S") != null ? (byte)1 : 0) != 0);
        Assert.assertTrue((frame.getHeader("n") != null ? (byte)1 : 0) != 0);
        Assert.assertTrue((frame.getHeader("byte") != null ? (byte)1 : 0) != 0);
        Assert.assertTrue((frame.getHeader("d") != null ? (byte)1 : 0) != 0);
        Assert.assertTrue((frame.getHeader("f") != null ? (byte)1 : 0) != 0);
        Assert.assertTrue((frame.getHeader("i") != null ? (byte)1 : 0) != 0);
        Assert.assertTrue((frame.getHeader("l") != null ? (byte)1 : 0) != 0);
        Assert.assertTrue((frame.getHeader("s") != null ? (byte)1 : 0) != 0);
        Assert.assertEquals((Object)"Hello World", (Object)frame.getBody());
        this.conn.disconnect();
    }

    @Test
    public void testSuccessiveTransactionsWithSameID() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.beginTransaction(this.conn, "tx1");
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).addHeader("transaction", "tx1").setBody("Hello World");
        this.conn.sendFrame(frame);
        StompV12Test.commitTransaction(this.conn, "tx1");
        Message message = consumer.receive(1000L);
        Assert.assertNotNull((String)"Should have received a message", (Object)message);
        StompV12Test.beginTransaction(this.conn, "tx1");
        frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).addHeader("transaction", "tx1").setBody("Hello World");
        this.conn.sendFrame(frame);
        StompV12Test.commitTransaction(this.conn, "tx1");
        message = consumer.receive(1000L);
        Assert.assertNotNull((String)"Should have received a message", (Object)message);
        this.conn.disconnect();
    }

    @Test
    public void testTransactionCommit() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.beginTransaction(this.conn, "tx1");
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).addHeader("transaction", "tx1").addHeader("receipt", "123").setBody("Hello World");
        frame = this.conn.sendFrame(frame);
        Assert.assertEquals((Object)"123", (Object)frame.getHeader("receipt-id"));
        Assert.assertNull((Object)consumer.receiveNoWait());
        StompV12Test.commitTransaction(this.conn, "tx1", true);
        Message message = consumer.receive(1000L);
        Assert.assertNotNull((String)"Should have received a message", (Object)message);
        this.conn.disconnect();
    }

    @Test
    public void testTransactionRollback() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.beginTransaction(this.conn, "tx1");
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).addHeader("transaction", "tx1").setBody("first message");
        this.conn.sendFrame(frame);
        StompV12Test.abortTransaction(this.conn, "tx1");
        StompV12Test.beginTransaction(this.conn, "tx1");
        frame = this.conn.createFrame("SEND").addHeader("destination", StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()).addHeader("transaction", "tx1").setBody("second message");
        this.conn.sendFrame(frame);
        StompV12Test.commitTransaction(this.conn, "tx1", true);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assert.assertNotNull((Object)message);
        Assert.assertEquals((Object)"second message", (Object)message.getText());
        this.conn.disconnect();
    }

    @Test
    public void testUnsubscribe() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "auto");
        this.sendJmsMessage("first message");
        ClientStompFrame frame = this.conn.receiveFrame();
        Assert.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        StompV12Test.unsubscribe(this.conn, "sub1", true);
        this.sendJmsMessage("second message");
        frame = this.conn.receiveFrame(100L);
        Assert.assertNull((Object)frame);
        this.conn.disconnect();
    }

    @Test
    public void testDisconnectWithoutUnsubscribe() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "auto");
        this.sendJmsMessage("first message");
        ClientStompFrame frame = this.conn.receiveFrame();
        Assert.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        this.conn.disconnect();
        this.sendJmsMessage("second message");
        this.conn = (StompClientConnectionV12)StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass);
        frame = this.conn.receiveFrame(100L);
        Assert.assertNull((String)("not expected: " + frame), (Object)frame);
        StompV12Test.subscribe(this.conn, "sub1", "auto");
        frame = this.conn.receiveFrame();
        Assert.assertNotNull((Object)frame);
        Assert.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        frame = this.conn.receiveFrame(100L);
        Assert.assertNull((String)("not expected: " + frame), (Object)frame);
        StompV12Test.unsubscribe(this.conn, "sub1", true);
        frame = this.conn.receiveFrame(100L);
        Assert.assertNull((Object)frame);
        this.conn.disconnect();
    }

    protected void assertSubscribeWithClientAckThenConsumeWithAutoAck(boolean sendDisconnect) throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        log.debug((Object)"Reconnecting!");
        if (sendDisconnect) {
            this.conn.disconnect();
            this.conn = (StompClientConnectionV12)StompClientConnectionFactory.createClientConnection(this.uri);
        } else {
            this.conn.destroy();
            this.conn = (StompClientConnectionV12)StompClientConnectionFactory.createClientConnection(this.uri);
        }
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, "sub1", null);
        frame = this.conn.receiveFrame();
        Assert.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        this.conn.disconnect();
        this.conn.destroy();
        this.conn = (StompClientConnectionV12)StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe((StompClientConnection)this.conn, "sub1", null, null, true);
        this.sendJmsMessage("shouldBeNextMessage");
        frame = this.conn.receiveFrame();
        Assert.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        Assert.assertEquals((Object)"shouldBeNextMessage", (Object)frame.getBody());
    }

    @Test
    public void testSendMessageToNonExistentQueueWithAutoCreation() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.send(this.conn, "NonExistentQueue" + UUID.randomUUID().toString(), null, "Hello World", true, RoutingType.ANYCAST);
        this.conn.disconnect();
    }

    @Test
    public void testInvalidStompCommand() throws Exception {
        try {
            this.conn.connect(this.defUser, this.defPass);
            ClientStompFrame frame = this.conn.createAnyFrame("INVALID");
            frame.setBody("Hello World");
            frame.addHeader("receipt", "1234");
            frame = this.conn.sendFrame(frame);
            Assert.assertTrue((boolean)frame.getCommand().equals("ERROR"));
        }
        finally {
            this.conn.destroy();
        }
    }

    @Test
    public void testSendAndReceiveWithEscapedCharactersInSenderId() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame response = StompV12Test.send(this.conn, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), "text/plain", "Hello World 1!");
        Assert.assertNull((Object)response);
        StompV12Test.subscribe(this.conn, "ID\\cMYMACHINE-50616-635482262727823605-1\\c1\\c1\\c1");
        ClientStompFrame frame = this.conn.receiveFrame();
        this.instanceLog.debug((Object)("Received: " + frame));
        Assert.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assert.assertEquals((Object)"ID:MYMACHINE-50616-635482262727823605-1:1:1:1", (Object)frame.getHeader("subscription"));
        Assert.assertNotNull((Object)frame.getHeader("message-id"));
        Assert.assertEquals((Object)(StompV12Test.getQueuePrefix() + StompV12Test.getQueueName()), (Object)frame.getHeader("destination"));
        Assert.assertEquals((Object)"Hello World 1!", (Object)frame.getBody());
        StompV12Test.unsubscribe(this.conn, "ID\\cMYMACHINE-50616-635482262727823605-1\\c1\\c1\\c1");
        this.conn.disconnect();
    }

    @Test
    public void testSubscribeWithZeroConsumerWindowSize() throws Exception {
        this.internalSubscribeWithZeroConsumerWindowSize("consumer-window-size", true);
    }

    @Test
    public void testSubscribeWithZeroConsumerWindowSizeLegacyHeader() throws Exception {
        this.internalSubscribeWithZeroConsumerWindowSize("activemq.prefetchSize", true);
    }

    @Test
    public void testSubscribeWithZeroConsumerWindowSizeAndNack() throws Exception {
        this.internalSubscribeWithZeroConsumerWindowSize("consumer-window-size", false);
    }

    @Test
    public void testSubscribeWithZeroConsumerWindowSizeLegacyHeaderAndNack() throws Exception {
        this.internalSubscribeWithZeroConsumerWindowSize("activemq.prefetchSize", false);
    }

    private void internalSubscribeWithZeroConsumerWindowSize(String consumerWindowSizeHeader, boolean ack) throws Exception {
        int TIMEOUT = 1000;
        int NEGATIVE_TIMEOUT = 100;
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, null, "client-individual", null, null, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), true, 0, consumerWindowSizeHeader);
        this.sendJmsMessage(this.getName());
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame1 = this.conn.receiveFrame(1000L);
        Assert.assertNotNull((Object)frame1);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame1.getCommand());
        String messageID = frame1.getHeader("message-id");
        ClientStompFrame frame2 = this.conn.receiveFrame(100L);
        Assert.assertNull((Object)frame2);
        if (ack) {
            this.ack((StompClientConnection)this.conn, messageID);
        } else {
            this.nack(this.conn, messageID);
        }
        ClientStompFrame frame3 = this.conn.receiveFrame(1000L);
        Assert.assertNotNull((Object)frame3);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame3.getCommand());
        messageID = frame3.getHeader("message-id");
        if (ack) {
            this.ack((StompClientConnection)this.conn, messageID);
        } else {
            this.nack(this.conn, messageID);
        }
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(100L);
        Assert.assertNull((Object)message);
    }

    @Test
    public void testSubscribeWithNonZeroConsumerWindowSize() throws Exception {
        this.internalSubscribeWithNonZeroConsumerWindowSize("consumer-window-size", true);
    }

    @Test
    public void testSubscribeWithNonZeroConsumerWindowSizeLegacyHeader() throws Exception {
        this.internalSubscribeWithNonZeroConsumerWindowSize("activemq.prefetchSize", true);
    }

    @Test
    public void testSubscribeWithNonZeroConsumerWindowSizeAndNack() throws Exception {
        this.internalSubscribeWithNonZeroConsumerWindowSize("consumer-window-size", false);
    }

    @Test
    public void testSubscribeWithNonZeroConsumerWindowSizeLegacyHeaderAndNack() throws Exception {
        this.internalSubscribeWithNonZeroConsumerWindowSize("activemq.prefetchSize", false);
    }

    private void internalSubscribeWithNonZeroConsumerWindowSize(String consumerWindowSizeHeader, boolean ack) throws Exception {
        int MESSAGE_SIZE = 270;
        int TIMEOUT = 1000;
        String MESSAGE = "foo-foo-foo";
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, null, "client-individual", null, null, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), true, 540, consumerWindowSizeHeader);
        this.sendJmsMessage("foo-foo-foo");
        this.sendJmsMessage("foo-foo-foo");
        this.sendJmsMessage("foo-foo-foo");
        ClientStompFrame frame1 = this.conn.receiveFrame(1000L);
        Assert.assertNotNull((Object)frame1);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame1.getCommand());
        String messageID1 = frame1.getHeader("message-id");
        ClientStompFrame frame2 = this.conn.receiveFrame(1000L);
        Assert.assertNotNull((Object)frame2);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame2.getCommand());
        String messageID2 = frame2.getHeader("message-id");
        ClientStompFrame frame3 = this.conn.receiveFrame(100L);
        Assert.assertNull((Object)frame3);
        if (ack) {
            this.ack((StompClientConnection)this.conn, messageID1);
            this.ack((StompClientConnection)this.conn, messageID2);
        } else {
            this.nack(this.conn, messageID1);
            this.nack(this.conn, messageID2);
        }
        ClientStompFrame frame4 = this.conn.receiveFrame(1000L);
        Assert.assertNotNull((Object)frame4);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame4.getCommand());
        String messageID4 = frame4.getHeader("message-id");
        if (ack) {
            this.ack((StompClientConnection)this.conn, messageID4);
        } else {
            this.nack(this.conn, messageID4);
        }
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receiveNoWait();
        Assert.assertNull((Object)message);
    }

    @Test
    public void testSubscribeWithNonZeroConsumerWindowSizeAndClientAck() throws Exception {
        Logger.getLogger((String)StompConnection.class.getName()).setLevel((Level)org.jboss.logmanager.Level.DEBUG);
        int MESSAGE_SIZE = 270;
        int TIMEOUT = 1000;
        String MESSAGE = "foo-foo-foo";
        this.conn.connect(this.defUser, this.defPass);
        StompV12Test.subscribe(this.conn, null, "client", null, null, StompV12Test.getQueuePrefix() + StompV12Test.getQueueName(), true, 540, "consumer-window-size");
        this.sendJmsMessage("foo-foo-foo");
        this.sendJmsMessage("foo-foo-foo");
        this.sendJmsMessage("foo-foo-foo");
        this.sendJmsMessage("foo-foo-foo");
        ClientStompFrame frame1 = this.conn.receiveFrame(1000L);
        Assert.assertNotNull((Object)frame1);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame1.getCommand());
        ClientStompFrame frame2 = this.conn.receiveFrame(1000L);
        Assert.assertNotNull((Object)frame2);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame2.getCommand());
        String messageID2 = frame2.getHeader("message-id");
        ClientStompFrame frame3 = this.conn.receiveFrame(100L);
        Assert.assertNull((Object)frame3);
        this.ack((StompClientConnection)this.conn, messageID2);
        ClientStompFrame frame4 = this.conn.receiveFrame(1000L);
        Assert.assertNotNull((Object)frame4);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame4.getCommand());
        ClientStompFrame frame5 = this.conn.receiveFrame(1000L);
        Assert.assertNotNull((Object)frame5);
        Assert.assertEquals((Object)"MESSAGE", (Object)frame5.getCommand());
        String messageID5 = frame5.getHeader("message-id");
        this.ack((StompClientConnection)this.conn, messageID5);
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receiveNoWait();
        Assert.assertNull((Object)message);
    }

    private void ack(StompClientConnection conn, ClientStompFrame frame) throws IOException, InterruptedException {
        String messageID = frame.getHeader("ack");
        ClientStompFrame ackFrame = conn.createFrame("ACK");
        ackFrame.addHeader("id", messageID);
        ClientStompFrame response = conn.sendFrame(ackFrame);
        if (response != null) {
            throw new IOException("failed to ack " + response);
        }
    }

    private void ack(StompClientConnection conn, String mid) throws IOException, InterruptedException {
        ClientStompFrame ackFrame = conn.createFrame("ACK");
        ackFrame.addHeader("id", mid);
        conn.sendFrame(ackFrame);
    }

    private void ack(StompClientConnection conn, String mid, String txID) throws IOException, InterruptedException {
        ClientStompFrame ackFrame = conn.createFrame("ACK");
        ackFrame.addHeader("id", mid);
        ackFrame.addHeader("transaction", txID);
        conn.sendFrame(ackFrame);
    }

    private void nack(StompClientConnection conn, String mid) throws IOException, InterruptedException {
        ClientStompFrame ackFrame = conn.createFrame("NACK");
        ackFrame.addHeader("id", mid);
        conn.sendFrame(ackFrame);
    }

    class SubscriberThread
    extends Thread {
        final StompClientConnection connection;
        final CountDownLatch startFlag;
        final CountDownLatch alignFlag;
        final AtomicInteger errors;

        SubscriberThread(String name, StompClientConnection connection, CountDownLatch startFlag, CountDownLatch alignFlag) {
            super(name);
            this.errors = new AtomicInteger(0);
            this.connection = connection;
            this.startFlag = startFlag;
            this.alignFlag = alignFlag;
        }

        @Override
        public void run() {
            try {
                this.alignFlag.countDown();
                this.startFlag.await();
                this.connection.connect(StompV12Test.this.defUser, StompV12Test.this.defPass, "myClientID");
                ClientStompFrame frame = StompTestBase.subscribeTopic(this.connection, UUID.randomUUID().toString(), "client-individual", "123");
                if (frame.getCommand().equals("ERROR")) {
                    this.errors.incrementAndGet();
                }
            }
            catch (Exception e) {
                e.printStackTrace();
                this.errors.incrementAndGet();
            }
            finally {
                try {
                    this.connection.disconnect();
                    StompV12Test.this.waitDisconnect((StompClientConnectionV12)this.connection);
                }
                catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }
}

