/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.faulttolerance.core.retry;

import io.smallrye.faulttolerance.core.FaultToleranceStrategy;
import io.smallrye.faulttolerance.core.InvocationContext;
import io.smallrye.faulttolerance.core.retry.RetryEvents;
import io.smallrye.faulttolerance.core.retry.RetryLogger;
import io.smallrye.faulttolerance.core.retry.SyncDelay;
import io.smallrye.faulttolerance.core.stopwatch.RunningStopwatch;
import io.smallrye.faulttolerance.core.stopwatch.Stopwatch;
import io.smallrye.faulttolerance.core.util.ExceptionDecision;
import io.smallrye.faulttolerance.core.util.Preconditions;
import io.smallrye.faulttolerance.core.util.SneakyThrow;
import java.util.function.Supplier;
import org.eclipse.microprofile.faulttolerance.exceptions.FaultToleranceException;

public class Retry<V>
implements FaultToleranceStrategy<V> {
    final FaultToleranceStrategy<V> delegate;
    final String description;
    private final ExceptionDecision exceptionDecision;
    final long maxRetries;
    final long maxTotalDurationInMillis;
    private final Supplier<SyncDelay> delayBetweenRetries;
    final Stopwatch stopwatch;

    public Retry(FaultToleranceStrategy<V> delegate, String description, ExceptionDecision exceptionDecision, long maxRetries, long maxTotalDurationInMillis, Supplier<SyncDelay> delayBetweenRetries, Stopwatch stopwatch) {
        this.delegate = Preconditions.checkNotNull(delegate, "Retry delegate must be set");
        this.description = Preconditions.checkNotNull(description, "Retry description must be set");
        this.exceptionDecision = Preconditions.checkNotNull(exceptionDecision, "Exception decision must be set");
        this.maxRetries = maxRetries < 0L ? Long.MAX_VALUE : maxRetries;
        this.maxTotalDurationInMillis = maxTotalDurationInMillis <= 0L ? Long.MAX_VALUE : maxTotalDurationInMillis;
        this.delayBetweenRetries = Preconditions.checkNotNull(delayBetweenRetries, "Delay must be set");
        this.stopwatch = Preconditions.checkNotNull(stopwatch, "Stopwatch must be set");
    }

    @Override
    public V apply(InvocationContext<V> ctx) throws Exception {
        RetryLogger.LOG.trace("Retry started");
        try {
            V v = this.doApply(ctx);
            return v;
        }
        finally {
            RetryLogger.LOG.trace("Retry finished");
        }
    }

    private V doApply(InvocationContext<V> ctx) throws Exception {
        long counter;
        SyncDelay delay = this.delayBetweenRetries.get();
        RunningStopwatch runningStopwatch = this.stopwatch.start();
        Throwable lastFailure = null;
        for (counter = 0L; counter <= this.maxRetries && runningStopwatch.elapsedTimeInMillis() < this.maxTotalDurationInMillis; ++counter) {
            if (counter > 0L) {
                RetryLogger.LOG.debugf("%s invocation failed, retrying (%d/%d)", this.description, counter, this.maxRetries);
                ctx.fireEvent(RetryEvents.Retried.INSTANCE);
                try {
                    delay.sleep(lastFailure);
                }
                catch (InterruptedException e) {
                    ctx.fireEvent(RetryEvents.Finished.EXCEPTION_NOT_RETRYABLE);
                    throw e;
                }
                catch (Exception e) {
                    ctx.fireEvent(RetryEvents.Finished.EXCEPTION_NOT_RETRYABLE);
                    if (Thread.interrupted()) {
                        throw new InterruptedException();
                    }
                    throw e;
                }
                if (runningStopwatch.elapsedTimeInMillis() >= this.maxTotalDurationInMillis) break;
            }
            try {
                V result = this.delegate.apply(ctx);
                ctx.fireEvent(RetryEvents.Finished.VALUE_RETURNED);
                return result;
            }
            catch (InterruptedException e) {
                throw e;
            }
            catch (Throwable e) {
                if (Thread.interrupted()) {
                    ctx.fireEvent(RetryEvents.Finished.EXCEPTION_NOT_RETRYABLE);
                    throw new InterruptedException();
                }
                if (this.shouldAbortRetrying(e)) {
                    ctx.fireEvent(RetryEvents.Finished.EXCEPTION_NOT_RETRYABLE);
                    throw e;
                }
                lastFailure = e;
                continue;
            }
        }
        if (counter > this.maxRetries) {
            ctx.fireEvent(RetryEvents.Finished.MAX_RETRIES_REACHED);
        } else {
            ctx.fireEvent(RetryEvents.Finished.MAX_DURATION_REACHED);
        }
        if (lastFailure != null) {
            throw SneakyThrow.sneakyThrow(lastFailure);
        }
        throw new FaultToleranceException(this.description + " reached max retries or max retry duration");
    }

    boolean shouldAbortRetrying(Throwable e) {
        return this.exceptionDecision.isConsideredExpected(e);
    }
}

