package org.apache.mina.common;

import java.util.concurrent.atomic.AtomicInteger;

/**
 * An {@link IoFuture} of {@link IoFuture}s.  It is useful when you want to
 * get notified when all {@link IoFuture}s are complete.  It is not recommended
 * to use {@link CompositeIoFuture} if you just want to wait for all futures.
 * In that case, please use {@link IoUtil#await(Iterable)} instead
 * for better performance.
 * 
 * @author The Apache MINA Project (dev@mina.apache.org)
 * @version $Rev: 589474 $, $Date: 2007-10-29 12:03:14 +0900 (월, 29 10월 2007) $
 *
 * @param <E> the type of the child futures.
 */
public class CompositeIoFuture<E extends IoFuture> extends DefaultIoFuture {
    
    private final NotifyingListener listener = new NotifyingListener();
    private final AtomicInteger unnotified = new AtomicInteger();
    private volatile boolean constructionFinished;
    
    public CompositeIoFuture(Iterable<E> children) {
        super(null);
        
        for (E f: children) {
            f.addListener(listener);
            unnotified.incrementAndGet();
        }
        
        constructionFinished = true;
        if (unnotified.get() == 0) {
            setValue(true);
        }
    }
    
    private class NotifyingListener implements IoFutureListener<IoFuture> {
        public void operationComplete(IoFuture future) {
            if (unnotified.decrementAndGet() == 0 && constructionFinished) {
                setValue(true);
            }
        }
    }
}
