/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 *
 */
package org.apache.mina.common;

import java.io.IOException;
import java.net.SocketAddress;
import java.util.List;

/**
 * Accepts incoming connection, communicates with clients, and fires events to
 * {@link IoHandler}s.
 * <p>
 * Please refer to
 * <a href="../../../../../xref-examples/org/apache/mina/examples/echoserver/Main.html">EchoServer</a>
 * example.
 * <p>
 * You should bind to the desired socket address to accept incoming
 * connections, and then events for incoming connections will be sent to
 * the specified default {@link IoHandler}.
 * <p>
 * Threads accept incoming connections start automatically when
 * {@link #bind()} is invoked, and stop when {@link #unbind()} is invoked.
 *
 * @author The Apache MINA Project (dev@mina.apache.org)
 * @version $Rev: 593496 $, $Date: 2007-11-09 20:36:01 +0900 (금, 09 11월 2007) $
 */
public interface IoAcceptor extends IoService {
    /**
     * Returns the local address to bind.  If more than one address are set,
     * one of them will be returned, but it's not necessarily the firstly
     * specified address in {@link #setLocalAddresses(SocketAddress...)}.
     */
    SocketAddress getLocalAddress();
    
    /**
     * Returns a {@link List} of the local addresses to bind.
     *
     * @throws IllegalStateException if this service is already running.
     */
    List<SocketAddress> getLocalAddresses();

    /**
     * Sets the local address to bind.
     *
     * @throws IllegalStateException if this service is already running.
     */
    void setLocalAddress(SocketAddress localAddress);
    
    /**
     * Sets the local addresses to bind.  If more than one address is
     * specified, {@link #bind()} method binds to all the addresses with
     * the same {@link IoHandler}.
     *
     * @throws IllegalStateException if this service is already running.
     */
    void setLocalAddresses(SocketAddress... localAddresses);
    
    /**
     * Sets the local addresses to bind.  If more than one address is
     * specified, {@link #bind()} method binds to all the addresses with
     * the same {@link IoHandler}.
     *
     * @throws IllegalStateException if this service is already running.
     */
    void setLocalAddresses(Iterable<SocketAddress> localAddresses);

    /**
     * Returns <tt>true</tt> if and only if all clients are disconnected
     * when this acceptor unbinds the related local address.
     */
    boolean isDisconnectOnUnbind();

    /**
     * Sets whether all clients are disconnected when this acceptor unbinds the
     * related local address.  The default value is <tt>true</tt>.
     */
    void setDisconnectOnUnbind(boolean disconnectOnUnbind);

    /**
     * Bind to the configured local address and start to accept incoming connections.
     *
     * @throws IOException if failed to bind
     */
    void bind() throws IOException;

    /**
     * Unbind from the configured local address and stop to accept incoming connections.
     * All managed connections will be closed if <tt>disconnectOnUnbind</tt> property is set.
     * This method does nothing if not bound yet.
     */
    void unbind();

    /**
     * (Optional) Returns an {@link IoSession} that is bound to the specified
     * <tt>localAddress</tt> and the specified <tt>remoteAddress</tt> which
     * reuses the local address that is already bound by this service.
     * <p>
     * This operation is optional.  Please throw {@link UnsupportedOperationException}
     * if the transport type doesn't support this operation.  This operation is
     * usually implemented for connectionless transport types.
     *
     * @throws UnsupportedOperationException if this operation is not supported
     * @throws IllegalStateException if this service is not running.
     * @throws IllegalArgumentException if this service is not bound to the
     *                                  specified <tt>localAddress</tt>.
     */
    IoSession newSession(SocketAddress remoteAddress, SocketAddress localAddress);
}