package aQute.junit.runtime;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestResult;
import junit.framework.TestSuite;

import org.apache.tools.ant.taskdefs.optional.junit.JUnitResultFormatter;
import org.apache.tools.ant.taskdefs.optional.junit.JUnitTest;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleException;

public class Target {
    final List             testNames  = new ArrayList();
    int                    port       = -1;
    boolean                keepAlive;
    String                 target;
    boolean                deferred   = false;
    boolean                clear      = false;
    boolean                verbose    = true;
    final Properties       properties = new Properties();
    final GenericFramework framework  = new GenericFramework(properties);
    String                 reportName = "test-report.xml";
    boolean                waitForEver;

    /*
     * -version 3 -port 55310 -testLoaderClass
     * org.eclipse.jdt.internal.junit.runner.junit3.JUnit3TestLoader
     * -loaderpluginname org.eclipse.jdt.junit.runtime -testNameFile
     * /tmp/testNames18041.txt
     */

    public static void main(String[] args) {
        Target target = new Target();
        System.exit(target.run(args));
    }

    int run(String args[]) {
        try {
            init(args);
            if (port == -1)
                return doTesting(new XMLReport(reportName));
            else
                return doTesting(new JUnitReport(port));
        } catch (Throwable e) {
            error("bnd runtime", e);
            framework.report(System.out);
            return -1;
        }
    }

    void init(String args[]) throws Exception {
        for (int i = 0; i < args.length; i++) {
            if ("-version".equals(args[i])) {
                if (!"3".equals(args[++i].trim()))
                    throw new RuntimeException(
                            "This target only works with JUnit protocol #3");
            } else if ("-port".equals(args[i]))
                port = Integer.parseInt(args[++i]);
            else if ("-deferred".equals(args[i])) {
                deferred = true;
            } else if ("-clear".equals(args[i])) {
                clear = true;
            } else if ("-testNameFile".equals(args[i]))
                processFile(new File(args[++i]));
            else if ("-testLoaderClass".equals(args[i])) // From old
                // interface
                i++;
            else if ("-loaderpluginname".equals(args[i])) // From
                // old
                // interface
                i++;
            else if ("-test".equals(args[i]))
                testNames.add(args[++i]);
            else if ("-classNames".equals(args[i]))
                testNames.add(args[++i]);
            else if ("-bundle".equals(args[i]))
                framework.addBundle(new File(args[++i]));
            else if ("-export".equals(args[i]))
                framework.addSystemPackage(args[++i]);
            else if ("-framework".equals(args[i]))
                framework.setFramework(args[++i]);
            else if ("-keepalive".equals(args[i]))
                keepAlive = true;
            else if ("-set".equals(args[i])) {
                properties.setProperty(args[++i], args[++i]);
            } else if ("-target".equals(args[i])) {
                target = args[++i];
                framework.addBundle(new File(target));
            } else if ("-storage".equals(args[i])) {
                framework.setStorage(new File(args[++i]));
            } else if ("-report".equals(args[i]))
                reportName = args[++i];
            else if ("-verbose".equals(args[i])) {
                verbose = true;
                System.out.println("bnd OSGi Runtime");
            } else
                warning("Do not understand arg: " + args[i]);
        }
        System.getProperties().putAll(properties);
    }

    /**
     * We are not started from JUnit. This means we start the environment,
     * install, the bundles, and run. If the target bundle has a Test-Cases
     * header, we will run JUnit tests on that class.
     */
    void checkTestCases(Bundle bundle) throws Throwable {
        String testcases = (String) bundle.getHeaders().get("Test-Cases");
        if (testcases == null)
            return;

        String[] classes = testcases.split("\\s*,\\s*");
        for (int i = 0; i < classes.length; i++)
            testNames.add(classes[i]);
    }

    /**
     * Main test routine.
     * 
     * @param testReporter
     * @param framework
     * @param targetBundle
     * @param testNames
     * @return
     * @throws Throwable
     */
    private int doTesting(TestReporter testReporter) throws Throwable {
        
    	Bundle targetBundle;
        
        long startTime = System.currentTimeMillis();
		try {
			List<Throwable> activateErrors = framework.activate();
			if (activateErrors.size() > 0) {
				handleSetupErrors(activateErrors, startTime);
				return activateErrors.size();
			}
				
			boolean report = properties.containsKey("report");
			if (report)
			    framework.report(System.out);

			targetBundle = framework.getBundle(target);
			if (targetBundle == null)
			    throw new IllegalArgumentException("Cannot obtain target bundle: " + target);

			// Verify if we have any test names set
			if (testNames.size() == 0)
			    checkTestCases(targetBundle);

			if (testNames.size() == 0) {
			    System.out.println("No test cases to run, waiting for the framework to quit");
			    framework.waitForStop(0);
			    System.out.println("And the framework is gone!");
			    return 0;
			}
		} catch (Throwable th) {
			handleSetupError(th, startTime);
			throw th;
		}

        int allErrors = 0;
        
        TestSuite suite = createSuite(targetBundle, testNames);
        testReporter.begin(framework, targetBundle, null, suite.countTestCases());
        
        try {
            Enumeration tests = suite.tests();
            while (tests.hasMoreElements())
            {
            	Test test = (Test) tests.nextElement();
            	
                TestResult result = new TestResult();
                
                // Note, this also sets the BundleContext
                BasicTestReport basicReport = new BasicTestReport();
                result.addListener(basicReport);
                result.addListener(testReporter);
                
                // Get the test name
            	String testName = null;
            	if (test instanceof TestSuite)
            		testName = ((TestSuite)test).getName();
            	else if (test instanceof TestCase)
            		testName = ((TestCase)test).getName();
                
                List<JUnitResultFormatter> formatters = getResultFormatters(testName);
                for (JUnitResultFormatter formatter : formatters) 
                    result.addListener(formatter);
                
				JUnitTest junitTest = new JUnitTest(testName);
                junitTest.setProperties(System.getProperties());
                
                // Start the report formatters
                startTime = System.currentTimeMillis();
                fireStartTestSuite(formatters, junitTest);
                basicReport.begin(framework, targetBundle, null, test.countTestCases());
                
                try {
                	
                	// Run the test
					test.run(result);
					
				} finally {
					
	                // End the report formatters
	                junitTest.setCounts(test.countTestCases(), result.failureCount(), result.errorCount());
	                junitTest.setRunTime(System.currentTimeMillis() - startTime);
	                fireEndTestSuite(formatters, junitTest);
	                basicReport.end();
				}
            	
            	allErrors += result.errorCount();
            }
			
            return allErrors;
            
        } finally {
            testReporter.end();
            
            if (properties.containsKey("wait")) {
                framework.waitForStop(10000000);
            }
            framework.deactivate();
        }
    }

	/**
	 *  Add the standard plain and xml formatters 
	 */
	private List<JUnitResultFormatter> getResultFormatters(String testName) throws IOException {
		String reportDir = new File(reportName).getParentFile() + "/test-reports";
		List<JUnitResultFormatter> formatters = new ArrayList<JUnitResultFormatter>();
		formatters.add(new XMLResultFormatter(reportDir, testName));
		formatters.add(new PlainResultFormatter(reportDir, testName));
		return formatters;
	}

	private void handleSetupError(Throwable th, long startTime) throws IOException {
		List<Throwable> errors = Arrays.asList(new Throwable[] { th } );
		handleSetupErrors(errors, startTime);
	}

    /**
     * Handles any error that might occur during
     *  
     *  #1 framework startup
     *  #2 list of test bundle install
     *  #3 list of test bundle start
     *  
     *  [TODO] 
     *  The above steps should probably be done as part of normal
     *  JUnit test setup. Special handling should not be necessary.  
     */
	private void handleSetupErrors(List<Throwable> errors, long startTime) throws IOException {

		// Print the stack traces to the legacy report file 
		PrintStream ps = new PrintStream(new FileOutputStream(reportName));
		for (Throwable ex : errors) {
			ex.printStackTrace(ps);
			ps.println();
		}
		ps.close();
		
        TestResult result = new TestResult();
        
        String targetName = new File (target).getName();
        if (targetName.endsWith(".jar"));
           	targetName = targetName.substring(0, targetName.length() - 4);
        
        List<JUnitResultFormatter> formatters = getResultFormatters(targetName);
        for (JUnitResultFormatter formatter : formatters) 
            result.addListener(formatter);
        
		JUnitTest junitTest = new JUnitTest(targetName);
        junitTest.setProperties(System.getProperties());
        
        // Start the report formatters
        fireStartTestSuite(formatters, junitTest);
		for (Throwable ex : errors) 
			result.addError(null, ex);
        
        // End the report formatters
        junitTest.setCounts(0, result.failureCount(), result.errorCount());
        junitTest.setRunTime(System.currentTimeMillis() - startTime);
        fireEndTestSuite(formatters, junitTest);
	}

    private void fireStartTestSuite(List<JUnitResultFormatter> formatters, JUnitTest junitTest) {
        for (JUnitResultFormatter formatter : formatters) {
            formatter.startTestSuite(junitTest);
        }
    }

    private void fireEndTestSuite(List<JUnitResultFormatter> formatters, JUnitTest junitTest) {
        for (JUnitResultFormatter formatter : formatters) {
            formatter.endTestSuite(junitTest);
        }
    }
    
    /**
     * Convert the test names to a test suite.
     * 
     * @param tfw
     * @param testNames
     * @return
     * @throws Exception
     */
    private TestSuite createSuite(Bundle tfw, List testNames) throws Exception {
        TestSuite suite = new TestSuite();
        for (Iterator i = testNames.iterator(); i.hasNext();) {
            String fqn = (String) i.next();
            int n = fqn.indexOf(':');
            if (n > 0) {
                String method = fqn.substring(n + 1);
                fqn = fqn.substring(0, n);
                Class clazz = tfw.loadClass(fqn);
                suite.addTest(TestSuite.createTest(clazz, method));
            } else {
                Class clazz = tfw.loadClass(fqn);
                suite.addTestSuite(clazz);
            }
        }
        return suite;
    }

    private void warning(String string) {
        System.out.println("warning: " + string);
    }

    private void error(String string, Throwable e) {
        if (e instanceof BundleException)
            GenericFramework.report((BundleException) e, System.out);
        else {
            System.out.println(string + " : " + e);

            if (verbose)
                e.printStackTrace();
        }
    }

    private void processFile(File file) throws IOException {
        FileReader rdr = new FileReader(file);
        BufferedReader brdr = new BufferedReader(rdr);
        String line = brdr.readLine();
        while (line != null) {
            testNames.add(line.trim());
            line = brdr.readLine();
        }
        rdr.close();
    }

}
