/*******************************************************************************
 * Copyright (c) 2004, 2008 Actuate Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *  Actuate Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.datatools.enablement.oda.xml.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.datatools.connectivity.oda.OdaException;

/**
 * This class is used to manager the XML data source. If the XML data source is 
 * passed as InputStream or URL using protocol other than File, then we will make
 * a local cache for it to accelerate the processing.
 */
public class XMLDataInputStreamCreator
{
	//	The template file name which might be used to create a temp file for the InputStream
	private static final String TEMPFILENAME = "tempXMLData";	//$NON-NLS-1$
    private static final String EMPTY_STRING = "";   //$NON-NLS-1$
	
	private static Map inputStreamCache = Collections.synchronizedMap( new HashMap( ));
	
	//
	private IInputXMLDataSource inputData = null;

	private String encoding = null; //the encoding the the XMLDataInputStream
	private File tempFile = null;
	
	/**
	 * @return the encoding
	 */
	public String getEncoding( )
	{
		return encoding;
	}

	
	/**
	 * @param encoding the encoding to set
	 */
	public void setEncoding( String encoding )
	{
		if ( EMPTY_STRING.equals(encoding) )
		{
			this.encoding = null;
		}
		else
		{
			this.encoding = encoding;
		}
	}
	
	/**
	 * Return an XMLDataInputStreamCreator instance according to the given key and encoding.
	 * @param key
	 * @param encoding
	 * @return
	 * @throws OdaException
	 */
	public static XMLDataInputStreamCreator getCreator( String key, String encoding ) throws OdaException
	{
		XMLDataInputStreamCreator creator = XMLDataInputStreamCreator.getCreator( key );
		creator.setEncoding( encoding );
		return creator;
	}

	/**
	 * Return an XMLDataInputStreamCreator instance according to the given key and encoding.
	 * @param key
	 * @param encoding
	 * @return
	 * @throws OdaException
	 */
	public static synchronized XMLDataInputStreamCreator getCreator( XMLCreatorContent content) throws OdaException
	{
		Object key = content.getKey( );
		XMLDataInputStreamCreator creator = null;

		if ( key != null )
		{
			if ( key instanceof String )
			{
				creator = XMLDataInputStreamCreator.getCreator( (String) key,
						content.getEncoding( ) );
			}
			else if ( key instanceof InputStream )
			{
				creator = XMLDataInputStreamCreator.getCreator( (InputStream) key, content );
			}
		}
		return creator;
	}

	/**
	 * Return an XMLDataInputStreamCreator instance according to the given key and encoding.
	 * @param key
	 * @param encoding
	 * @return
	 * @throws OdaException
	 */
	public static XMLDataInputStreamCreator getCreator( InputStream key,
			XMLCreatorContent content ) throws OdaException
	{
		synchronized ( inputStreamCache )
		{
			if ( inputStreamCache.get( key ) == null )
			{
				XMLDataInputStreamCreator creator;
				// memory cache is used
				MemoryCacheDataSource cache = new MemoryCacheDataSource( key,
						content.getMemorySize( ) );
				if ( cache.canUseMemoryCache( ) )
				{
					creator = new XMLDataInputStreamCreator( cache );
				}
				else
				{
					creator = new XMLDataInputStreamCreator( key, true, cache );
				}
				inputStreamCache.put( key, creator );
				return creator;
			}
			else
			{
				return (XMLDataInputStreamCreator) inputStreamCache.get( key );
			}
		}
	}
	
	/**
	 * 
	 * @param key
	 * @return
	 * @throws OdaException
	 */
	public static XMLDataInputStreamCreator getCreator( String key ) throws OdaException
	{
		synchronized ( inputStreamCache )
		{
			if ( inputStreamCache.get( key ) == null )
			{
				XMLDataInputStreamCreator creator = new XMLDataInputStreamCreator( key );
				inputStreamCache.put( key, creator );
				return creator;
			}
			else
			{
				return (XMLDataInputStreamCreator) inputStreamCache.get( key );
			}
		}
	}

	/**
	 * @param is
	 * @throws OdaException
	 */
	private File createTemporaryFile( InputStream is, IInputXMLDataSource dataSource ) throws OdaException
	{
		BufferedInputStream bis = new BufferedInputStream(is);
		
		try
		{
			File file = createFileWithUniqueName( );
			
			OutputStream fos = new BufferedOutputStream( new FileOutputStream( file ));
			int abyte;
			if ( dataSource != null )
			{
				while ( ( abyte = dataSource.read( ) ) != -1 )
				{
					fos.write( abyte );
				}
			}
			while( (abyte = bis.read())!= -1)
			{	
				fos.write(abyte);
			}
			fos.close();
			bis.close( );
			inputData = new URLInputStream( file.toURL( ) );
			return file;
		}
		catch ( IOException e )
		{
			throw new OdaException( e.getLocalizedMessage() );
		}
	}
	
	/**
	 * Delete the temp file and remove the creator from cache.
	 * 
	 * @param key
	 */
	public static void close( Object key )
	{
		if ( key != null )
		{
			synchronized ( inputStreamCache )
			{

				XMLDataInputStreamCreator creator = (XMLDataInputStreamCreator) inputStreamCache.get( key );
				if ( creator != null )
				{
					creator.close( );
					inputStreamCache.remove( key );
					creator = null;
				}
			}
		}
	}
	
	private void close( )
	{
		if ( this.tempFile != null && !this.tempFile.delete( ) )
		{
			this.tempFile.deleteOnExit( );
		}
	}
	
	/**
	 * @return
	 * @throws IOException 
	 */
	private File createFileWithUniqueName( ) throws IOException
	{
		File file = File.createTempFile( TEMPFILENAME, null );
		file.deleteOnExit( );
		return file;
	}
	
	/**
	 * 
	 * @param xmlFile
	 * @throws OdaException
	 */
	protected XMLDataInputStreamCreator( String xmlFile ) throws OdaException
	{
		URL url = null;

		// First try to parse the input string as file name.
		try
		{
			File f = new File( xmlFile );
			if( f.exists() )
				url = f.toURI( ).toURL( );
			this.inputData = new URLInputStream( url );
		}
		catch ( IOException e )
		{
		}
		try
		{
			// Then try to parse the input string as a url in web.
			if ( url == null )
			{

				url = new URL( xmlFile );
				tempFile = this.createTemporaryFile( url.openStream( ), null );
			}
		}
		catch ( MalformedURLException e )
		{
			throw new OdaException( e.getLocalizedMessage() );
		}
		catch ( IOException e )
		{
			throw new OdaException( e.getLocalizedMessage() );
		}
	}
	
	/**
	 * 
	 * @param stream
	 */
	protected XMLDataInputStreamCreator( IInputXMLDataSource stream )
	{
		this.inputData = stream;
	}
	
	/**
	 * 
	 * @param stream
	 */
	protected XMLDataInputStreamCreator( InputStream stream, boolean isNestedQuery, IInputXMLDataSource inputSource )
	{
		if ( isNestedQuery )
		{
			try
			{
				this.tempFile = this.createTemporaryFile( stream, inputSource );
			}
			catch ( OdaException e )
			{
			}
		}
		else
			this.inputData = new InputStreamST( stream ); ;
	}
	
	/**
	 * 
	 * @return
	 * @throws OdaException
	 */
	public XMLDataInputStream createXMLDataInputStream( ) throws OdaException
	{
		return new XMLDataInputStream( this.inputData , encoding);
	}
}
