/*******************************************************************************
 * Copyright (c) 2006 NEC Soft, Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     NEC Soft, Ltd.- initial API and implementation
 *******************************************************************************/
package org.eclipse.datatools.sqltools.tablewizard.ui.wizardpages.columns;

import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.datatools.connectivity.sqm.core.definition.DatabaseDefinition;
import org.eclipse.datatools.connectivity.sqm.internal.core.RDBCorePlugin;
import org.eclipse.datatools.modelbase.dbdefinition.PredefinedDataTypeDefinition;
import org.eclipse.datatools.modelbase.sql.datatypes.DataType;
import org.eclipse.datatools.modelbase.sql.schema.Database;
import org.eclipse.datatools.modelbase.sql.schema.SQLSchemaPackage;
import org.eclipse.datatools.modelbase.sql.tables.Column;
import org.eclipse.datatools.modelbase.sql.tables.PersistentTable;
import org.eclipse.datatools.modelbase.sql.tables.SQLTablesPackage;
import org.eclipse.datatools.sqltools.tablewizard.ui.utils.EObjectListPropertyContentProvider;
import org.eclipse.datatools.sqltools.tablewizard.ui.utils.ResourceLoader;
import org.eclipse.datatools.sqltools.tablewizard.ui.utils.TableColumnsPropertyLabelProvider;
import org.eclipse.datatools.sqltools.tablewizard.ui.wizardpages.columns.actions.AddColumnAction;
import org.eclipse.datatools.sqltools.tablewizard.ui.wizardpages.columns.actions.DeleteColumnAction;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Text;

/**
 * @author yuanying
 * 
 */
public class GenericColumnsPage extends WizardPage {
	
	private static final String COLUMN_PAGE_TITLE_LABEL = "column_page.title.label"; //$NON-NLS-1$
	private static final String COLUMN_PAGE_DESCRIPTION_LABEL = "column_page.description.label"; //$NON-NLS-1$

	private PersistentTable table;

	private Map dataTypeMap = new HashMap();

	private DatabaseDefinition dbDifinition;

	// private List columnList;

	public GenericColumnsPage(String pageName, PersistentTable table) {
		super(pageName);
		this.table = table;
		Database db = this.table.getSchema().getDatabase() != null?
							this.table.getSchema().getDatabase():
							this.table.getSchema().getCatalog().getDatabase();
		this.dbDifinition = RDBCorePlugin.getDefault()
				.getDatabaseDefinitionRegistry().getDefinition(db);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		this.setTitle(ResourceLoader.queryString(COLUMN_PAGE_TITLE_LABEL));
		this.setDescription(ResourceLoader.queryString(COLUMN_PAGE_DESCRIPTION_LABEL));
		GenericColumnsComposite composite = new GenericColumnsComposite(parent,
				SWT.NONE);
		this.setControl(composite);

		this.bindTableName(composite);

		this.bindColumns(composite);

		this.bindColumn(composite);
	}

	/**
	 * 
	 * @param composite
	 */
	private void bindColumn(final GenericColumnsComposite composite) {
		final TableViewer columnsViewer = composite.getColumnsTableViewer();
		final ComboViewer dataTypeComboViewer = composite
				.getDataTypeComboViewer();
		dataTypeComboViewer.setContentProvider(new ArrayContentProvider());
		dataTypeComboViewer.setLabelProvider(new LabelProvider() {
			public String getText(Object element) {
				DataType dataType = (DataType) element;
				return dataType.getName();
			}
		});
		// TODO consider editing table wizard. now not consider "table edit".
		columnsViewer
				.addSelectionChangedListener(new ISelectionChangedListener() {
					public void selectionChanged(SelectionChangedEvent event) {
						if (!event.getSelection().isEmpty()) {
							IStructuredSelection ss = (IStructuredSelection) event
									.getSelection();
							Column key = (Column) ss.getFirstElement();
							DataType[] dataTypes = (DataType[]) dataTypeMap
									.get(key);
							if (dataTypes == null) {
								dataTypes = computeDataTypes();
								dataTypeMap.put(key, dataTypes);
							}
							dataTypeComboViewer.setInput(dataTypes);
						}
					}
				});

		columnsViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				IStructuredSelection ss = ((IStructuredSelection) event.getSelection());
				if (ss.isEmpty()) {
					composite.getColumnNameText().setText(""); //$NON-NLS-1$
					composite.getDefaultValueText().setText(""); //$NON-NLS-1$
					composite.getNotNullCheck().setSelection(false);
					composite.getDataTypeComboViewer().setSelection(new StructuredSelection("")); //$NON-NLS-1$
					return;
				}
				Column c = (Column) ss.getFirstElement();
				if (c.getName() != null) {
				composite.getColumnNameText().setText(c.getName());
				}
				if (c.getDefaultValue() != null) {
				composite.getDefaultValueText().setText(c.getDefaultValue());
				}
				composite.getNotNullCheck().setSelection(c.isNullable());
				if (c.getDataType() != null) {
				composite.getDataTypeComboViewer().setSelection(new StructuredSelection(c.getDataType()));
				}
			}
		});
		
		composite.getColumnNameText().addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent event) {
				Text text = (Text) event.getSource();
				IStructuredSelection ss = ((IStructuredSelection) columnsViewer.getSelection());
				if (ss.isEmpty()) {
					return;
				}
				Column c = (Column) ss.getFirstElement();
				c.setName(text.getText());
			}
		});
		
		composite.getDefaultValueText().addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent event) {
				Text text = (Text) event.getSource();
				IStructuredSelection ss = ((IStructuredSelection) columnsViewer.getSelection());
				if (ss.isEmpty()) {
					return;
				}
				Column c = (Column) ss.getFirstElement();
				c.setDefaultValue(text.getText());
			}
		});
		
		composite.getNotNullCheck().addSelectionListener(new SelectionListener() {
			public void widgetDefaultSelected(SelectionEvent event) {}
			public void widgetSelected(SelectionEvent event) {
				IStructuredSelection ss = ((IStructuredSelection) columnsViewer.getSelection());
				if (ss.isEmpty()) {
					return;
				}
				Column c = (Column) ss.getFirstElement();
				Button button = (Button) event.widget;
				c.setNullable(button.getSelection());
			}
		});
		
		composite.getDataTypeComboViewer().addSelectionChangedListener(new ISelectionChangedListener(){
			public void selectionChanged(SelectionChangedEvent event) {
				IStructuredSelection ss = ((IStructuredSelection) columnsViewer.getSelection());
				if (ss.isEmpty()) {
					return;
				}
				Column c = (Column) ss.getFirstElement();
				
				IStructuredSelection ss2 = ((IStructuredSelection) event.getSelection());
				DataType dataType = (DataType) ss2.getFirstElement();
				c.setDataType(dataType);
			}
		});

		composite.getDataTypeComboViewer().addSelectionChangedListener(
				new ISelectionChangedListener() {
					public void selectionChanged(SelectionChangedEvent event) {
						if (!event.getSelection().isEmpty()) {
							IStructuredSelection ss = (IStructuredSelection) event
									.getSelection();
							DataType dataType = (DataType) ss.getFirstElement();
							PredefinedDataTypeDefinition predefinedDataTypeDefinition = dbDifinition
									.getPredefinedDataTypeDefinition(dataType
											.getName());
							if (predefinedDataTypeDefinition
									.isLengthSupported()) {
								composite.getDataTypeLengthText().setVisible(
										true);
								composite.getDataTypeLengthLabel().setVisible(
										true);
								composite.getDataTypeLengthText().setText(
										new Integer(getLength(dataType))
												.toString());
								return;
							}
						}
						composite.getDataTypeLengthText().setVisible(false);
						composite.getDataTypeLengthLabel().setVisible(false);
						composite.getDataTypeLengthText().setText(""); //$NON-NLS-1$
					}
				});
		composite.getDataTypeLengthText().addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				String lengthText = composite.getDataTypeLengthText().getText();
				if (!composite.getDataTypeComboViewer().getSelection().isEmpty() && lengthText.length() > 0) {
					IStructuredSelection ss = (IStructuredSelection) composite.getDataTypeComboViewer().getSelection();
					DataType dataType = (DataType) ss.getFirstElement();
					setLength(dataType, new Integer(lengthText).intValue());
				}
			}});
		composite.getDataTypeLengthText().addVerifyListener(new VerifyListener() {
			public void verifyText(VerifyEvent e) {
				char c = e.character;
				if (c < 0x0022) {
				} else if (c >= 0x0030 && c <= 0x0039) {
				} else {
					e.doit = false;
				}
			}});
	}

	/**
	 * 
	 * @param dataType
	 * @param length
	 */
	private void setLength(DataType dataType, int length) {
		try {
			PropertyDescriptor descriptor = getLengthPropertyDescriptor(dataType);
			Method writeMethod = descriptor.getWriteMethod();
			if (!writeMethod.isAccessible()) {
				writeMethod.setAccessible(true);
			}
			writeMethod.invoke(dataType, new Object[] {new Integer(length)});
		} catch (Exception e) {
			// XXX exception handling
			e.printStackTrace();
		}
	}

	/**
	 * 
	 * @param dataType
	 * @return
	 * @throws IntrospectionException
	 */
	private PropertyDescriptor getLengthPropertyDescriptor(DataType dataType)
			throws IntrospectionException {
		PropertyDescriptor descriptor = new PropertyDescriptor("length", //$NON-NLS-1$
				dataType.getClass());
		return descriptor;
	}

	/**
	 * 
	 * @param dataType
	 * @return
	 */
	private int getLength(DataType dataType) {
		try {
			PropertyDescriptor descriptor = getLengthPropertyDescriptor(dataType);
			Method readMethod = descriptor.getReadMethod();
			if (!readMethod.isAccessible()) {
				readMethod.setAccessible(true);
			}
			return ((Integer) readMethod.invoke(dataType, null)).intValue();
		} catch (Exception e) {
			// XXX exception handling
			e.printStackTrace();
			return 0;
		}
	}

	/**
	 * Computes supported datatypes in this schema
	 * 
	 * @return
	 */
	public DataType[] computeDataTypes() {
		ArrayList dataTypeList = new ArrayList();
		Iterator iter = this.dbDifinition.getPredefinedDataTypes();
		while (iter.hasNext()) {
			PredefinedDataTypeDefinition dataTypeDefinition = (PredefinedDataTypeDefinition) iter
					.next();
			DataType dataType = this.dbDifinition
					.getPredefinedDataType(dataTypeDefinition);
			dataTypeList.add(dataType);
		}
		return (DataType[]) dataTypeList.toArray(new DataType[dataTypeList
				.size()]);
	}

	/**
	 * 
	 * @param composite
	 */
	private void bindColumns(GenericColumnsComposite composite) {

		// columns table binding
		TableViewer columnsViewer = composite.getColumnsTableViewer();
		EObjectListPropertyContentProvider columnsContentProvider = new EObjectListPropertyContentProvider(SQLTablesPackage.TABLE__COLUMNS);
		columnsViewer.setContentProvider(columnsContentProvider);

		int[] observeChildrenIDs = new int[] { EcorePackage.ENAMED_ELEMENT__NAME,
				// FIXME if we support user defined type(referenced
				// type), fix this
				// code.
				SQLSchemaPackage.TYPED_ELEMENT__CONTAINED_TYPE,
				SQLTablesPackage.COLUMN__NULLABLE,
				SQLTablesPackage.COLUMN__DEFAULT_VALUE};
		
		columnsViewer.setLabelProvider(new TableColumnsPropertyLabelProvider(this.table,
				SQLTablesPackage.TABLE__COLUMNS,
				observeChildrenIDs));
		List columnList = this.table.getColumns();
		columnsViewer.setInput(this.table);

		final AddColumnAction addColumnAction = new AddColumnAction(columnList, columnsViewer);
		final DeleteColumnAction deleteColumnAction = new DeleteColumnAction(columnList, columnsViewer);

		// add button setting
		composite.getAddButton().addSelectionListener(addColumnAction);

		// delete button setting
		composite.getDeleteButton().addSelectionListener(deleteColumnAction);

		MenuManager menuManager = new MenuManager();
		menuManager.setRemoveAllWhenShown(true);
		menuManager.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager manager) {
				manager.add(addColumnAction);
				manager.add(deleteColumnAction);
			}
		});

		Menu menu = menuManager.createContextMenu(composite
				.getColumnsTableViewer().getControl());
		composite.getColumnsTableViewer().getControl().setMenu(menu);
	}

	/**
	 * 
	 * @param composite
	 */
	private void bindTableName(GenericColumnsComposite composite) {
		// table name binding
		composite.getTableNameText().setText(this.table.getName());
		composite.getTableNameText().addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent event) {
				Text text = (Text) event.getSource();
				table.setName(text.getText());
			}
		});
		// XXX if table name is changed, change text.
	}
}
