/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.derby.internal.ui.connection.dialogs;

import java.text.MessageFormat;

import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.wst.rdb.connection.internal.ui.factories.RSCCoreUIWidgetFactory;
import org.eclipse.wst.rdb.connection.internal.ui.providers.IServerConnectionInformation;
import org.eclipse.wst.rdb.connection.internal.ui.providers.IServerConnectionUIProvider;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.NewConnectionWizard;
import org.eclipse.wst.rdb.derby.internal.ui.util.ResourceLoader;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;

/**
 * @author groux
 */
public abstract class AbstractDerbyConnectionProvider implements IServerConnectionUIProvider, Listener
{

    protected static final RSCCoreUIWidgetFactory factory = RSCCoreUIWidgetFactory.INSTANCE;

    protected static final ResourceLoader resource = ResourceLoader.INSTANCE;
    protected static final String CUI_NEWCW_DATABASE_LBL_UI_ = resource.queryString("CUI_NEWCW_UNI_DATABASE_LBL_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_CREATEDB_BTN_UI_ = resource.queryString("CUI_NEWCW_CREATEDB_BTN_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_UPGRADEDB_BTN_UI_ = resource.queryString("CUI_NEWCW_UPGRADEDB_BTN_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_JDBCDRIVERCLS_LBL_UI_ = resource.queryString("CUI_NEWCW_JDBCDRIVERCLS_LBL_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_HOST_LBL_UI_ = resource.queryString("CUI_NEWCW_UNI_HOST_LBL_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_PORT_LBL_UI_ = resource.queryString("CUI_NEWCW_UNI_PORT_LBL_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_JDBCCLSLOCATION_LBL_UI_ = resource.queryString("CUI_NEWCW_JDBCCLSLOCATION_LBL_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_JARBROWSE_BTN_UI_ = resource.queryString("CUI_NEWCW_JARBROWSE_BTN_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_CONNECTIONURL_LBL_UI_ = resource.queryString("CUI_NEWCW_CONNECTIONURL_LBL_UI_"); //$NON-NLS-1$

    protected static final String CREATE_EQUALS_TRUE_TEXT = "create=true"; //$NON-NLS-1$
    protected static final String UPGRADE_EQUALS_TRUE_TEXT = "upgrade=true"; //$NON-NLS-1$        
    
    protected IServerConnectionInformation connectionInformation;

    protected Composite parentComposite;
    protected Label databaseLabel;
    protected Text databaseCombo;
    protected Button createCheck;
    protected boolean displayUpgrade = false;
    protected Button upgradeCheck;
    protected Label hostLabel;
    protected Text hostCombo;
    protected Label portLabel;
    protected Label driverLabel;
    protected Text driverDisplay;
    protected Label locationLabel;
    protected Text locationCombo;
    protected Button browseLocation;
    protected Label urlLabel;
    protected Text urlDisplay;
    protected Text portCombo;
    protected WizardPage parentPage;

    protected abstract void updateURL();
    
    protected abstract String getDriverClassName();

    public boolean determinePageCompletion()
    {
        return validateControl();
    }

    protected void setConnectionInformation()
    {
        connectionInformation.setDriverClassName(getDriverClassName());
        connectionInformation.setDatabaseName(databaseCombo.getText());
        connectionInformation.setLoadingPath(locationCombo.getText());
        connectionInformation.setURL(urlDisplay.getText());        
        if (determinePageCompletion()) {
            connectionInformation.setInformationComplete(true);
        }
    }

    protected boolean validateControl()
    {
        if (databaseCombo.getText() == null || databaseCombo.getText().equals("")) { //$NON-NLS-1$
            parentPage.setErrorMessage(MessageFormat.format(NewConnectionWizard
                    .getResourceString("CUI_NEWCW_VALIDATE_DATABASE_REQ_UI_"), //$NON-NLS-1$
                    new String[] { NewConnectionWizard.getResourceString("CUI_NEWCW_DEFDBNAME_VAL_UI_") })); //$NON-NLS-1$
            return false;
        } else if (hostCombo != null && hostCombo.isEnabled() && hostCombo.getText().length() < 1) {
            parentPage.setErrorMessage(NewConnectionWizard.getResourceString("CUI_NEWCW_VALIDATE_HOST_REQ_UI_")); //$NON-NLS-1$
            return false;
        } else if (portCombo != null && portCombo.isEnabled() && portCombo.getText().length() < 1) {
            parentPage.setErrorMessage(NewConnectionWizard.getResourceString("CUI_NEWCW_VALIDATE_PORT_REQ_UI_")); //$NON-NLS-1$
            return false;
        } else if (driverDisplay != null && driverDisplay.getEnabled() && driverDisplay.getText().trim().length() < 1) {
            parentPage.setErrorMessage(NewConnectionWizard.getResourceString("CUI_NEWCW_VALIDATE_DRIVERCLASS_REQ_UI_")); //$NON-NLS-1$
            return false;
        } else if (locationCombo != null && locationCombo.getText().trim().length() < 1) {
            parentPage.setErrorMessage(NewConnectionWizard
                    .getResourceString("CUI_NEWCW_VALIDATE_CLASSLOCATION_REQ_UI_")); //$NON-NLS-1$
            return false;
        } else if (urlDisplay != null && urlDisplay.getText().trim().length() == 0) {
            parentPage.setErrorMessage(NewConnectionWizard.getResourceString("CUI_NEWCW_VALIDATE_URL_REQ_UI_")); //$NON-NLS-1$
            return false;
        }
        return true;
    }

    protected void initialize()
    {
        driverDisplay.setText(getDriverClassName());
        this.connectionInformation.setDriverClassName(getDriverClassName());

        // Set default values
        databaseCombo.setText("sample"); //$NON-NLS-1$
        hostCombo.setText("localhost"); //$NON-NLS-1$
        portCombo.setText("1527"); //$NON-NLS-1$

        updateURL();
        addListeners();

        determinePageCompletion();
    }

    public void addListeners()
    {
        locationCombo.addListener(SWT.Modify, this);
        browseLocation.addListener(SWT.Selection, this);
        databaseCombo.addListener(SWT.Modify, this);
        createCheck.addListener(SWT.Selection, this);
        if (displayUpgrade)
            upgradeCheck.addListener(SWT.Selection, this);
        hostCombo.addListener(SWT.Modify, this);
        portCombo.addListener(SWT.Modify, this);
        urlDisplay.addListener(SWT.Modify, this);
    }

    protected void removeListeners() {
        locationCombo.removeListener(SWT.Modify, this);
        browseLocation.removeListener(SWT.Selection, this);
        databaseCombo.removeListener(SWT.Modify, this);
        createCheck.removeListener(SWT.Selection, this);
        if (displayUpgrade)
            upgradeCheck.removeListener(SWT.Selection, this);
        hostCombo.removeListener(SWT.Modify, this);
        portCombo.removeListener(SWT.Modify, this);
        urlDisplay.removeListener(SWT.Modify, this);
    }

    public Composite getServerConnectionUI (Composite parent)
    {
        if (parentComposite == null || parentComposite.isDisposed()) {
            parentComposite = createServerConnectionUI(parent);
        }
        setConnectionInformation();
        return parentComposite;
    }
    
    protected Composite createServerConnectionUI(Composite parent)
    {
        parentComposite = factory.createComposite(parent, SWT.NULL);
        GridLayout layout = new GridLayout(3, false);
        parentComposite.setLayout(layout);

        GridData gd;

        databaseLabel = factory.createLabel(parentComposite, SWT.NONE);
        databaseLabel.setText(CUI_NEWCW_DATABASE_LBL_UI_);

        databaseCombo = factory.createText(parentComposite, SWT.SINGLE | SWT.BORDER);
        gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
        gd.horizontalSpan = 2;
        databaseCombo.setLayoutData(gd);

        createCheck = factory.createButton(parentComposite, SWT.CHECK);
        createCheck.setText(CUI_NEWCW_CREATEDB_BTN_UI_);
        createCheck.setSelection(true);
        gd = new GridData();
        gd.horizontalSpan = 3;
        createCheck.setLayoutData(gd);
        
        if (displayUpgrade) {
	        upgradeCheck = factory.createButton(parentComposite, SWT.CHECK);
	        upgradeCheck.setText(CUI_NEWCW_UPGRADEDB_BTN_UI_);
	        upgradeCheck.setSelection(false);
	        gd = new GridData();
	        gd.horizontalSpan = 3;
	        upgradeCheck.setLayoutData(gd);
        }

        hostLabel = factory.createLabel(parentComposite, SWT.NONE);
        hostLabel.setText(CUI_NEWCW_HOST_LBL_UI_);

        hostCombo = factory.createText(parentComposite, SWT.SINGLE | SWT.BORDER);
        gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
        gd.horizontalSpan = 2;
        hostCombo.setLayoutData(gd);

        portLabel = factory.createLabel(parentComposite, SWT.NONE);
        portLabel.setText(CUI_NEWCW_PORT_LBL_UI_);

        portCombo = factory.createText(parentComposite, SWT.SINGLE | SWT.BORDER);
        gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
        gd.horizontalSpan = 2;
        portCombo.setLayoutData(gd);

        driverLabel = factory.createLabel(parentComposite, SWT.NONE);
        driverLabel.setText(CUI_NEWCW_JDBCDRIVERCLS_LBL_UI_);

        driverDisplay = factory.createText(parentComposite, SWT.SINGLE | SWT.BORDER | SWT.READ_ONLY);
        gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
        gd.horizontalSpan = 2;
        driverDisplay.setLayoutData(gd);

        locationLabel = factory.createLabel(parentComposite, SWT.NONE);
        locationLabel.setText(CUI_NEWCW_JDBCCLSLOCATION_LBL_UI_);
        
        locationCombo = factory.createText(parentComposite, SWT.SINGLE | SWT.BORDER);
        gd = new GridData(GridData.FILL_HORIZONTAL);
        locationCombo.setLayoutData(gd);

        browseLocation = factory.createButton(parentComposite, SWT.PUSH);
        browseLocation.setText(CUI_NEWCW_JARBROWSE_BTN_UI_);
        browseLocation.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));
        gd = new GridData(GridData.HORIZONTAL_ALIGN_CENTER);
        gd.horizontalSpan = 1;
        browseLocation.setLayoutData(gd);
        
        urlLabel = factory.createLabel(parentComposite, SWT.NONE);
        urlLabel.setText(CUI_NEWCW_CONNECTIONURL_LBL_UI_);

        urlDisplay = factory.createText(parentComposite, SWT.SINGLE | SWT.BORDER | SWT.READ_ONLY);
        gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
        gd.horizontalSpan = 2;
        urlDisplay.setLayoutData(gd);

        initialize();
        
        return parentComposite;
    }

    public void setServerConnectionInformation(IServerConnectionInformation connectionInformation)
    {
        this.connectionInformation = connectionInformation;
    }

    public void setWizardPage(WizardPage parentPage)
    {
        this.parentPage = parentPage;
    }
    
    public void setDatabaseDefinition(DatabaseDefinition definition)
    {
    }

    public void handleEvent(Event event) {
        Widget source = event.widget;
        if (source == databaseCombo) {
            updateURL();
        } else if (source == createCheck) {
            updateURL();
        } else if (displayUpgrade && source == upgradeCheck) {
            updateURL();
        } else if (source == locationCombo) {
        } else if (source == this.hostCombo) {
            updateURL();
        } else if (source == this.portCombo) {
            updateURL();
        } else if (source == browseLocation) {
            String file = NewConnectionWizard.getFile((Wizard) parentPage
                    .getWizard(), NewConnectionWizard.fExtensions);
            if (file != null && file.length() > 0) {
                locationCombo.removeListener(SWT.Modify, this);
                locationCombo.setText(file);
                locationCombo.addListener(SWT.Modify, this);

                this.connectionInformation.setLoadingPath(file);
            }
        }
        setConnectionInformation();
    }
}
