/*******************************************************************************
 * Copyright (c) 2007, 2008 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0, which accompanies this distribution
 * and is available at http://www.eclipse.org/legal/epl-v10.html.
 * 
 * Contributors:
 *     Oracle - initial API and implementation
 ******************************************************************************/
package org.eclipse.jpt.core.internal.context.java;

import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jpt.core.context.Query;
import org.eclipse.jpt.core.context.QueryHint;
import org.eclipse.jpt.core.context.java.JavaJpaContextNode;
import org.eclipse.jpt.core.context.java.JavaQuery;
import org.eclipse.jpt.core.context.java.JavaQueryHint;
import org.eclipse.jpt.core.resource.java.QueryAnnotation;
import org.eclipse.jpt.core.resource.java.QueryHintAnnotation;
import org.eclipse.jpt.core.utility.TextRange;
import org.eclipse.jpt.utility.internal.CollectionTools;
import org.eclipse.jpt.utility.internal.iterators.CloneListIterator;


public abstract class AbstractJavaQuery extends AbstractJavaJpaContextNode 
	implements JavaQuery
{
	protected String name;

	protected String query;

	protected final List<JavaQueryHint> hints;

	protected QueryAnnotation queryAnnotation;
	
	protected AbstractJavaQuery(JavaJpaContextNode parent) {
		super(parent);
		this.hints = new ArrayList<JavaQueryHint>();
	}

	protected QueryAnnotation getQueryResource() {
		return this.queryAnnotation;
	}
	
	public String getName() {
		return this.name;
	}

	public void setName(String newName) {
		String oldName = this.name;
		this.name = newName;
		this.queryAnnotation.setName(newName);
		firePropertyChanged(Query.NAME_PROPERTY, oldName, newName);
	}
	
	protected void setName_(String newName) {
		String oldName = this.name;
		this.name = newName;
		firePropertyChanged(Query.NAME_PROPERTY, oldName, newName);
	}

	public String getQuery() {
		return this.query;
	}

	public void setQuery(String newQuery) {
		String oldQuery = this.query;
		this.query = newQuery;
		this.queryAnnotation.setQuery(newQuery);
		firePropertyChanged(Query.QUERY_PROPERTY, oldQuery, newQuery);
	}
	
	protected void setQuery_(String newQuery) {
		String oldQuery = this.query;
		this.query = newQuery;
		firePropertyChanged(Query.QUERY_PROPERTY, oldQuery, newQuery);
	}

	public ListIterator<JavaQueryHint> hints() {
		return new CloneListIterator<JavaQueryHint>(this.hints);
	}

	public int hintsSize() {
		return this.hints.size();
	}
	
	public JavaQueryHint addHint(int index) {
		JavaQueryHint hint = getJpaFactory().buildJavaQueryHint(this);
		this.hints.add(index, hint);
		this.getQueryResource().addHint(index);
		this.fireItemAdded(Query.HINTS_LIST, index, hint);
		return hint;
	}

	protected void addHint(int index, JavaQueryHint hint) {
		addItemToList(index, hint, this.hints, Query.HINTS_LIST);
	}
	
	public void removeHint(QueryHint queryHint) {
		removeHint(this.hints.indexOf(queryHint));
	}
	
	public void removeHint(int index) {
		JavaQueryHint removedHint = this.hints.remove(index);
		this.getQueryResource().removeHint(index);
		fireItemRemoved(Query.HINTS_LIST, index, removedHint);
	}
	
	protected void removeHint_(JavaQueryHint hint) {
		removeItemFromList(hint, this.hints, Query.HINTS_LIST);
	}
	
	public void moveHint(int targetIndex, int sourceIndex) {
		CollectionTools.move(this.hints, targetIndex, sourceIndex);
		this.getQueryResource().moveHint(targetIndex, sourceIndex);
		fireItemMoved(Query.HINTS_LIST, targetIndex, sourceIndex);		
	}
	
	protected void initializeFromResource(QueryAnnotation queryAnnotation) {
		this.queryAnnotation = queryAnnotation;
		this.name = queryAnnotation.getName();
		this.query = queryAnnotation.getQuery();
		this.initializeQueryHints(queryAnnotation);
	}

	protected void update(QueryAnnotation queryAnnotation) {
		this.queryAnnotation = queryAnnotation;
		this.setName_(queryAnnotation.getName());
		this.setQuery_(queryAnnotation.getQuery());
		this.updateQueryHints(queryAnnotation);
	}

	protected void initializeQueryHints(QueryAnnotation queryResource) {
		ListIterator<QueryHintAnnotation> annotations = queryResource.hints();
		
		while(annotations.hasNext()) {
			this.hints.add(createQueryHint(annotations.next()));
		}
	}
	
	protected void updateQueryHints(QueryAnnotation queryResource) {
		ListIterator<JavaQueryHint> hints = hints();
		ListIterator<QueryHintAnnotation> resourceHints = queryResource.hints();
		
		while (hints.hasNext()) {
			JavaQueryHint hint = hints.next();
			if (resourceHints.hasNext()) {
				hint.update(resourceHints.next());
			}
			else {
				removeHint_(hint);
			}
		}
		
		while (resourceHints.hasNext()) {
			addHint(hintsSize(), createQueryHint(resourceHints.next()));
		}
	}

	protected JavaQueryHint createQueryHint(QueryHintAnnotation hintResource) {
		JavaQueryHint queryHint =  getJpaFactory().buildJavaQueryHint(this);
		queryHint.initializeFromResource(hintResource);
		return queryHint;
	}

	public TextRange getValidationTextRange(CompilationUnit astRoot) {
		return this.queryAnnotation.getTextRange(astRoot);
	}
	
	public TextRange getNameTextRange(CompilationUnit astRoot) {
		return this.queryAnnotation.getNameTextRange(astRoot);
	}
	
	public boolean overrides(Query query) {
		// java is at the base of the tree
		return false;
	}
}
