/*
 * Decompiled with CFR 0.152.
 */
package org.opensaml.xml.security;

import java.io.File;
import java.io.IOException;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyException;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.security.interfaces.DSAParams;
import java.security.interfaces.DSAPrivateKey;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.spec.DSAPublicKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.util.HashSet;
import java.util.Set;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import org.apache.commons.ssl.PKCS8Key;
import org.apache.xml.security.Init;
import org.apache.xml.security.algorithms.JCEMapper;
import org.opensaml.xml.Configuration;
import org.opensaml.xml.encryption.EncryptionParameters;
import org.opensaml.xml.encryption.KeyEncryptionParameters;
import org.opensaml.xml.security.SecurityConfiguration;
import org.opensaml.xml.security.SecurityException;
import org.opensaml.xml.security.SigningUtil;
import org.opensaml.xml.security.credential.BasicCredential;
import org.opensaml.xml.security.credential.Credential;
import org.opensaml.xml.security.keyinfo.KeyInfoGenerator;
import org.opensaml.xml.security.keyinfo.KeyInfoGeneratorFactory;
import org.opensaml.xml.security.keyinfo.NamedKeyInfoGeneratorManager;
import org.opensaml.xml.security.x509.BasicX509Credential;
import org.opensaml.xml.signature.KeyInfo;
import org.opensaml.xml.signature.Signature;
import org.opensaml.xml.util.DatatypeHelper;
import org.opensaml.xml.util.LazySet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SecurityHelper {
    private static Logger log = LoggerFactory.getLogger(SecurityHelper.class);
    private static Set<String> rsaAlgorithmURIs;
    private static Set<String> dsaAlgorithmURIs;
    private static Set<String> ecdsaAlgorithmURIs;

    static {
        if (!Init.isInitialized()) {
            Init.init();
        }
        dsaAlgorithmURIs = new LazySet<String>();
        dsaAlgorithmURIs.add("http://www.w3.org/2000/09/xmldsig#dsa-sha1");
        ecdsaAlgorithmURIs = new LazySet<String>();
        ecdsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha1");
        rsaAlgorithmURIs = new HashSet<String>(10);
        rsaAlgorithmURIs.add("http://www.w3.org/2000/09/xmldsig#rsa-sha1");
        rsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#rsa-sha256");
        rsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#rsa-sha384");
        rsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#rsa-sha512");
        rsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#rsa-sha512");
        rsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#rsa-ripemd160");
        rsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#rsa-md5");
    }

    private SecurityHelper() {
    }

    public static String getAlgorithmIDFromURI(String algorithmURI) {
        return DatatypeHelper.safeTrimOrNullString(JCEMapper.translateURItoJCEID((String)algorithmURI));
    }

    public static boolean isHMAC(String signatureAlgorithm) {
        String algoClass = DatatypeHelper.safeTrimOrNullString(JCEMapper.getAlgorithmClassFromURI((String)signatureAlgorithm));
        return "Mac".equals(algoClass);
    }

    public static String getKeyAlgorithmFromURI(String algorithmURI) {
        String apacheValue = DatatypeHelper.safeTrimOrNullString(JCEMapper.getJCEKeyAlgorithmFromURI((String)algorithmURI));
        if (apacheValue != null) {
            return apacheValue;
        }
        if (SecurityHelper.isHMAC(algorithmURI)) {
            return null;
        }
        if (rsaAlgorithmURIs.contains(algorithmURI)) {
            return "RSA";
        }
        if (dsaAlgorithmURIs.contains(algorithmURI)) {
            return "DSA";
        }
        if (ecdsaAlgorithmURIs.contains(algorithmURI)) {
            return "ECDSA";
        }
        return null;
    }

    public static Integer getKeyLengthFromURI(String algorithmURI) {
        String algoClass = DatatypeHelper.safeTrimOrNullString(JCEMapper.getAlgorithmClassFromURI((String)algorithmURI));
        if ("BlockEncryption".equals(algoClass) || "SymmetricKeyWrap".equals(algoClass)) {
            try {
                int keyLength = JCEMapper.getKeyLengthFromURI((String)algorithmURI);
                return new Integer(keyLength);
            }
            catch (NumberFormatException e) {
                log.warn("XML Security config contained invalid key length value for algorithm URI: " + algorithmURI);
            }
        }
        log.info("Mapping from algorithm URI {} to key length not available", (Object)algorithmURI);
        return null;
    }

    public static SecretKey generateSymmetricKey(String algoURI) throws NoSuchAlgorithmException, KeyException {
        String jceAlgorithmName = SecurityHelper.getKeyAlgorithmFromURI(algoURI);
        if (DatatypeHelper.isEmpty(jceAlgorithmName)) {
            log.error("Mapping from algorithm URI '" + algoURI + "' to key algorithm not available, key generation failed");
            throw new NoSuchAlgorithmException("Algorithm URI'" + algoURI + "' is invalid for key generation");
        }
        Integer keyLength = SecurityHelper.getKeyLengthFromURI(algoURI);
        if (keyLength == null) {
            log.error("Key length could not be determined from algorithm URI, can't generate key");
            throw new KeyException("Key length not determinable from algorithm URI, could not generate new key");
        }
        KeyGenerator keyGenerator = KeyGenerator.getInstance(jceAlgorithmName);
        keyGenerator.init(keyLength);
        return keyGenerator.generateKey();
    }

    public static Key extractEncryptionKey(Credential credential) {
        if (credential == null) {
            return null;
        }
        if (credential.getPublicKey() != null) {
            return credential.getPublicKey();
        }
        return credential.getSecretKey();
    }

    public static Key extractDecryptionKey(Credential credential) {
        if (credential == null) {
            return null;
        }
        if (credential.getPrivateKey() != null) {
            return credential.getPrivateKey();
        }
        return credential.getSecretKey();
    }

    public static Key extractSigningKey(Credential credential) {
        if (credential == null) {
            return null;
        }
        if (credential.getPrivateKey() != null) {
            return credential.getPrivateKey();
        }
        return credential.getSecretKey();
    }

    public static Key extractVerificationKey(Credential credential) {
        if (credential == null) {
            return null;
        }
        if (credential.getPublicKey() != null) {
            return credential.getPublicKey();
        }
        return credential.getSecretKey();
    }

    public static Integer getKeyLength(Key key) {
        if (key instanceof SecretKey && "RAW".equals(key.getFormat())) {
            return key.getEncoded().length * 8;
        }
        log.debug("Unable to determine length in bits of specified Key instance");
        return null;
    }

    public static BasicCredential getSimpleCredential(SecretKey secretKey) {
        if (secretKey == null) {
            throw new IllegalArgumentException("A secret key is required");
        }
        BasicCredential cred = new BasicCredential();
        cred.setSecretKey(secretKey);
        return cred;
    }

    public static BasicCredential getSimpleCredential(PublicKey publicKey, PrivateKey privateKey) {
        if (publicKey == null) {
            throw new IllegalArgumentException("A public key is required");
        }
        BasicCredential cred = new BasicCredential();
        cred.setPublicKey(publicKey);
        cred.setPrivateKey(privateKey);
        return cred;
    }

    public static BasicX509Credential getSimpleCredential(X509Certificate cert, PrivateKey privateKey) {
        if (cert == null) {
            throw new IllegalArgumentException("A certificate is required");
        }
        BasicX509Credential cred = new BasicX509Credential();
        cred.setEntityCertificate(cert);
        cred.setPrivateKey(privateKey);
        return cred;
    }

    public static SecretKey decodeSecretKey(byte[] key, char[] password) throws KeyException {
        throw new UnsupportedOperationException("This method is not yet supported");
    }

    public static PublicKey decodePublicKey(byte[] key, char[] password) throws KeyException {
        throw new UnsupportedOperationException("This method is not yet supported");
    }

    public static PublicKey derivePublicKey(PrivateKey key) throws KeyException {
        if (key instanceof DSAPrivateKey) {
            DSAPrivateKey dsaKey = (DSAPrivateKey)key;
            DSAParams keyParams = dsaKey.getParams();
            BigInteger y = keyParams.getQ().modPow(dsaKey.getX(), keyParams.getP());
            DSAPublicKeySpec pubKeySpec = new DSAPublicKeySpec(y, keyParams.getP(), keyParams.getQ(), keyParams.getG());
            try {
                KeyFactory factory = KeyFactory.getInstance("DSA");
                return factory.generatePublic(pubKeySpec);
            }
            catch (GeneralSecurityException e) {
                throw new KeyException("Unable to derive public key from DSA private key", e);
            }
        }
        if (key instanceof RSAPrivateCrtKey) {
            RSAPrivateCrtKey rsaKey = (RSAPrivateCrtKey)key;
            RSAPublicKeySpec pubKeySpec = new RSAPublicKeySpec(rsaKey.getModulus(), rsaKey.getPublicExponent());
            try {
                KeyFactory factory = KeyFactory.getInstance("RSA");
                return factory.generatePublic(pubKeySpec);
            }
            catch (GeneralSecurityException e) {
                throw new KeyException("Unable to derive public key from RSA private key", e);
            }
        }
        throw new KeyException("Private key was not a DSA or RSA key");
    }

    public static PrivateKey decodePrivateKey(File key, char[] password) throws KeyException {
        if (!key.exists()) {
            throw new KeyException("Key file " + key.getAbsolutePath() + " does not exist");
        }
        if (!key.canRead()) {
            throw new KeyException("Key file " + key.getAbsolutePath() + " is not readable");
        }
        try {
            return SecurityHelper.decodePrivateKey(DatatypeHelper.fileToByteArray(key), password);
        }
        catch (IOException e) {
            throw new KeyException("Error reading Key file " + key.getAbsolutePath(), e);
        }
    }

    public static PrivateKey decodePrivateKey(byte[] key, char[] password) throws KeyException {
        try {
            PKCS8Key deocodedKey = new PKCS8Key(key, password);
            return deocodedKey.getPrivateKey();
        }
        catch (GeneralSecurityException e) {
            throw new KeyException("Unable to decode private key", e);
        }
    }

    public static boolean matchKeyPair(PublicKey pubKey, PrivateKey privKey) throws SecurityException {
        if (pubKey == null || privKey == null) {
            throw new SecurityException("Either public or private key was null");
        }
        SecurityConfiguration secConfig = Configuration.getGlobalSecurityConfiguration();
        if (secConfig == null) {
            throw new SecurityException("Global security configuration was null, could not resolve signing algorithm");
        }
        String algoURI = secConfig.getSignatureAlgorithmURI(privKey.getAlgorithm());
        if (algoURI == null) {
            throw new SecurityException("Can't determine algorithm URI from key algorithm: " + privKey.getAlgorithm());
        }
        String jcaAlgoID = SecurityHelper.getAlgorithmIDFromURI(algoURI);
        if (jcaAlgoID == null) {
            throw new SecurityException("Can't determine JCA algorithm ID from algorithm URI: " + algoURI);
        }
        if (log.isDebugEnabled()) {
            log.debug("Attempting to match key pair containing key algorithms public '{}' private '{}', using JCA signature algorithm '{}'", new Object[]{pubKey.getAlgorithm(), privKey.getAlgorithm(), jcaAlgoID});
        }
        byte[] data = "This is the data to sign".getBytes();
        byte[] signature = SigningUtil.sign(privKey, jcaAlgoID, data);
        return SigningUtil.verify(pubKey, jcaAlgoID, signature, data);
    }

    public static void prepareSignatureParams(Signature signature, Credential signingCredential, SecurityConfiguration config, String keyInfoGenName) throws SecurityException {
        SecurityConfiguration secConfig = config != null ? config : Configuration.getGlobalSecurityConfiguration();
        String signAlgo = signature.getSignatureAlgorithm();
        if (signAlgo == null) {
            signAlgo = secConfig.getSignatureAlgorithmURI(signingCredential);
            signature.setSignatureAlgorithm(signAlgo);
        }
        if (SecurityHelper.isHMAC(signAlgo) && signature.getHMACOutputLength() == null) {
            signature.setHMACOutputLength(secConfig.getSignatureHMACOutputLength());
        }
        if (signature.getCanonicalizationAlgorithm() == null) {
            signature.setCanonicalizationAlgorithm(secConfig.getSignatureCanonicalizationAlgorithm());
        }
        if (signature.getKeyInfo() == null) {
            KeyInfoGenerator kiGenerator = SecurityHelper.getKeyInfoGenerator(signingCredential, secConfig, keyInfoGenName);
            if (kiGenerator != null) {
                try {
                    KeyInfo keyInfo = kiGenerator.generate(signingCredential);
                    signature.setKeyInfo(keyInfo);
                }
                catch (SecurityException e) {
                    log.error("Error generating KeyInfo from credential", (Throwable)e);
                    throw e;
                }
            } else {
                log.info("No factory for named KeyInfoGenerator {} was found for credential type {}", (Object)keyInfoGenName, (Object)signingCredential.getCredentialType().getName());
                log.info("No KeyInfo will be generated for Signature");
            }
        }
    }

    public static EncryptionParameters buildDataEncryptionParams(Credential encryptionCredential, SecurityConfiguration config, String keyInfoGenName) {
        SecurityConfiguration secConfig = config != null ? config : Configuration.getGlobalSecurityConfiguration();
        EncryptionParameters encParams = new EncryptionParameters();
        encParams.setEncryptionCredential(encryptionCredential);
        if (encryptionCredential == null) {
            encParams.setAlgorithm(secConfig.getAutoGeneratedDataEncryptionKeyAlgorithmURI());
        } else {
            encParams.setAlgorithm(secConfig.getDataEncryptionAlgorithmURI(encryptionCredential));
            KeyInfoGenerator kiGenerator = SecurityHelper.getKeyInfoGenerator(encryptionCredential, secConfig, keyInfoGenName);
            if (kiGenerator != null) {
                encParams.setKeyInfoGenerator(kiGenerator);
            } else {
                log.info("No factory for named KeyInfoGenerator {} was found for credential type{}", (Object)keyInfoGenName, (Object)encryptionCredential.getCredentialType().getName());
                log.info("No KeyInfo will be generated for EncryptedData");
            }
        }
        return encParams;
    }

    public static KeyEncryptionParameters buildKeyEncryptionParams(Credential encryptionCredential, String wrappedKeyAlgorithm, SecurityConfiguration config, String keyInfoGenName, String recipient) throws SecurityException {
        SecurityConfiguration secConfig = config != null ? config : Configuration.getGlobalSecurityConfiguration();
        KeyEncryptionParameters kekParams = new KeyEncryptionParameters();
        kekParams.setEncryptionCredential(encryptionCredential);
        if (encryptionCredential == null) {
            throw new SecurityException("Key encryption credential may not be null");
        }
        kekParams.setAlgorithm(secConfig.getKeyTransportEncryptionAlgorithmURI(encryptionCredential, wrappedKeyAlgorithm));
        KeyInfoGenerator kiGenerator = SecurityHelper.getKeyInfoGenerator(encryptionCredential, secConfig, keyInfoGenName);
        if (kiGenerator != null) {
            kekParams.setKeyInfoGenerator(kiGenerator);
        } else {
            log.info("No factory for named KeyInfoGenerator {} was found for credential type {}", (Object)keyInfoGenName, (Object)encryptionCredential.getCredentialType().getName());
            log.info("No KeyInfo will be generated for EncryptedKey");
        }
        kekParams.setRecipient(recipient);
        return kekParams;
    }

    public static KeyInfoGenerator getKeyInfoGenerator(Credential credential, SecurityConfiguration config, String keyInfoGenName) {
        SecurityConfiguration secConfig = config != null ? config : Configuration.getGlobalSecurityConfiguration();
        NamedKeyInfoGeneratorManager kiMgr = secConfig.getKeyInfoGeneratorManager();
        if (kiMgr != null) {
            KeyInfoGeneratorFactory kiFactory = null;
            kiFactory = DatatypeHelper.isEmpty(keyInfoGenName) ? kiMgr.getDefaultManager().getFactory(credential) : kiMgr.getFactory(keyInfoGenName, credential);
            if (kiFactory != null) {
                return kiFactory.newInstance();
            }
        }
        return null;
    }
}

