//$Id: MassIndexTest.java 13946 2007-08-21 18:42:37Z epbernard $
package org.hibernate.search.test.session;

import java.util.List;
import java.util.Iterator;

import org.apache.lucene.analysis.StopAnalyzer;
import org.apache.lucene.index.Term;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.hibernate.ScrollMode;
import org.hibernate.ScrollableResults;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.search.Environment;
import org.hibernate.search.FullTextSession;
import org.hibernate.search.Search;
import org.hibernate.search.impl.FullTextSessionImpl;
import org.hibernate.search.test.SearchTestCase;

/**
 * @author Emmanuel Bernard
 */
public class MassIndexTest extends SearchTestCase {

	public void testBatchSize() throws Exception {
		FullTextSession s = Search.createFullTextSession( openSession() );
		Transaction tx = s.beginTransaction();
		int loop = 14;
		for (int i = 0; i < loop; i++) {
			s.connection().createStatement().executeUpdate( "insert into Email(id, title, body, header) values( + "
					+ ( i + 1 ) + ", 'Bob Sponge', 'Meet the guys who create the software', 'nope')" );
		}
		tx.commit();
		s.close();

		//check non created object does get found!!1
		s = new FullTextSessionImpl( openSession() );
		tx = s.beginTransaction();
		ScrollableResults results = s.createCriteria( Email.class ).scroll( ScrollMode.FORWARD_ONLY );
		int index = 0;
		while ( results.next() ) {
			index++;
			s.index( results.get( 0 ) );
			if ( index % 5 == 0 ) s.clear();
		}
		tx.commit();
		s.clear();
		tx = s.beginTransaction();
		QueryParser parser = new QueryParser( "id", new StopAnalyzer() );
		List result = s.createFullTextQuery( parser.parse( "body:create" ) ).list();
		assertEquals( 14, result.size() );
		for (Object object : result) {
			s.delete( object );
		}
		tx.commit();
		s.close();
	}

	public void testTransactional() throws Exception {
		FullTextSession s = Search.createFullTextSession( openSession() );
		Transaction tx = s.beginTransaction();
		int loop = 4;
		for (int i = 0; i < loop; i++) {
			Email email = new Email();
			email.setId( (long) i + 1 );
			email.setTitle( "JBoss World Berlin" );
			email.setBody( "Meet the guys who wrote the software" );
			s.persist( email );
		}
		tx.commit();
		s.close();

		//check non created object does get found!!1
		s = new FullTextSessionImpl( openSession() );
		tx = s.beginTransaction();
		QueryParser parser = new QueryParser( "id", new StopAnalyzer() );
		List result = s.createFullTextQuery( parser.parse( "body:create" ) ).list();
		assertEquals( 0, result.size() );
		tx.commit();
		s.close();

		s = new FullTextSessionImpl( openSession() );
		s.getTransaction().begin();
		s.connection().createStatement().executeUpdate( "update Email set body='Meet the guys who write the software'" );
		//insert an object never indexed
		s.connection().createStatement().executeUpdate( "insert into Email(id, title, body, header) values( + "
				+ ( loop + 1 ) + ", 'Bob Sponge', 'Meet the guys who create the software', 'nope')" );
		s.getTransaction().commit();
		s.close();

		s = new FullTextSessionImpl( openSession() );
		tx = s.beginTransaction();
		parser = new QueryParser( "id", new StopAnalyzer() );
		result = s.createFullTextQuery( parser.parse( "body:write" ) ).list();
		assertEquals( 0, result.size() );
		result = s.createCriteria( Email.class ).list();
		for (int i = 0; i < loop / 2; i++)
			s.index( result.get( i ) );
		tx.commit(); //do the process
		s.index( result.get( loop / 2 ) ); //do the process out of tx
		tx = s.beginTransaction();
		for (int i = loop / 2 + 1; i < loop; i++)
			s.index( result.get( i ) );
		tx.commit(); //do the process
		s.close();

		s = Search.createFullTextSession( openSession() );
		tx = s.beginTransaction();
		//object never indexed
		Email email = (Email) s.get( Email.class, new Long( loop + 1 ) );
		s.index( email );
		tx.commit();
		s.close();

		//check non indexed object get indexed by s.index
		s = new FullTextSessionImpl( openSession() );
		tx = s.beginTransaction();
		result = s.createFullTextQuery( parser.parse( "body:create" ) ).list();
		assertEquals( 1, result.size() );
		tx.commit();
		s.close();
	}

	public void testLazyLoading() throws Exception {
		Categorie cat = new Categorie( "Livre" );
		Entite ent = new Entite( "Le temple des songes", cat );
		Session s = openSession();
		Transaction tx = s.beginTransaction();
		s.persist( cat );
		s.persist( ent );
		tx.commit();
		s.close();

		s = openSession();
		FullTextSession session = Search.createFullTextSession( s );
		Query luceneQuery = new TermQuery( new Term( "categorie.nom", "livre" ) );
		List result = session.createFullTextQuery( luceneQuery, Entite.class ).list();
		assertEquals( 1, result.size() );
		s.close();

		s = openSession();
		ent = (Entite) s.get( Entite.class, ent.getId() );
		session = Search.createFullTextSession( s );
		session.index( ent );
		s.close();

		s = openSession();
		session = Search.createFullTextSession( s );
		luceneQuery = new TermQuery( new Term( "categorie.nom", "livre" ) );
		result = session.createFullTextQuery( luceneQuery, Entite.class ).list();
		assertEquals( "test lazy loading and indexing", 1, result.size() );
		s.close();

		s = openSession();
		Iterator it = s.createQuery( "from Entite where id = :id").setParameter( "id", ent.getId() ).iterate();
		session = Search.createFullTextSession( s );
		while ( it.hasNext() ) {
			ent = (Entite) it.next();
			session.index( ent );
		}
		s.close();

		s = openSession();
		session = Search.createFullTextSession( s );
		luceneQuery = new TermQuery( new Term( "categorie.nom", "livre" ) );
		result = session.createFullTextQuery( luceneQuery, Entite.class ).list();
		assertEquals( "test lazy loading and indexing", 1, result.size() );
		ent = (Entite) result.get( 0 );
		cat = ent.getCategorie();
		ent.setCategorie( null );
		session.delete( cat );
		session.delete( ent );
		s.close();
	}

	protected void configure(org.hibernate.cfg.Configuration cfg) {
		super.configure( cfg );
		cfg.setProperty( "hibernate.search.worker.batch_size", "5" );
		cfg.setProperty( Environment.ANALYZER_CLASS, StopAnalyzer.class.getName() );
	}

	protected Class[] getMappings() {
		return new Class[] {
				Email.class,
				Entite.class,
				Categorie.class
		};
	}
}
