/*
 * Decompiled with CFR 0.152.
 */
package io.agroal.pool;

import io.agroal.api.configuration.AgroalConnectionPoolConfiguration;
import io.agroal.api.configuration.InterruptProtection;
import io.agroal.api.transaction.TransactionIntegration;
import io.agroal.pool.ConnectionFactory;
import io.agroal.pool.ConnectionHandler;
import io.agroal.pool.ConnectionWrapper;
import io.agroal.pool.DataSource;
import io.agroal.pool.util.AgroalDataSourceListenerHelper;
import io.agroal.pool.util.AgroalSynchronizer;
import io.agroal.pool.util.PriorityScheduledExecutor;
import io.agroal.pool.util.StampedCopyOnWriteArrayList;
import io.agroal.pool.util.UncheckedArrayList;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

public class ConnectionPool
implements AutoCloseable {
    private final AgroalConnectionPoolConfiguration configuration;
    private final DataSource dataSource;
    private final ThreadLocal<UncheckedArrayList<ConnectionHandler>> localCache;
    private final StampedCopyOnWriteArrayList<ConnectionHandler> allConnections;
    private final AgroalSynchronizer synchronizer = new AgroalSynchronizer();
    private final ConnectionFactory connectionFactory;
    private final PriorityScheduledExecutor housekeepingExecutor;
    private final InterruptProtection interruptProtection;
    private final TransactionIntegration transactionIntegration;
    private final boolean leakEnabled;
    private final boolean validationEnable;
    private final boolean reapEnable;
    private volatile long maxUsed = 0L;

    public ConnectionPool(AgroalConnectionPoolConfiguration configuration, DataSource dataSource) {
        this.configuration = configuration;
        this.dataSource = dataSource;
        this.allConnections = new StampedCopyOnWriteArrayList<ConnectionHandler>(ConnectionHandler.class);
        this.localCache = ThreadLocal.withInitial(() -> new UncheckedArrayList(ConnectionHandler.class));
        this.connectionFactory = new ConnectionFactory(configuration.connectionFactoryConfiguration());
        this.housekeepingExecutor = new PriorityScheduledExecutor(1, "Agroal@" + System.identityHashCode(this) + "_");
        this.interruptProtection = configuration.connectionFactoryConfiguration().interruptProtection();
        this.transactionIntegration = configuration.transactionIntegration();
        this.leakEnabled = !configuration.leakTimeout().isZero();
        this.validationEnable = !configuration.validationTimeout().isZero();
        this.reapEnable = !configuration.reapTimeout().isZero();
    }

    public void init() {
        switch (this.configuration.preFillMode()) {
            default: {
                break;
            }
            case MIN: {
                this.fill(this.configuration.minSize());
                break;
            }
            case MAX: {
                this.fill(this.configuration.maxSize());
            }
        }
        if (this.leakEnabled) {
            this.housekeepingExecutor.schedule(new LeakTask(), this.configuration.leakTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }
        if (this.validationEnable) {
            this.housekeepingExecutor.schedule(new ValidationTask(), this.configuration.validationTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }
        if (this.reapEnable) {
            this.housekeepingExecutor.schedule(new ReapTask(), this.configuration.reapTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }
    }

    private void fill(int newSize) {
        long connectionCount = newSize - this.allConnections.size();
        while (connectionCount-- > 0L) {
            this.newConnectionHandler();
        }
    }

    @Override
    public void close() {
        this.housekeepingExecutor.shutdownNow();
    }

    private Future<?> newConnectionHandler() {
        return this.housekeepingExecutor.executeNow(() -> {
            if (this.allConnections.size() >= this.configuration.maxSize()) {
                return;
            }
            AgroalDataSourceListenerHelper.fireBeforeConnectionCreation(this.dataSource);
            long metricsStamp = this.dataSource.metricsRepository().beforeConnectionCreation();
            try {
                ConnectionHandler handler = new ConnectionHandler(this.connectionFactory.createConnection());
                handler.setConnectionPool(this);
                handler.setState(ConnectionHandler.State.CHECKED_IN);
                this.allConnections.add(handler);
                this.maxUsedCount();
                this.dataSource.metricsRepository().afterConnectionCreation(metricsStamp);
                AgroalDataSourceListenerHelper.fireOnConnectionCreation(this.dataSource, handler);
            }
            catch (SQLException e) {
                throw new RuntimeException(e);
            }
            finally {
                this.synchronizer.releaseConditional();
            }
        });
    }

    public Connection getConnection() throws SQLException {
        AgroalDataSourceListenerHelper.fireBeforeConnectionAcquire(this.dataSource);
        long metricsStamp = this.dataSource.metricsRepository().beforeConnectionAcquire();
        ConnectionHandler checkedOutHandler = null;
        ConnectionWrapper connectionWrapper = this.wrapperFromTransaction();
        if (connectionWrapper != null) {
            checkedOutHandler = connectionWrapper.getHandler();
        }
        if (checkedOutHandler == null) {
            checkedOutHandler = this.handlerFromLocalCache();
        }
        if (checkedOutHandler == null) {
            checkedOutHandler = this.handlerFromSharedCache();
        }
        this.dataSource.metricsRepository().afterConnectionAcquire(metricsStamp);
        AgroalDataSourceListenerHelper.fireOnConnectionAcquired(this.dataSource, checkedOutHandler);
        if (this.leakEnabled || this.reapEnable) {
            checkedOutHandler.setLastAccess(System.nanoTime());
        }
        if (this.leakEnabled) {
            checkedOutHandler.setHoldingThread(Thread.currentThread());
        }
        if (connectionWrapper == null) {
            connectionWrapper = new ConnectionWrapper(checkedOutHandler, this.interruptProtection);
            this.transactionIntegration.associate((Connection)connectionWrapper);
        }
        return connectionWrapper;
    }

    private ConnectionWrapper wrapperFromTransaction() throws SQLException {
        Connection connection = this.transactionIntegration.getConnection();
        if (connection != null) {
            return (ConnectionWrapper)connection;
        }
        return null;
    }

    private ConnectionHandler handlerFromLocalCache() throws SQLException {
        UncheckedArrayList<ConnectionHandler> cachedConnections = this.localCache.get();
        while (!cachedConnections.isEmpty()) {
            ConnectionHandler handler = cachedConnections.removeLast();
            if (!handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.CHECKED_OUT)) continue;
            return handler;
        }
        return null;
    }

    /*
     * Unable to fully structure code
     */
    private ConnectionHandler handlerFromSharedCache() throws SQLException {
        remaining = this.configuration.acquisitionTimeout().toNanos();
        remaining = remaining > 0L ? remaining : 0x7FFFFFFFFFFFFFFFL;
        try {
            while (true) lbl-1000:
            // 4 sources

            {
                for (ConnectionHandler handler : this.allConnections.getUnderlyingArray()) {
                    if (!handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.CHECKED_OUT)) continue;
                    return handler;
                }
                if (this.allConnections.size() < this.configuration.maxSize()) {
                    this.newConnectionHandler().get();
                    ** continue;
                }
                synchronizationStamp = this.synchronizer.getStamp();
                start = System.nanoTime();
                if (remaining < 0L || !this.synchronizer.tryAcquireNanos(synchronizationStamp, remaining)) {
                    throw new SQLException("Sorry, acquisition timeout!");
                }
                remaining -= System.nanoTime() - start;
            }
        }
        catch (InterruptedException e) {
            throw new SQLException("Interrupted while acquiring");
        }
        catch (ExecutionException e) {
            throw new SQLException("Exception while creating new connection", e);
        }
    }

    public void returnConnection(ConnectionHandler handler) throws SQLException {
        AgroalDataSourceListenerHelper.fireBeforeConnectionReturn(this.dataSource, handler);
        if (this.leakEnabled) {
            handler.setHoldingThread(null);
        }
        if (this.reapEnable) {
            handler.setLastAccess(System.nanoTime());
        }
        if (this.transactionIntegration.disassociate(handler.getConnection())) {
            this.localCache.get().add(handler);
            handler.setState(ConnectionHandler.State.CHECKED_IN);
            this.synchronizer.releaseConditional();
            this.dataSource.metricsRepository().afterConnectionReturn();
            AgroalDataSourceListenerHelper.fireOnConnectionReturn(this.dataSource, handler);
        }
    }

    private void closeConnectionSafely(ConnectionHandler handler) {
        try {
            AgroalDataSourceListenerHelper.fireBeforeConnectionDestroy(this.dataSource, handler);
            handler.closeConnection();
            AgroalDataSourceListenerHelper.fireOnConnectionDestroy(this.dataSource, handler);
        }
        catch (SQLException e) {
            AgroalDataSourceListenerHelper.fireOnWarning(this.dataSource, e);
        }
    }

    private long activeCount(ConnectionHandler[] handlers) {
        int l = 0;
        for (ConnectionHandler handler : handlers) {
            if (!handler.isActive()) continue;
            ++l;
        }
        return l;
    }

    public long activeCount() {
        return this.activeCount(this.allConnections.getUnderlyingArray());
    }

    public long availableCount() {
        ConnectionHandler[] handlers = this.allConnections.getUnderlyingArray();
        return (long)handlers.length - this.activeCount(handlers);
    }

    public long maxUsedCount() {
        this.maxUsed = Math.max(this.maxUsed, (long)this.allConnections.size());
        return this.maxUsed;
    }

    public void resetMaxUsedCount() {
        this.maxUsed = 0L;
    }

    public long awaitingCount() {
        return this.synchronizer.getQueueLength();
    }

    private class DestroyConnectionTask
    implements Runnable {
        private ConnectionHandler handler;

        public DestroyConnectionTask(ConnectionHandler handler) {
            this.handler = handler;
        }

        @Override
        public void run() {
            AgroalDataSourceListenerHelper.fireBeforeConnectionDestroy(ConnectionPool.this.dataSource, this.handler);
            ConnectionPool.this.closeConnectionSafely(this.handler);
            this.handler.setState(ConnectionHandler.State.DESTROYED);
            AgroalDataSourceListenerHelper.fireOnConnectionDestroy(ConnectionPool.this.dataSource, this.handler);
        }
    }

    private class ReapTask
    implements Runnable {
        private ReapTask() {
        }

        @Override
        public void run() {
            ConnectionPool.this.allConnections.forEach(hadler -> ConnectionPool.this.housekeepingExecutor.submit(new ReapConnectionTask((ConnectionHandler)hadler)));
            ConnectionPool.this.housekeepingExecutor.schedule(this, ConnectionPool.this.configuration.reapTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }

        private class ReapConnectionTask
        implements Runnable {
            private ConnectionHandler handler;

            public ReapConnectionTask(ConnectionHandler handler) {
                this.handler = handler;
            }

            @Override
            public void run() {
                AgroalDataSourceListenerHelper.fireBeforeConnectionReap(ConnectionPool.this.dataSource, this.handler);
                if (ConnectionPool.this.allConnections.size() > ConnectionPool.this.configuration.minSize() && this.handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.FLUSH)) {
                    if (System.nanoTime() - this.handler.getLastAccess() > ConnectionPool.this.configuration.reapTimeout().toNanos()) {
                        ConnectionPool.this.allConnections.remove(this.handler);
                        ConnectionPool.this.dataSource.metricsRepository().afterConnectionReap();
                        AgroalDataSourceListenerHelper.fireOnConnectionReap(ConnectionPool.this.dataSource, this.handler);
                        ConnectionPool.this.housekeepingExecutor.execute(new DestroyConnectionTask(this.handler));
                    } else {
                        this.handler.setState(ConnectionHandler.State.CHECKED_IN);
                    }
                }
            }
        }
    }

    private class ValidationTask
    implements Runnable {
        private ValidationTask() {
        }

        @Override
        public void run() {
            ConnectionPool.this.allConnections.forEach(hadler -> ConnectionPool.this.housekeepingExecutor.submit(new ValidateConnectionTask((ConnectionHandler)hadler)));
            ConnectionPool.this.housekeepingExecutor.schedule(this, ConnectionPool.this.configuration.validationTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }

        private class ValidateConnectionTask
        implements Runnable {
            private ConnectionHandler handler;

            public ValidateConnectionTask(ConnectionHandler handler) {
                this.handler = handler;
            }

            @Override
            public void run() {
                AgroalDataSourceListenerHelper.fireBeforeConnectionValidation(ConnectionPool.this.dataSource, this.handler);
                if (this.handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.VALIDATION)) {
                    if (ConnectionPool.this.configuration.connectionValidator().isValid(this.handler.getConnection())) {
                        this.handler.setState(ConnectionHandler.State.CHECKED_IN);
                        AgroalDataSourceListenerHelper.fireOnConnectionValid(ConnectionPool.this.dataSource, this.handler);
                    } else {
                        this.handler.setState(ConnectionHandler.State.FLUSH);
                        ConnectionPool.this.allConnections.remove(this.handler);
                        ConnectionPool.this.dataSource.metricsRepository().afterConnectionInvalid();
                        AgroalDataSourceListenerHelper.fireOnConnectionInvalid(ConnectionPool.this.dataSource, this.handler);
                        ConnectionPool.this.housekeepingExecutor.execute(new DestroyConnectionTask(this.handler));
                    }
                }
            }
        }
    }

    private class LeakTask
    implements Runnable {
        private LeakTask() {
        }

        @Override
        public void run() {
            for (ConnectionHandler handler : (ConnectionHandler[])ConnectionPool.this.allConnections.getUnderlyingArray()) {
                ConnectionPool.this.housekeepingExecutor.submit(new LeakConnectionTask(handler));
            }
            ConnectionPool.this.housekeepingExecutor.schedule(this, ConnectionPool.this.configuration.leakTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }

        private class LeakConnectionTask
        implements Runnable {
            private ConnectionHandler handler;

            public LeakConnectionTask(ConnectionHandler handler) {
                this.handler = handler;
            }

            @Override
            public void run() {
                AgroalDataSourceListenerHelper.fireBeforeConnectionLeak(ConnectionPool.this.dataSource, this.handler);
                Thread thread = this.handler.getHoldingThread();
                if (thread != null && System.nanoTime() - this.handler.getLastAccess() > ConnectionPool.this.configuration.leakTimeout().toNanos()) {
                    ConnectionPool.this.dataSource.metricsRepository().afterLeakDetection();
                    AgroalDataSourceListenerHelper.fireOnConnectionLeak(ConnectionPool.this.dataSource, this.handler);
                }
            }
        }
    }
}

