/*
 * Copyright 2013 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Eclipse Public License version 1.0, available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package io.escalante.artifact.maven

import org.sonatype.aether.repository.LocalRepository
import org.sonatype.aether.util.artifact.DefaultArtifact
import org.sonatype.aether.resolution.DependencyRequest
import org.sonatype.aether.collection.CollectRequest
import org.sonatype.aether.{RepositorySystemSession, RepositorySystem}
import collection.JavaConversions._
import org.sonatype.aether.graph.{DependencyFilter, Dependency}
import java.util.Collections
import org.apache.maven.repository.internal.{MavenServiceLocator, MavenRepositorySystemSession}
import org.sonatype.aether.spi.connector.RepositoryConnectorFactory
import org.sonatype.aether.connector.wagon.{WagonRepositoryConnectorFactory, WagonProvider}
import java.io._
import io.escalante.util.classload.ClassLoading

/**
 * Maven dependency resolver.
 *
 * @author Galder Zamarreño
 * @since 1.0
 */
object MavenDependencyResolver {

  private val DEFAULT_REPOSITORY_PATH =
    SecurityActions.getSystemProperty("user.home").concat("/.m2/repository")

  private lazy val SYSTEM = createSystem()

  private val SESSION = createSession()

  private val SETTINGS = new MavenSettings

  private lazy val SYSTEM_OUT = new PrintStream(new BufferedOutputStream(
    new FileOutputStream(FileDescriptor.out), 128), true)

  /**
   * Resolve a maven artifact into a collection of File instances.
   *
   * @param artifact MavenArtifact to resolve
   * @return a sequence of files
   */
  def resolveArtifact(artifact: MavenArtifact): Seq[File] =
    resolveArtifact(artifact, artifact.filter)

  /**
   * Resolve a maven artifact into a collection of File instances.
   *
   * @param artifact MavenArtifact to resolve
   * @param filter filter the dependencies of the artifact
   * @return a sequence of files
   */
  private def resolveArtifact(
      artifact: MavenArtifact,
      filter: Option[DependencyFilter]): Seq[File] = {
    val aetherArtifact = new DefaultArtifact(artifact.coordinates)
    val dependency = new Dependency(aetherArtifact, null)

    val request = new CollectRequest(Collections.singletonList(dependency),
      null, SETTINGS.getRemoteRepositories)

    val results = SYSTEM.resolveDependencies(SESSION,
      new DependencyRequest(request, filter.getOrElse(null))).getArtifactResults

    asScalaIterator(results.iterator())
      .filter(_.getArtifact.getExtension != "pom")
      .map(_.getArtifact.getFile).toSeq
  }

  private def createSystem(): RepositorySystem = {
    val cl = this.getClass.getClassLoader
    ClassLoading.withContextClassLoader[RepositorySystem](cl) {
      // Without ioc, yayyy!
      val locator = new MavenServiceLocator()
      locator.setServices(classOf[WagonProvider], ManualWagonProvider)
      locator.addService(classOf[RepositoryConnectorFactory],
        classOf[WagonRepositoryConnectorFactory])
      locator.getService(classOf[RepositorySystem])
    }
  }

  private def createSession(): RepositorySystemSession = {
    val localRepo = SYSTEM.newLocalRepositoryManager(
      new LocalRepository(DEFAULT_REPOSITORY_PATH))
    val session = new MavenRepositorySystemSession()
    session.setLocalRepositoryManager(localRepo)
    session.setTransferListener(new ConsoleTransferLog(SYSTEM_OUT))
    session
  }

}
