/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.test.remoting.classloader.currentthread;

import java.net.InetAddress;
import java.util.HashMap;
import java.util.Map;

import javax.management.MBeanServer;

import junit.framework.TestCase;

import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.jboss.logging.XLevel;
import org.jboss.remoting.Client;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.Remoting;
import org.jboss.remoting.ServerInvocationHandler;
import org.jboss.remoting.ServerInvoker;
import org.jboss.remoting.callback.InvokerCallbackHandler;
import org.jboss.remoting.transport.Connector;
import org.jboss.remoting.transport.PortUtil;


/**
 * Unit tests for JBREM-1127
 * 
 * @author <a href="ron.sigal@jboss.com">Ron Sigal</a>
 * @version 
 * <p>
 * Copyright May 10, 2009
 * </p>
 */
public class CurrentThreadContextClassLoaderTestCase extends TestCase
{
   private static Logger log = Logger.getLogger(CurrentThreadContextClassLoaderTestCase.class);
   
   private static boolean firstTime = true;
   
   protected String host;
   protected int port;
   protected String locatorURI;
   protected InvokerLocator serverLocator;
   protected Connector connector;
   protected TestInvocationHandler invocationHandler;
   protected String currentPackage;
   protected ClassLoader mainContextClassLoader;
   protected TestClassLoader testClassLoader1;
   protected TestClassLoader testClassLoader2;
   protected TestCallingThread testCallingThread;

   
   public void setUp() throws Exception
   {
      if (firstTime)
      {
         firstTime = false;
         Logger.getLogger("org.jboss.remoting").setLevel(XLevel.INFO);
         Logger.getLogger("org.jboss.test.remoting").setLevel(Level.INFO);
         String pattern = "[%d{ABSOLUTE}] [%t] %5p (%F:%L) - %m%n";
         PatternLayout layout = new PatternLayout(pattern);
         ConsoleAppender consoleAppender = new ConsoleAppender(layout);
         Logger.getRootLogger().addAppender(consoleAppender);         
      }
      
      String className = getClass().getName();
      int index = className.lastIndexOf('.');
      currentPackage = className.substring(0, index);
   }

   
   public void tearDown()
   {
   }
   
   
   public void testDefault() throws Throwable
   {
      log.info("entering " + getName());
      
      // Start server.
      setupServer();
      
      mainContextClassLoader = Thread.currentThread().getContextClassLoader();
      
      // Create testClassLoader1 for testCallingThread.
      testClassLoader1 = new TestClassLoader("testClassLoader1");
      
      // Create testClassLoader2 for main thread.
      testClassLoader2 = new TestClassLoader("testClassLoader2");
      
      // Create client.
      InvokerLocator clientLocator = new InvokerLocator(locatorURI);
      HashMap clientConfig = new HashMap();
      clientConfig.put(InvokerLocator.FORCE_REMOTE, "true");
      addExtraClientConfig(clientConfig);
      Client client = new Client(clientLocator, clientConfig);
      client.connect();
      log.info("client is connected");
      
      // Make invocation, creating unmarshaller with testClassLoader1, in separate thread.
      TestCallingThread t = new TestCallingThread(client);
      t.run();
      synchronized (t)
      {
         if (!t.done)
         {
            t.wait();
         }
      }
      
      // Make invocation in main thread with context classloader testClassLoader2.
      testClassLoader2.setClassLoader(mainContextClassLoader);
      Thread.currentThread().setContextClassLoader(testClassLoader2);
      log.info("main thread result: " + client.invoke("2"));
      assertTrue(testClassLoader1.counter > 0);
      assertTrue(testClassLoader2.counter == 0);
      
      client.disconnect();
      shutdownServer();
      log.info(getName() + " PASSES");
   }
   
   
   public void testUseCurrentThreadClassLoaderFalse() throws Throwable
   {
      log.info("entering " + getName());
      
      // Start server.
      setupServer();
      
      mainContextClassLoader = Thread.currentThread().getContextClassLoader();
      
      // Create testClassLoader1 for testCallingThread.
      testClassLoader1 = new TestClassLoader("testClassLoader1");
      
      // Create testClassLoader2 for main thread.
      testClassLoader2 = new TestClassLoader("testClassLoader2");
      
      // Create client.
      InvokerLocator clientLocator = new InvokerLocator(locatorURI);
      HashMap clientConfig = new HashMap();
      clientConfig.put(InvokerLocator.FORCE_REMOTE, "true");
      clientConfig.put(Remoting.USE_CURRENT_THREAD_CLASS_LOADER, "false");
      addExtraClientConfig(clientConfig);
      Client client = new Client(clientLocator, clientConfig);
      client.connect();
      log.info("client is connected");
      
      // Make invocation, creating unmarshaller with testClassLoader1, in separate thread.
      TestCallingThread t = new TestCallingThread(client);
      t.run();
      synchronized (t)
      {
         if (!t.done)
         {
            t.wait();
         }
      }
      
      // Make invocation in main thread with context classloader testClassLoader2.
      testClassLoader2.setClassLoader(mainContextClassLoader);
      Thread.currentThread().setContextClassLoader(testClassLoader2);
      log.info("main thread result: " + client.invoke("2"));
      assertTrue(testClassLoader1.counter > 0);
      assertTrue(testClassLoader2.counter == 0);
      
      client.disconnect();
      shutdownServer();
      log.info(getName() + " PASSES");
   }
   
   
   public void testUseCurrentThreadClassLoaderTrue() throws Throwable
   {
      log.info("entering " + getName());
      
      // Start server.
      setupServer();
      
      mainContextClassLoader = Thread.currentThread().getContextClassLoader();
      
      // Create testClassLoader1 for testCallingThread.
      testClassLoader1 = new TestClassLoader("testClassLoader1");
      
      // Create testClassLoader2 for main thread.
      testClassLoader2 = new TestClassLoader("testClassLoader2");
      
      // Create client.
      InvokerLocator clientLocator = new InvokerLocator(locatorURI);
      HashMap clientConfig = new HashMap();
      clientConfig.put(InvokerLocator.FORCE_REMOTE, "true");
      clientConfig.put(Remoting.USE_CURRENT_THREAD_CLASS_LOADER, "true");
      addExtraClientConfig(clientConfig);
      Client client = new Client(clientLocator, clientConfig);
      client.connect();
      log.info("client is connected");
      
      // Make invocation, creating unmarshaller with testClassLoader1, in separate thread.
      TestCallingThread t = new TestCallingThread(client);
      t.run();
      synchronized (t)
      {
         if (!t.done)
         {
            t.wait();
         }
      }
      
      // Make invocation in main thread with context classloader testClassLoader2.
      testClassLoader2.setClassLoader(mainContextClassLoader);
      Thread.currentThread().setContextClassLoader(testClassLoader2);
      log.info("main thread result: " + client.invoke("2"));
      assertTrue(testClassLoader1.counter > 0);
      assertTrue(testClassLoader2.counter > 0);
      
      client.disconnect();
      shutdownServer();
      log.info(getName() + " PASSES");
   }
   
   
   protected String getTransport()
   {
      return "socket";
   }
   
   
   protected void addExtraClientConfig(Map config) {}
   protected void addExtraServerConfig(Map config) {}
   

   protected void setupServer() throws Exception
   {
      host = InetAddress.getLocalHost().getHostAddress();
      port = PortUtil.findFreePort(host);
      locatorURI = getTransport() + "://" + host + ":" + port;
      locatorURI += "/?" + Remoting.CLASSLOADING_PARENT_FIRST_DELEGATION + "=false";
      String metadata = System.getProperty("remoting.metadata");
      if (metadata != null)
      {
         locatorURI += "&" + metadata;
      }
      serverLocator = new InvokerLocator(locatorURI);
      log.info("Starting remoting server with locator uri of: " + locatorURI);
      HashMap config = new HashMap();
      config.put(InvokerLocator.FORCE_REMOTE, "true");
      addExtraServerConfig(config);
      connector = new Connector(serverLocator, config);
      connector.create();
      invocationHandler = new TestInvocationHandler();
      connector.addInvocationHandler("test", invocationHandler);
      connector.start();
   }
   
   
   protected void shutdownServer() throws Exception
   {
      if (connector != null)
         connector.stop();
   }
   
   
   class TestInvocationHandler implements ServerInvocationHandler
   {
      public void addListener(InvokerCallbackHandler callbackHandler) {}
      public Object invoke(final InvocationRequest invocation) throws Throwable
      {
         Object o = invocation.getParameter();
         if ("1".equals(o))
         {
            return Class.forName(currentPackage + ".TestClass1").newInstance();
         }
         else if ("2".equals(o))
         {
            return Class.forName(currentPackage + ".TestClass2").newInstance();         
         }
         else
         {
            throw new Exception("unexpected request: " + o);
         }
      }
      public void removeListener(InvokerCallbackHandler callbackHandler) {}
      public void setMBeanServer(MBeanServer server) {}
      public void setInvoker(ServerInvoker invoker) {}
   }
   
   
   class TestCallingThread extends Thread
   {
      Client client;
      boolean done;
      
      public TestCallingThread(Client client)
      {
         this.client = client;
      }
      
      public void run()
      {
         try
         {
            // Make testClassloader1 the context classloader for this TestCallingThread.
            testClassLoader1.setClassLoader(mainContextClassLoader);
            Thread.currentThread().setContextClassLoader(testClassLoader1);
            log.info("TestCallingThread result: " + client.invoke("1"));
            synchronized (this)
            {
               done = true;
               notifyAll();
            }
         }
         catch (Throwable e)
         {
            log.error("Error", e);
         }
      }
   }
   
   
   static class TestClassLoader extends ClassLoader
   {
      ClassLoader cl;
      String name;
      int counter;
      
      public TestClassLoader(String name)
      {
         this.name = name;
      }
      
      public Class loadClass(String name) throws ClassNotFoundException
      {
         counter++;
         log.info(this + " called: " + name);
//         log.info(this + " called: " + name, new Exception());
         return cl.loadClass(name);
      }
      
      public void setClassLoader(ClassLoader cl)
      {
         this.cl = cl;
      }
      
      public String toString()
      {
       return name;  
      }
   }
}