/*
 * JBoss, Home of Professional Open Source
 * Copyright 2009, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.test.remoting.lease;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.HashMap;
import java.util.Map;

import javax.management.MBeanServer;
import javax.net.ServerSocketFactory;

import junit.framework.TestCase;

import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.PatternLayout;
import org.jboss.logging.Logger;
import org.jboss.logging.Log4jLoggerPlugin;
import org.jboss.logging.XLevel;
import org.jboss.remoting.CannotConnectException;
import org.jboss.remoting.Client;
import org.jboss.remoting.ConnectionListener;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.Remoting;
import org.jboss.remoting.ServerInvocationHandler;
import org.jboss.remoting.ServerInvoker;
import org.jboss.remoting.callback.InvokerCallbackHandler;
import org.jboss.remoting.transport.Connector;
import org.jboss.remoting.transport.PortUtil;


/**
 * Unit test for JBREM-1238.
 * 
 * @author <a href="ron.sigal@jboss.com">Ron Sigal</a>
 * @version 
 * <p>
 * Copyright July 28, 2010
 * </p>
 */
public class LeaseCreationTimeoutTestCase extends TestCase
{
   private static Logger log = Logger.getLogger(LeaseCreationTimeoutTestCase.class);

   protected static int breakServer;
   protected static int counter;

   protected String host;
   protected int port;
   protected String locatorURI;
   protected InvokerLocator serverLocator;
   protected Connector connector;
   protected TestInvocationHandler invocationHandler;
   protected PrintStream originalPrintStream;
   protected ByteArrayOutputStream baos;


   public void setUp() throws Exception
   {
      baos = new ByteArrayOutputStream();
      originalPrintStream = System.out;
      PrintStream ps = new PrintStream(baos, true);
      System.setOut(ps);

      String pattern = "[%d{ABSOLUTE}] [%t] %5p (%F:%L) - %m%n";
      PatternLayout layout = new PatternLayout(pattern);
      ConsoleAppender consoleAppender = new ConsoleAppender(layout);
      consoleAppender.setTarget(ConsoleAppender.SYSTEM_OUT);

      Log4jLoggerPlugin loggerPlugin = (Log4jLoggerPlugin) Logger.getLogger("org.jboss.remoting").getLoggerPlugin();
      org.apache.log4j.Logger logger = loggerPlugin.getLogger();
      logger.setLevel(XLevel.TRACE);
      logger.addAppender(consoleAppender);
      loggerPlugin = (Log4jLoggerPlugin) Logger.getLogger("org.jboss.test.remoting").getLoggerPlugin();
      logger = loggerPlugin.getLogger();
      logger.setLevel(XLevel.TRACE);
      logger.addAppender(consoleAppender);

      breakServer = -1;
      counter = 0;
   }


   public void tearDown()
   {
   }


   /**
    * Verifies that default value of leaseCreationTimeout is set properly.
    */
   public void testLeaseCreationTimeoutDefault() throws Throwable
   {
      log.info("entering " + getName());
      Client client = null;

      try
      {
         // Start server.
         setupServer(-1);

         // Create client.
         InvokerLocator clientLocator = new InvokerLocator(locatorURI);
         HashMap clientConfig = new HashMap();
         clientConfig.put(InvokerLocator.FORCE_REMOTE, "true");
         clientConfig.put(Client.ENABLE_LEASE, "true");
         addExtraClientConfig(clientConfig);
         client = new Client(clientLocator, clientConfig);
         client.connect();
         log.info("client is connected");

         // Test connection.
         assertEquals("abc", client.invoke("abc"));
         log.info("connection is good");

         String s = baos.toString();
         assertTrue(s.indexOf("setting timeout to 30000 for this invocation") > -1);
         assertTrue(s.indexOf("starting lease timer with ping period of") > -1);
         log.info(getName() + " PASSES");
      }
      finally
      {
         client.disconnect();
         shutdownServer();
         originalPrintStream.flush();
         System.setOut(originalPrintStream);
         System.out.println("starting to write log");
         String s = baos.toString();
         System.out.println(s);
      }
   }


   /**
    * Verifies that value of leaseCreationTimeout is set properly from configuration map.
    */
   public void testLeaseCreationTimeoutConfig() throws Throwable
   {
      log.info("entering " + getName());
      Client client = null;

      try
      {
         // Start server.
         setupServer(-1);

         // Create client.
         InvokerLocator clientLocator = new InvokerLocator(locatorURI);
         HashMap clientConfig = new HashMap();
         clientConfig.put(InvokerLocator.FORCE_REMOTE, "true");
         clientConfig.put(Client.ENABLE_LEASE, "true");
         clientConfig.put(Remoting.LEASE_CREATION_TIMEOUT, "12345");
         clientConfig.put(ServerInvoker.TIMEOUT, "0");
         addExtraClientConfig(clientConfig);
         client = new Client(clientLocator, clientConfig);
         client.connect();
         log.info("client is connected");

         // Test connection.
         assertEquals("abc", client.invoke("abc"));
         log.info("connection is good");

         String s = baos.toString();
         assertTrue(s.indexOf("setting timeout to 12345 for this invocation") > -1);
         assertTrue(s.indexOf("starting lease timer with ping period of") > -1);
         log.info(getName() + " PASSES");
      }
      finally
      {
          client.disconnect();
          shutdownServer();
          originalPrintStream.flush();
          System.setOut(originalPrintStream);
          System.out.println("starting to write log");
          String s = baos.toString();
          System.out.println(s);
      }
   }


   /**
    * Verifies that MicroRemotingClientInvoker.establishLease() is terminated according to
    * the value of Remoting.LEASE_CREATION_TIMEOUT so that another new connection can
    * set up a lease.
    */
   public void testLeaseCreationWithBrokenEstablishLease() throws Throwable
   {
      log.info("entering " + getName());
      Client client1 = null;
      Client client2 = null;

      try
      {
         // Start server.
         setupServer(1);

         // Create client.
         InvokerLocator clientLocator = new InvokerLocator(locatorURI);
         HashMap clientConfig = new HashMap();
         clientConfig.put(InvokerLocator.FORCE_REMOTE, "true");
         clientConfig.put(ServerInvoker.TIMEOUT, "0");
         clientConfig.put(Client.ENABLE_LEASE, "true");
         clientConfig.put(Remoting.LEASE_CREATION_TIMEOUT, "5000");
         addExtraClientConfig(clientConfig);
         client1 = new Client(clientLocator, clientConfig);

         // Should not connect.
         BrokenEstablisthLeaseTestThread btt = new BrokenEstablisthLeaseTestThread(client1);
         btt.run();

         // Should connect.
         Thread.sleep(10000);
         client2 = new Client(clientLocator, clientConfig);
         WorkingTestThread ttt = new WorkingTestThread(client2);
         ttt.run();

         // Test lease creation.
         Thread.sleep(5000);
         assertTrue(btt.ok);
         assertTrue(ttt.ok);

         log.info(getName() + " PASSES");
      }
      finally
      {
          client1.disconnect();
          client2.disconnect();
          shutdownServer();
          originalPrintStream.flush();
          System.setOut(originalPrintStream);
          System.out.println("starting to write log");
          String s = baos.toString();
          System.out.println(s);
          Thread.sleep(10000);
      }
   }


   /**
    * Verifies that MicroRemotingClientInvoker.terminateLease() is terminated according to
    * the value of Remoting.LEASE_CREATION_TIMEOUT so that another new connection can
    * set up a lease.  In this test the hang up is in LeasePinger.removeClient().
    */
   public void testLeaseTerminationWithBrokenRemoveClient() throws Throwable
   {
      log.info("entering " + getName());
      Client client1 = null;
      Client client2 = null;

      try
      {
         // Start server.
         setupServer(4);

         // Create client.
         InvokerLocator clientLocator = new InvokerLocator(locatorURI);
         HashMap clientConfig = new HashMap();
         clientConfig.put(InvokerLocator.FORCE_REMOTE, "true");
         clientConfig.put(ServerInvoker.TIMEOUT, "0");
         clientConfig.put(Client.ENABLE_LEASE, "true");
         clientConfig.put(Remoting.LEASE_CREATION_TIMEOUT, "5000");
         addExtraClientConfig(clientConfig);
         client1 = new Client(clientLocator, clientConfig);

         // Should not terminate properly in call to LeasePinger.removeClient().
         BrokenTerminateLeaseRemoveClientTestThread btt = new BrokenTerminateLeaseRemoveClientTestThread(client1, baos);
         btt.start();

         // Should connect.
         Thread.sleep(10000);
         client2 = new Client(clientLocator, clientConfig);
         WorkingTestThread ttt = new WorkingTestThread(client2);
         ttt.start();

         // Test lease creation.
         Thread.sleep(5000);
         assertTrue(btt.ok);
         assertTrue(ttt.ok);

         log.info(getName() + " PASSES");
      }
      finally
      {
          shutdownServer();
          originalPrintStream.flush();
          System.setOut(originalPrintStream);
          System.out.println("starting to write log");
          String s = baos.toString();
          System.out.println(s);
          Thread.sleep(10000);
      }
   }

   
   /**
    * Verifies that MicroRemotingClientInvoker.terminateLease() is terminated according to
    * the value of Remoting.LEASE_CREATION_TIMEOUT so that another new connection can
    * set up a lease.  In this test the hang up is in LeasePinger.stopPing().
    */
   public void testLeaseTerminationWithBrokenStopPing() throws Throwable
   {
      log.info("entering " + getName());

      try
      {
         // Start server.
         setupServer(5);

         // Create client.
         InvokerLocator clientLocator = new InvokerLocator(locatorURI);
         HashMap clientConfig = new HashMap();
         clientConfig.put(InvokerLocator.FORCE_REMOTE, "true");
         clientConfig.put(ServerInvoker.TIMEOUT, "0");
         clientConfig.put(Client.ENABLE_LEASE, "true");
         clientConfig.put(Remoting.LEASE_CREATION_TIMEOUT, "5000");
         addExtraClientConfig(clientConfig);
         Client client1 = new Client(clientLocator, clientConfig);

         // Should not terminate properly in call to LeasePinger.stopPing().
         BrokenTerminateLeaseStopPingTestThread btt = new BrokenTerminateLeaseStopPingTestThread(client1, baos);
         btt.start();

         // Should connect.
         Thread.sleep(10000);
         Client client2 = new Client(clientLocator, clientConfig);
         WorkingTestThread ttt = new WorkingTestThread(client2);
         ttt.start();

         // Test lease creation.
         Thread.sleep(5000);
         assertTrue(btt.ok);
         assertTrue(ttt.ok);

         shutdownServer();
         log.info(getName() + " PASSES");

      }
      finally
      {
         originalPrintStream.flush();
         System.setOut(originalPrintStream);
         System.out.println("starting to write log");
         String s = baos.toString();
         System.out.println(s);
      }
   }
   
   
   protected String getTransport()
   {
      return "socket";
   }


   protected void addExtraClientConfig(Map config) {}
   protected void addExtraServerConfig(Map config) {}


   protected void setupServer(int breakServer) throws Exception
   {
      host = InetAddress.getLocalHost().getHostAddress();
      port = PortUtil.findFreePort(host);
      locatorURI = getTransport() + "://" + host + ":" + port;
      String metadata = System.getProperty("remoting.metadata");
      if (metadata != null)
      {
         locatorURI += "/?" + metadata;
      }
      serverLocator = new InvokerLocator(locatorURI);
      log.info("Starting remoting server with locator uri of: " + locatorURI);
      HashMap config = new HashMap();
      config.put(InvokerLocator.FORCE_REMOTE, "true");
      config.put(Remoting.USE_CLIENT_CONNECTION_IDENTITY, "true");
      config.put("clientLeasePeriod", "10000");
      if (breakServer > -1)
      {
         LeaseCreationTimeoutTestCase.breakServer = breakServer;
         config.put(Remoting.CUSTOM_SERVER_SOCKET_FACTORY, new TestServerSocketFactory());
      }
      addExtraServerConfig(config);
      connector = new Connector(serverLocator, config);
      connector.create();
      invocationHandler = new TestInvocationHandler();
      connector.addInvocationHandler("test", invocationHandler);
      connector.addConnectionListener(new TestConnectionListener());
      connector.start();
   }


   protected void shutdownServer() throws Exception
   {
      if (connector != null)
         connector.stop();
   }


   static class TestInvocationHandler implements ServerInvocationHandler
   {
      public void addListener(InvokerCallbackHandler callbackHandler) {}
      public Object invoke(final InvocationRequest invocation) throws Throwable
      {
         return invocation.getParameter();
      }
      public void removeListener(InvokerCallbackHandler callbackHandler) {}
      public void setMBeanServer(MBeanServer server) {}
      public void setInvoker(ServerInvoker invoker) {}
   }

   
   static class TestConnectionListener implements ConnectionListener
   {
      public void handleConnectionException(Throwable throwable, Client client)
      {
      }
   }

   
   static public class TestServerSocketFactory extends ServerSocketFactory
   {
      public TestServerSocketFactory()
      {
      }      
      public ServerSocket createServerSocket() throws IOException
      {
         ServerSocket ss = new TestServerSocket();
         log.info(this + " returning: " + ss);
         return ss;
      }
      public ServerSocket createServerSocket(int port) throws IOException
      {
         ServerSocket ss = new TestServerSocket(port);
         log.info(this + " returning: " + ss);
         return ss;
      }

      public ServerSocket createServerSocket(int port, int backlog) throws IOException
      {
         ServerSocket ss = new TestServerSocket(port, backlog);
         log.info(this + " returning: " + ss);
         return ss;
      }

      public ServerSocket createServerSocket(int port, int backlog, InetAddress ifAddress) throws IOException
      {
         ServerSocket ss = new TestServerSocket(port, backlog, ifAddress);
         log.info(this + " returning: " + ss);
         return ss;
      }
   }

   
   static class TestServerSocket extends ServerSocket
   {
      public TestServerSocket() throws IOException
      {
         super();
      }
      public TestServerSocket(int port) throws IOException
      {
         super(port);
      }
      public TestServerSocket(int port, int backlog) throws IOException
      {
         super(port, backlog);
      }
      public TestServerSocket(int port, int backlog, InetAddress bindAddr) throws IOException
      {
         super(port, backlog, bindAddr);
      }
      public Socket accept() throws IOException
      {
         Socket s = super.accept();
         TestSocket ts = new TestSocket(s);
         log.info(this + " returning: " + ts);
         return ts;
      }
      public String toString()
      {
         return "TestServerSocket[" + getLocalPort() + "]";
      }
   }

   
   static class TestSocket extends Socket
   {
      Socket s;

      public TestSocket(Socket s)
      {
         this.s = s;  
      }
      public void close() throws IOException
      {
         s.close();
      }
      public InputStream getInputStream() throws IOException
      {
         return s.getInputStream();
      }
      public OutputStream getOutputStream() throws IOException
      {
         return new TestOutputStream(s.getOutputStream());
      }
      public InetAddress getInetAddress()
      {
         return s.getInetAddress(); 
      }
      public String toString()
      {
         return "TestSocket[" + s.getLocalPort() + "->" + s.getPort() + "]";
      }
   }


   static class TestOutputStream extends OutputStream
   {
      OutputStream os;
      boolean closed;
      boolean doThrow = true;

      public TestOutputStream(OutputStream os)
      {
         this.os = os;
      }
      public void close()throws IOException
      {
         closed = true;
         super.close();
         log.info(this + " closed");
      }
      public void write(int b) throws IOException
      {
         System.out.print("b: " + b);
         if (closed)
         {
            log.info("TestOutputStream closed, cannot write");
            throw new IOException("closed");
         }
         if (doThrow && counter++ == breakServer)
         {
            log.info("sleeping");
            while (true)
            {
               try
               {
                  Thread.sleep(30000);
               }
               catch (InterruptedException e)
               {
               }
            }
         }
         os.write(b);
      }
      public void write(byte b[], int off, int len) throws IOException
      {
         System.out.print("b: ");
         for (int i = 0; i < len; i++)
         {
            System.out.print(b[i] + " ");
         }
         System.out.println("");
         if (closed)
         {
            log.info("TestOutputStream closed, cannot write");
            throw new IOException("closed");
         }
         log.info("TestOutputStream: counter = " + counter + ", broken = " + breakServer);
         if (counter++ == breakServer)
         {
            log.info(this + " sleeping");
            try
            {
               Thread.sleep(20000);
            }
            catch (InterruptedException e)
            {
            }
            log.info(this + " woke up");
         }
         if (closed)
         {
            log.info("TestOutputStream closed, cannot write");
            throw new IOException("closed");
         }
         try
         {
            log.info(this + " calling write()");
            doThrow = false;
            os.write(b, off, len);
            os.flush();
            doThrow = true;
            log.info(this + " back from write()");
         }
         catch (IOException e)
         {
            log.info("exception: ", e);
            throw e;
         }
      }
      public void flush() throws IOException
      {
         os.flush();
      }
   }


   static class BrokenEstablisthLeaseTestThread extends Thread
   {
      public boolean ok;
      Client client;

      public BrokenEstablisthLeaseTestThread(Client client)
      {
         this.client = client;
         setName("BrokenEstablisthLeaseTestThread[" + client + "]");
      }
      public void run()
      {
         try
         {
            log.info(this + " STARTING");
            log.info(this + " trying to connect " + client);
            client.connect();
         }
         catch (CannotConnectException e)
         {
            if ("Error setting up client lease upon performing connect.".equals(e.getMessage()))
            {
               ok = true;
               log.info(this + " got expected exception");
            }
            else
            {
               log.info(this + " got expected exception with unexpected message: " + e.getMessage());    
            }
         }
         catch (Exception e)
         {
            log.error(this + " got unexpected exception: " + e);
         }
      }
      public String toString()
      {
         return getName();
      }
   }


   static class BrokenTerminateLeaseRemoveClientTestThread extends Thread
   {
      public boolean ok;
      Client client;
      ByteArrayOutputStream baos;

      public BrokenTerminateLeaseRemoveClientTestThread(Client client, ByteArrayOutputStream baos)
      {
         this.client = client;
         this.baos = baos;
         setName("BrokenTerminateLeaseRemoveClientTestThread[" + client + "]");
      }
      public void run()
      {
         try
         {
            log.info(this + " STARTING");
            log.info(this + " trying to connect " + client);
            client.connect();
            assertEquals("def", client.invoke("def"));
            client.disconnect();
            String s = baos.toString();
            assertTrue(s.indexOf("Terminating individual lease for session id") > -1);
            assertTrue(s.indexOf("failed sending disconnect for client lease") > -1);
            ok = true;
         }
         catch (Throwable t)
         {
            log.error(this + " got unexpected exception: " + t);
         }
      }
      public String toString()
      {
         return getName();
      }
   }


   static class BrokenTerminateLeaseStopPingTestThread extends Thread
   {
      public boolean ok;
      Client client;
      ByteArrayOutputStream baos;

      public BrokenTerminateLeaseStopPingTestThread(Client client, ByteArrayOutputStream baos)
      {
         this.client = client;
         this.baos = baos;
         setName("BrokenTerminateLeaseStopPingTestThread[" + client + "]");
      }
      public void run()
      {
         try
         {
            log.info(this + " STARTING");
            log.info(this + " trying to connect " + client);
            client.connect();
            assertEquals("def", client.invoke("def"));
            client.disconnect();
            String s = baos.toString();
            assertTrue(s.indexOf("Terminating individual lease for session id") > -1);
            assertFalse(s.indexOf("failed sending disconnect for client lease") > -1);
            assertTrue(s.indexOf(": disconnectTimeout: 5000") > -1);
            assertTrue(s.indexOf("error shutting down lease pinger") > -1);
            ok = true;
         }
         catch (Throwable t)
         {
            log.error(this + " got unexpected exception: " + t);
         }
      }
      public String toString()
      {
         return getName();
      }
   }
   
   
   static class WorkingTestThread extends Thread
   {
      public boolean ok;
      Client client;

      public WorkingTestThread(Client client)
      {
         this.client = client;
         setName("WorkingTestThread[" + client + "]");
      }
      public void run()
      {
         log.info(this + " STARTING");
         try
         {
            log.info(this + " trying to connect " + client);
            client.connect();
            assertEquals("abc", client.invoke("abc"));
            log.info(this + " connection is good");
            ok = true;
         }
         catch (Throwable t)
         {
            log.error(this + " got unexpected exception: " + t);
         }
      }
      public String toString()
      {
         return getName();
      }
   }
}