/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.test.remoting.shutdown;

import java.net.InetAddress;
import java.util.HashMap;
import java.util.Map;

import javax.management.MBeanServer;

import junit.framework.TestCase;

import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.jboss.logging.XLevel;
import org.jboss.remoting.CannotConnectException;
import org.jboss.remoting.Client;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.Remoting;
import org.jboss.remoting.ServerInvocationHandler;
import org.jboss.remoting.ServerInvoker;
import org.jboss.remoting.callback.InvokerCallbackHandler;
import org.jboss.remoting.transport.Connector;
import org.jboss.remoting.transport.PortUtil;

/**
 * Unit test for JBREM-1073.
 * 
 * @author <a href="ron.sigal@jboss.com">Ron Sigal</a>
 * @version 
 * <p>
 * Copyright Dec 26, 2008
 * </p>
 */
public class ShutdownExceptionTestCase extends TestCase
{
   private static Logger log = Logger.getLogger(ShutdownExceptionTestCase.class);
   
   private static boolean firstTime = true;
   
   protected String host;
   protected int port;
   protected String locatorURI;
   protected InvokerLocator serverLocator;
   protected Connector connector;
   protected TestInvocationHandler invocationHandler;

   
   public void setUp() throws Exception
   {
      if (firstTime)
      {
         firstTime = false;
         Logger.getLogger("org.jboss.remoting").setLevel(XLevel.INFO);
         Logger.getLogger("org.jboss.test.remoting").setLevel(Level.INFO);
         String pattern = "[%d{ABSOLUTE}] [%t] %5p (%F:%L) - %m%n";
         PatternLayout layout = new PatternLayout(pattern);
         ConsoleAppender consoleAppender = new ConsoleAppender(layout);
         Logger.getRootLogger().addAppender(consoleAppender);  
      }
   }

   
   public void tearDown()
   {
   }
   
   
   public void testDefaultBehavior() throws Throwable
   {
      log.info("entering " + getName());
      HashMap config = new HashMap();
      doTestDefaultBehavior(config);
      log.info(getName() + " PASSES");
   }
   
   
   public void testChangeInvalidStateToCannotConnectFalse() throws Throwable
   {
      System.out.println("");
      log.info("entering " + getName());
      HashMap config = new HashMap();
      config.put(Remoting.CHANGE_INVALID_STATE_TO_CANNOT_CONNECT, "false");
      doTestDefaultBehavior(config);
      log.info(getName() + " PASSES");
   }
   
   
   public void testChangeInvalidStateToCannotConnectTrue() throws Throwable
   {
      System.out.println("");
      log.info("entering " + getName());

      try
      {
         HashMap config = new HashMap();
         config.put(Remoting.CHANGE_INVALID_STATE_TO_CANNOT_CONNECT, "true");
         doTest(config);
         fail("No Exception: expected CannotConnectException");
      }
      catch (CannotConnectException e)
      {
         log.info("got expected CannotConnectException");
      }
      catch (Throwable t)
      {
         log.error("Expected CannotConnectException", t);
         fail("Expected CannotConnectException: got " + t);
      }
      
      log.info(getName() + " PASSES");
   }
   
   
   protected void doTestDefaultBehavior(Map config) throws Throwable
   {
      try
      {
         doTest(config);
         fail("No Exception: expected InvalidStateException");
      }
      catch (ServerInvoker.InvalidStateException e)
      {
         log.info("got expected InvalidStateException");
      }
      catch (Throwable t)
      {
         log.error("Expected InvalidStateException", t);
         fail("Expected InvalidStateException: got " + t);
      }
   }
   
   
   protected void doTest(Map config) throws Throwable
   {
      // Start server.
      setupServer();
      
      // Create client.
      InvokerLocator clientLocator = new InvokerLocator(locatorURI);
      config.put(InvokerLocator.FORCE_REMOTE, "true");
      addExtraClientConfig(config);
      Client client = new Client(clientLocator, config);
      client.connect();
      log.info("client is connected");
      
      // Test connection.
      assertEquals("abc", client.invoke("abc"));
      log.info("connection is good");
      
      // Give ServerThread time to start to read version (socket and bisocket transports).
      Thread.sleep(1000);
      
      // Shut down server.
      shutdownServer();
      
      // Try invocation.
      client.invoke("xyz");
      
      client.disconnect();
   }
   
   
   protected String getTransport()
   {
      return "socket";
   }
   
   
   protected void addExtraClientConfig(Map config) {}
   protected void addExtraServerConfig(Map config) {}
   

   protected void setupServer() throws Exception
   {
      host = InetAddress.getLocalHost().getHostAddress();
      port = PortUtil.findFreePort(host);
      locatorURI = getTransport() + "://" + host + ":" + port;
      String metadata = System.getProperty("remoting.metadata");
      if (metadata != null)
      {
         locatorURI += "/?" + metadata;
      }
      serverLocator = new InvokerLocator(locatorURI);
      log.info("Starting remoting server with locator uri of: " + locatorURI);
      HashMap config = new HashMap();
      config.put(InvokerLocator.FORCE_REMOTE, "true");
      addExtraServerConfig(config);
      connector = new Connector(serverLocator, config);
      connector.create();
      invocationHandler = new TestInvocationHandler();
      connector.addInvocationHandler("test", invocationHandler);
      connector.start();
   }
   
   
   protected void shutdownServer() throws Exception
   {
      if (connector != null)
         connector.stop();
   }
   
   
   static class TestInvocationHandler implements ServerInvocationHandler
   {
      public void addListener(InvokerCallbackHandler callbackHandler) {}
      public Object invoke(final InvocationRequest invocation) throws Throwable
      {
         return invocation.getParameter();
      }
      public void removeListener(InvokerCallbackHandler callbackHandler) {}
      public void setMBeanServer(MBeanServer server) {}
      public void setInvoker(ServerInvoker invoker) {}
   }
}