/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

/*
 * Created on Feb 26, 2006
 */
package org.jboss.test.remoting.transport.multiplex;

import java.net.InetSocketAddress;
import java.util.Collection;

import javax.management.MBeanServer;

import org.jboss.jrunit.extensions.ServerTestCase;
import org.jboss.logging.Logger;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.ServerInvocationHandler;
import org.jboss.remoting.ServerInvoker;
import org.jboss.remoting.callback.InvokerCallbackHandler;
import org.jboss.remoting.transport.Connector;
import org.jboss.remoting.transport.multiplex.MultiplexServerInvoker;
import org.jboss.remoting.transport.multiplex.MultiplexingManager;


/**
 * A MultiplexInvokerShutdownTestServer.

 * @author <a href="mailto:r.sigal@computer.org">Ron Sigal</a>
 * @version $Revision: 1977 $
 * <p>
 * Copyright (c) 2005
 * </p>
 */

public class MultiplexInvokerShutdownTestServer extends ServerTestCase
{
   public static final String locatorString = "multiplex://localhost:5281";
   public static final String CREATED_EVENT = "clientsCreatedEvent";
   public static final String SHUTDOWN_CONSENT_EVENT = "serverConsentShutdownEvent";
   public static final String CLIENT_SHUTDOWN_EVENT = "clientsShutdownEvent";
   public static final String SERVER_SHUTDOWN_EVENT = "servdrShutdownEvent";
   
   private static final Logger log = Logger.getLogger(MultiplexInvokerShutdownTestServer.class);
   private static EventRegistry_Impl eventRegistry;
   private Connector connector;
   
   
   public void setUp() throws Exception
   {
      System.out.println("entering setUp()");
      if (eventRegistry == null)
      {
         eventRegistry = new EventRegistry_Impl();
         
         try
         {
            eventRegistry.start();
         }
         catch (Exception e)
         {
            log.error(e);
         }
      }
      
      InvokerLocator locator = new InvokerLocator(locatorString);
      connector = new Connector();
      connector.setInvokerLocator(locator.getLocatorURI());
      connector.create();
      connector.addInvocationHandler("sample",  new SampleInvocationHandler());
      connector.start();
      System.out.println("Started remoting server with locator uri of: " + locatorString);
      log.info("Started remoting server with locator uri of: " + locatorString);
   }
 
   
   public void tearDown()
   {
      if (eventRegistry != null)
         eventRegistry.stop();
      
      if (connector != null)
         connector.stop();
   }
   
   
   public void testInvokerShutdown()
   {  
      try
      {
         if (eventRegistry == null)
            setUp();
         
         eventRegistry.waitOnEvent(CREATED_EVENT);
         MultiplexServerInvoker serverInvoker = (MultiplexServerInvoker) connector.getServerInvoker();
         Collection virtualInvokers = serverInvoker.getServerInvokers();
         assertFalse(virtualInvokers.isEmpty());
         InetSocketAddress localAddress = new InetSocketAddress("localhost", 5281);
         InetSocketAddress remoteAddress = new InetSocketAddress("localhost", 5282);
         assertTrue(MultiplexingManager.checkForManagerByAddressPair(localAddress, remoteAddress));
         
         eventRegistry.postEvent(SHUTDOWN_CONSENT_EVENT);
         eventRegistry.waitOnEvent(CLIENT_SHUTDOWN_EVENT);
         
         for (int i = 0; i < 3; i++)
         {
            if (virtualInvokers.isEmpty())
               break;
            
            Thread.sleep((i + 1) * 1000);
         }
         
         assertTrue(virtualInvokers.isEmpty());
         
         
         for (int i = 0; i < 3; i++)
         {
            if (!MultiplexingManager.checkForManagerByAddressPair(localAddress, remoteAddress))
               break;
            
            Thread.sleep((i + 1) * 1000);
         }
         
         assertFalse(MultiplexingManager.checkForManagerByAddressPair(localAddress, remoteAddress));
     
         // Tell client that server is done and it's OK to shut down.
         eventRegistry.postEvent(SERVER_SHUTDOWN_EVENT);
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      
      log.info("testInvokerShutdown PASSES");
      System.out.println("testInvokerShutdown PASSES");
   }
   
   
   public static void main(String[] args)
   {
      MultiplexInvokerShutdownTestServer server = null;
      
      try
      {
         server = new MultiplexInvokerShutdownTestServer();
         server.setUp();
         server.testInvokerShutdown();
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
      }
      finally
      {
         server.tearDown();
      }
   }
   
   
   public static class SampleInvocationHandler implements ServerInvocationHandler
   {
      public void setMBeanServer(MBeanServer server)
      {
      }

      public void setInvoker(ServerInvoker invoker)
      {
      }

      public Object invoke(InvocationRequest invocation) throws Throwable
      {
         return null;
      }

      public void addListener(InvokerCallbackHandler callbackHandler)
      { 
      }

      public void removeListener(InvokerCallbackHandler callbackHandler)
      {
      }
      
   }
}

