/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.test.remoting.transport.socket;

import java.net.InetAddress;
import java.util.HashMap;
import java.util.Map;

import javax.management.MBeanServer;

import junit.framework.TestCase;

import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.jboss.logging.XLevel;
import org.jboss.remoting.Client;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.ServerInvocationHandler;
import org.jboss.remoting.ServerInvoker;
import org.jboss.remoting.callback.InvokerCallbackHandler;
import org.jboss.remoting.transport.Connector;
import org.jboss.remoting.transport.PortUtil;
import org.jboss.remoting.transport.socket.SocketServerInvoker;

/**
 * Unit test for JBREM-1230.
 * 
 * @author <a href="ron.sigal@jboss.com">Ron Sigal</a>
 * @version $Revision: 1.1 $
 * <p>
 * Copyright Jul 30, 2010
 */
public class IdleTimerTaskNotifyTestCase extends TestCase
{
   private static Logger log = Logger.getLogger(IdleTimerTaskNotifyTestCase.class);
   
   private static boolean firstTime = true;
   private static String DELAY = "delay";
   private static String MAX_POOL_SIZE = "2";
   private static String IDLE_TIMEOUT = "10";
   
   protected String host;
   protected int port;
   protected String locatorURI;
   protected InvokerLocator serverLocator;
   protected Connector connector;
   protected TestInvocationHandler invocationHandler;

   
   public void setUp() throws Exception
   {
      if (firstTime)
      {
         firstTime = false;
         Logger.getLogger("org.jboss.remoting").setLevel(XLevel.TRACE);
         Logger.getLogger("org.jboss.test.remoting").setLevel(XLevel.INFO);
         String pattern = "[%d{ABSOLUTE}] [%t] %5p (%F:%L) - %m%n";
         PatternLayout layout = new PatternLayout(pattern);
         ConsoleAppender consoleAppender = new ConsoleAppender(layout);
         Logger.getRootLogger().addAppender(consoleAppender);  
      }
   }

   
   public void tearDown()
   {
   }
   
   
   public void testIdleTimerTaskNotify() throws Throwable
   {
      log.info("entering " + getName());
      
      // Start server and verify key parameters.
      setupServer();
      SocketServerInvoker serverInvoker = (SocketServerInvoker) connector.getServerInvoker();
      assertEquals(MAX_POOL_SIZE, Integer.toString(serverInvoker.getMaxPoolSize()));
      assertEquals(IDLE_TIMEOUT, Integer.toString(serverInvoker.getIdleTimeout()));
      
      // Set up invocations.
      InvokerLocator clientLocator = new InvokerLocator(locatorURI);
      HashMap clientConfig = new HashMap();
      clientConfig.put(InvokerLocator.FORCE_REMOTE, "true");
      addExtraClientConfig(clientConfig);

      TestThread tt1 = new TestThread("1", new Integer(25000), clientLocator, clientConfig);
      TestThread tt2 = new TestThread("2", new Integer(5000), clientLocator, clientConfig);
      TestThread tt3 = new TestThread("3", new Integer(0), clientLocator, clientConfig);
      tt1.start();
      Thread.sleep(1000);
      tt2.start();
      Thread.sleep(1000);
      tt3.start();
      Thread.sleep(10000);
      
      // At this point, SocketServerInvoker has created a socket for the invocation from tt3,
      // has called clientpool.evict(), and is waiting on WorkerThread#0 to shut down.  
      // * WorkerThread#0 is still running the invocation from tt1.
      // * WorkerThread#1 has completed the invocation from tt2 and is waiting for another one.
      // * The invocation from tt2 is still waiting for a thread.
      assertFalse(tt1.ok);
      assertTrue(tt2.ok);
      assertFalse(tt3.ok);
      Thread.sleep(15000);
      
      // At this point, WorkerThread#0 is still executing the invocation from tt1, but IdleTimerTask
      // has detected that WorkerThread#1 has been idle.  It shuts down WorkerThread#1 and calls
      // clientpool.notify().  SocketServerInvoker wakes up and creates WorkerThread#2, which processes
      // the invocation from tt3.
      assertTrue(tt3.ok);
      Thread.sleep(5000);
      
      // At this point, the invocation from tt1 has completed.
      assertTrue(tt1.ok);
      
      shutdownServer();
      log.info(getName() + " PASSES");
   }
   
   
   protected String getTransport()
   {
      return "socket";
   }
   
   
   protected void addExtraClientConfig(Map config) {}
   protected void addExtraServerConfig(Map config) {}
   

   protected void setupServer() throws Exception
   {
      host = InetAddress.getLocalHost().getHostAddress();
      port = PortUtil.findFreePort(host);
      locatorURI = getTransport() + "://" + host + ":" + port + "/?";
      String metadata = System.getProperty("remoting.metadata");
      if (metadata != null)
      {
         locatorURI += "&" + metadata;
      }
      serverLocator = new InvokerLocator(locatorURI);
      log.info("Starting remoting server with locator uri of: " + locatorURI);
      HashMap config = new HashMap();
      config.put(InvokerLocator.FORCE_REMOTE, "true");
      config.put("maxPoolSize", MAX_POOL_SIZE);
      config.put("idleTimeout", IDLE_TIMEOUT);
      addExtraServerConfig(config);
      connector = new Connector(serverLocator, config);
      connector.create();
      invocationHandler = new TestInvocationHandler();
      connector.addInvocationHandler("test", invocationHandler);
      connector.start();
   }
   
   
   protected void shutdownServer() throws Exception
   {
      if (connector != null)
         connector.stop();
   }
   
   
   static class TestInvocationHandler implements ServerInvocationHandler
   {
      public void addListener(InvokerCallbackHandler callbackHandler) {}
      public Object invoke(final InvocationRequest invocation) throws Throwable
      {
         Map metadata = invocation.getRequestPayload();
         int delay = 5000;
         if (metadata != null)
         {
            Object o = metadata.get(DELAY);
            if (o != null && o instanceof Integer)
            {
               delay = ((Integer)o).intValue();
            }
         }
         log.info(this + " sleeping for " + delay + " ms");
         Thread.sleep(delay);
         log.info(this + " returned from sleep");
         return invocation.getParameter();
      }
      public void removeListener(InvokerCallbackHandler callbackHandler) {}
      public void setMBeanServer(MBeanServer server) {}
      public void setInvoker(ServerInvoker invoker) {}
   }
   
   
   static class TestThread extends Thread
   {
      public boolean ok;
      String id;
      Integer delay;
      InvokerLocator locator;
      Map config;
      
      public TestThread(String id, Integer delay, InvokerLocator locator, Map config)
      {
         this.id = id;
         this.delay = delay;
         this.locator = locator;
         this.config = config;
         setName("TestThread[" + id + "]");
      }
      
      public void run()
      {
         Client client = null;
         
         try
         {
            client = new Client(locator, config);
            client.connect();
            log.info(this + " client connected");
            Map metadata = new HashMap();
            metadata.put(DELAY, delay);
            if (id.equals(client.invoke(id, metadata)))
            {
               ok = true;
               log.info(this + " invocation successful");
            }
         }
         catch (Throwable e)
         {
            log.error("invocation on " + client + " failed", e);
         }
         finally
         {
            client.disconnect();
         }
      }
      
      public String toString()
      {
         return getName();
      }
   }
}