/*
 * Decompiled with CFR 0.152.
 */
package com.webauthn4j.validator.attestation.statement.tpm;

import com.webauthn4j.data.attestation.authenticator.AAGUID;
import com.webauthn4j.data.attestation.authenticator.AuthenticatorData;
import com.webauthn4j.data.attestation.statement.AttestationType;
import com.webauthn4j.data.attestation.statement.COSEAlgorithmIdentifier;
import com.webauthn4j.data.attestation.statement.ECCUnique;
import com.webauthn4j.data.attestation.statement.RSAUnique;
import com.webauthn4j.data.attestation.statement.SignatureAlgorithm;
import com.webauthn4j.data.attestation.statement.TPMAttestationStatement;
import com.webauthn4j.data.attestation.statement.TPMGenerated;
import com.webauthn4j.data.attestation.statement.TPMIAlgHash;
import com.webauthn4j.data.attestation.statement.TPMIAlgPublic;
import com.webauthn4j.data.attestation.statement.TPMISTAttest;
import com.webauthn4j.data.attestation.statement.TPMSAttest;
import com.webauthn4j.data.attestation.statement.TPMSCertifyInfo;
import com.webauthn4j.data.attestation.statement.TPMSECCParms;
import com.webauthn4j.data.attestation.statement.TPMSRSAParms;
import com.webauthn4j.data.attestation.statement.TPMTPublic;
import com.webauthn4j.data.attestation.statement.TPMUPublicId;
import com.webauthn4j.data.extension.authenticator.RegistrationExtensionAuthenticatorOutput;
import com.webauthn4j.data.x500.X500Name;
import com.webauthn4j.util.MessageDigestUtil;
import com.webauthn4j.util.SignatureUtil;
import com.webauthn4j.util.UnsignedNumberUtil;
import com.webauthn4j.util.exception.NotImplementedException;
import com.webauthn4j.validator.RegistrationObject;
import com.webauthn4j.validator.attestation.statement.AbstractStatementValidator;
import com.webauthn4j.validator.attestation.statement.tpm.NullTPMDevicePropertyValidator;
import com.webauthn4j.validator.attestation.statement.tpm.TPMDeviceProperty;
import com.webauthn4j.validator.attestation.statement.tpm.TPMDevicePropertyValidator;
import com.webauthn4j.validator.exception.BadAttestationStatementException;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.EllipticCurve;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import org.apache.kerby.asn1.type.Asn1Utf8String;
import org.apache.kerby.asn1.util.HexUtil;

public class TPMAttestationStatementValidator
extends AbstractStatementValidator<TPMAttestationStatement> {
    private static final String ID_FIDO_GEN_CE_AAGUID = "1.3.6.1.4.1.45724.1.1.4";
    private TPMDevicePropertyValidator tpmDevicePropertyValidator = new NullTPMDevicePropertyValidator();

    @Override
    public AttestationType validate(RegistrationObject registrationObject) {
        if (!this.supports(registrationObject)) {
            throw new IllegalArgumentException("Specified format is not supported by " + this.getClass().getName());
        }
        TPMAttestationStatement attestationStatement = (TPMAttestationStatement)registrationObject.getAttestationObject().getAttestationStatement();
        if (!attestationStatement.getVer().equals("2.0")) {
            throw new BadAttestationStatementException("TPM version is not supported.");
        }
        TPMSAttest certInfo = attestationStatement.getCertInfo();
        TPMTPublic pubArea = attestationStatement.getPubArea();
        AuthenticatorData<RegistrationExtensionAuthenticatorOutput> authenticatorData = registrationObject.getAttestationObject().getAuthenticatorData();
        this.validatePublicKeyEquality(pubArea, authenticatorData);
        byte[] attToBeSigned = this.getAttToBeSigned(registrationObject);
        if (certInfo.getMagic() != TPMGenerated.TPM_GENERATED_VALUE) {
            throw new BadAttestationStatementException("magic must be TPM_GENERATED_VALUE");
        }
        if (certInfo.getType() != TPMISTAttest.TPM_ST_ATTEST_CERTIFY) {
            throw new BadAttestationStatementException("type must be TPM_ST_ATTEST_CERTIFY");
        }
        COSEAlgorithmIdentifier alg = attestationStatement.getAlg();
        String messageDigestJcaName = this.getMessageDigestJcaName(alg);
        byte[] hash = MessageDigestUtil.createMessageDigest((String)messageDigestJcaName).digest(attToBeSigned);
        if (!Arrays.equals(certInfo.getExtraData(), hash)) {
            throw new BadAttestationStatementException("extraData must be equals to the hash of attToBeSigned");
        }
        TPMSCertifyInfo certifyInfo = (TPMSCertifyInfo)certInfo.getAttested();
        TPMIAlgHash hashAlg = certifyInfo.getName().getHashAlg();
        String algJcaName = this.getAlgJcaName(hashAlg);
        byte[] pubAreaDigest = MessageDigestUtil.createMessageDigest((String)algJcaName).digest(pubArea.getBytes());
        if (!Arrays.equals(pubAreaDigest, certifyInfo.getName().getDigest())) {
            throw new BadAttestationStatementException("hash of `attested` doesn't match with name field of certifyInfo");
        }
        if (attestationStatement.getX5c() != null) {
            this.validateX5c(attestationStatement, certInfo, authenticatorData);
            return AttestationType.ATT_CA;
        }
        if (attestationStatement.getEcdaaKeyId() != null) {
            throw new NotImplementedException();
        }
        throw new BadAttestationStatementException("`x5c` or `ecdaaKeyId` must be present.");
    }

    private String getMessageDigestJcaName(COSEAlgorithmIdentifier alg) {
        String messageDigestJcaName;
        try {
            SignatureAlgorithm signatureAlgorithm = SignatureAlgorithm.create(alg);
            messageDigestJcaName = signatureAlgorithm.getMessageDigestJcaName();
        }
        catch (IllegalArgumentException e) {
            throw new BadAttestationStatementException("alg is not signature algorithm", e);
        }
        return messageDigestJcaName;
    }

    private void validateX5c(TPMAttestationStatement attestationStatement, TPMSAttest certInfo, AuthenticatorData authenticatorData) {
        X509Certificate aikCert = attestationStatement.getX5c().getEndEntityAttestationCertificate().getCertificate();
        String jcaName = this.getJcaName(attestationStatement.getAlg());
        Signature certInfoSignature = SignatureUtil.createSignature((String)jcaName);
        try {
            certInfoSignature.initVerify(aikCert.getPublicKey());
            certInfoSignature.update(certInfo.getBytes());
            if (!certInfoSignature.verify(attestationStatement.getSig())) {
                throw new BadAttestationStatementException("hash of certInfo doesn't match with sig.");
            }
        }
        catch (InvalidKeyException | SignatureException e) {
            throw new BadAttestationStatementException("Failed to validate the signature.", e);
        }
        this.validateAikCert(aikCert);
        byte[] aaguidBytes = aikCert.getExtensionValue(ID_FIDO_GEN_CE_AAGUID);
        if (aaguidBytes != null && !Objects.equals(new AAGUID(aaguidBytes), authenticatorData.getAttestedCredentialData().getAaguid())) {
            throw new BadAttestationStatementException("AAGUID in aikCert doesn't match with that in authenticatorData");
        }
    }

    String getAlgJcaName(TPMIAlgHash alg) {
        String algJcaName;
        switch (alg) {
            case TPM_ALG_SHA1: {
                algJcaName = "SHA-1";
                break;
            }
            case TPM_ALG_SHA256: {
                algJcaName = "SHA-256";
                break;
            }
            case TPM_ALG_SHA384: {
                algJcaName = "SHA-384";
                break;
            }
            case TPM_ALG_SHA512: {
                algJcaName = "SHA-512";
                break;
            }
            default: {
                throw new BadAttestationStatementException("nameAlg '" + alg.name() + "' is not supported.");
            }
        }
        return algJcaName;
    }

    public TPMDevicePropertyValidator getTpmDevicePropertyValidator() {
        return this.tpmDevicePropertyValidator;
    }

    public void setTpmDevicePropertyValidator(TPMDevicePropertyValidator tpmDevicePropertyValidator) {
        this.tpmDevicePropertyValidator = tpmDevicePropertyValidator;
    }

    private void validatePublicKeyEquality(TPMTPublic pubArea, AuthenticatorData authenticatorData) {
        PublicKey publicKeyInAuthData = authenticatorData.getAttestedCredentialData().getCOSEKey().getPublicKey();
        TPMUPublicId publicKeyInPubArea = pubArea.getUnique();
        if (pubArea.getType() == TPMIAlgPublic.TPM_ALG_RSA && publicKeyInPubArea instanceof RSAUnique) {
            RSAPublicKey rsaPublicKey = (RSAPublicKey)publicKeyInAuthData;
            TPMSRSAParms parms = (TPMSRSAParms)pubArea.getParameters();
            RSAUnique rsaUnique = (RSAUnique)publicKeyInPubArea;
            long exponent = UnsignedNumberUtil.getUnsignedInt((byte[])parms.getExponent());
            if (exponent == 0L) {
                exponent = 65537L;
            }
            if (rsaPublicKey.getModulus().equals(new BigInteger(1, rsaUnique.getN())) && rsaPublicKey.getPublicExponent().equals(BigInteger.valueOf(exponent))) {
                return;
            }
        } else if (pubArea.getType() == TPMIAlgPublic.TPM_ALG_ECDSA && publicKeyInPubArea instanceof ECCUnique) {
            ECPublicKey ecPublicKey = (ECPublicKey)publicKeyInAuthData;
            TPMSECCParms parms = (TPMSECCParms)pubArea.getParameters();
            EllipticCurve curveInParms = parms.getCurveId().getEllipticCurve();
            ECCUnique eccUnique = (ECCUnique)publicKeyInPubArea;
            if (ecPublicKey.getParams().getCurve().equals(curveInParms) && ecPublicKey.getW().getAffineX().equals(new BigInteger(1, eccUnique.getX())) && ecPublicKey.getW().getAffineY().equals(new BigInteger(1, eccUnique.getY()))) {
                return;
            }
        }
        throw new BadAttestationStatementException("publicKey in authData and publicKey in unique pubArea doesn't match");
    }

    void validateAikCert(X509Certificate certificate) {
        try {
            if (!Objects.equals(certificate.getVersion(), 3)) {
                throw new BadAttestationStatementException("x5c must be version 3.");
            }
            if (!certificate.getSubjectDN().getName().isEmpty()) {
                throw new BadAttestationStatementException("x5c subject field MUST be set to empty");
            }
            this.validateSubjectAlternativeName(certificate);
            if (certificate.getExtendedKeyUsage() == null || !certificate.getExtendedKeyUsage().contains("2.23.133.8.3")) {
                throw new BadAttestationStatementException("Attestation certificate doesn't contain tcg-kp-AIKCertificate (2.23.133.8.3) OID");
            }
            if (certificate.getBasicConstraints() != -1) {
                throw new BadAttestationStatementException("The Basic Constraints extension of attestation certificate must have the CA component set to false");
            }
        }
        catch (CertificateParsingException e) {
            throw new BadAttestationStatementException("Failed to parse attestation certificate", e);
        }
    }

    private void validateSubjectAlternativeName(X509Certificate certificate) throws CertificateParsingException {
        try {
            for (List<?> entry : certificate.getSubjectAlternativeNames()) {
                if (!entry.get(0).equals(4)) continue;
                X500Name directoryName = new X500Name((String)entry.get(1));
                TPMDeviceProperty tpmDeviceProperty = this.parseTPMDeviceProperty(directoryName);
                this.tpmDevicePropertyValidator.validate(tpmDeviceProperty);
                return;
            }
        }
        catch (IOException | RuntimeException e) {
            throw new BadAttestationStatementException("The Subject Alternative Name extension of attestation certificate does not contain a TPM device property", e);
        }
        throw new BadAttestationStatementException("The Subject Alternative Name extension of attestation certificate does not contain a TPM device property");
    }

    TPMDeviceProperty parseTPMDeviceProperty(X500Name directoryName) throws IOException {
        Map<String, String> map = directoryName.stream().flatMap(attributes -> attributes.entrySet().stream()).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        String manufacturerAttr = map.get("2.23.133.2.1");
        String partNumberAttr = map.get("2.23.133.2.2");
        String firmwareVersionAttr = map.get("2.23.133.2.3");
        String manufacturer = this.decodeAttr(manufacturerAttr);
        String partNumber = this.decodeAttr(partNumberAttr);
        String firmwareVersion = this.decodeAttr(firmwareVersionAttr);
        return new TPMDeviceProperty(manufacturer, partNumber, firmwareVersion);
    }

    private String decodeAttr(String attr) throws IOException {
        String value = null;
        if (attr != null) {
            byte[] bytes = HexUtil.hex2bytes((String)attr.substring(1));
            Asn1Utf8String attrAsn1Utf8String = new Asn1Utf8String();
            attrAsn1Utf8String.decode(bytes);
            value = (String)attrAsn1Utf8String.getValue();
        }
        return value;
    }

    private byte[] getAttToBeSigned(RegistrationObject registrationObject) {
        MessageDigest messageDigest = MessageDigestUtil.createSHA256();
        byte[] authenticatorData = registrationObject.getAuthenticatorDataBytes();
        byte[] clientDataHash = messageDigest.digest(registrationObject.getCollectedClientDataBytes());
        return ByteBuffer.allocate(authenticatorData.length + clientDataHash.length).put(authenticatorData).put(clientDataHash).array();
    }
}

