/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.authentication.authenticators.x509;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLConnection;
import java.security.GeneralSecurityException;
import java.security.cert.CRLException;
import java.security.cert.CertPathValidatorException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.InitialDirContext;
import org.keycloak.common.util.CRLUtils;
import org.keycloak.common.util.OCSPUtils;
import org.keycloak.services.ServicesLogger;

public class CertificateValidator {
    private static final ServicesLogger logger = ServicesLogger.LOGGER;
    X509Certificate[] _certChain;
    int _keyUsageBits;
    List<String> _extendedKeyUsage;
    boolean _crlCheckingEnabled;
    boolean _crldpEnabled;
    CRLLoaderImpl _crlLoader;
    boolean _ocspEnabled;
    OCSPChecker ocspChecker;

    public CertificateValidator() {
    }

    protected CertificateValidator(X509Certificate[] certChain, int keyUsageBits, List<String> extendedKeyUsage, boolean cRLCheckingEnabled, boolean cRLDPCheckingEnabled, CRLLoaderImpl crlLoader, boolean oCSPCheckingEnabled, OCSPChecker ocspChecker) {
        this._certChain = certChain;
        this._keyUsageBits = keyUsageBits;
        this._extendedKeyUsage = extendedKeyUsage;
        this._crlCheckingEnabled = cRLCheckingEnabled;
        this._crldpEnabled = cRLDPCheckingEnabled;
        this._crlLoader = crlLoader;
        this._ocspEnabled = oCSPCheckingEnabled;
        this.ocspChecker = ocspChecker;
        if (ocspChecker == null) {
            throw new IllegalArgumentException("ocspChecker");
        }
    }

    private static void validateKeyUsage(X509Certificate[] certs, int expected) throws GeneralSecurityException {
        boolean[] keyUsageBits = certs[0].getKeyUsage();
        if (keyUsageBits == null) {
            if (expected != 0) {
                String message = "Key usage extension is expected, but unavailable.";
                throw new GeneralSecurityException(message);
            }
            return;
        }
        boolean isCritical = false;
        Set<String> critSet = certs[0].getCriticalExtensionOIDs();
        if (critSet != null) {
            isCritical = critSet.contains("2.5.29.15");
        }
        int n = expected;
        StringBuilder sb = new StringBuilder();
        int i = 0;
        while (i < keyUsageBits.length) {
            boolean value = keyUsageBits[i];
            if ((n & 1) == 1 && !value) {
                String message = String.format("Key Usage bit '%s' is not set.", KeyUsageBits.fromValue(i).getName());
                if (sb.length() > 0) {
                    sb.append("\n");
                }
                sb.append(message);
                logger.warn(message);
            }
            ++i;
            n >>= 1;
        }
        if (sb.length() > 0 && isCritical) {
            throw new GeneralSecurityException(sb.toString());
        }
    }

    private static void validateExtendedKeyUsage(X509Certificate[] certs, List<String> expectedEKU) throws GeneralSecurityException {
        if (expectedEKU == null || expectedEKU.size() == 0) {
            logger.debug("Extended Key Usage validation is not enabled.");
            return;
        }
        List<String> extendedKeyUsage = certs[0].getExtendedKeyUsage();
        if (extendedKeyUsage == null) {
            String message = "Extended key usage extension is expected, but unavailable";
            throw new GeneralSecurityException(message);
        }
        boolean isCritical = false;
        Set<String> critSet = certs[0].getCriticalExtensionOIDs();
        if (critSet != null) {
            isCritical = critSet.contains("2.5.29.37");
        }
        LinkedList ekuList = new LinkedList();
        extendedKeyUsage.forEach(s -> ekuList.add(s.toLowerCase()));
        for (String eku : expectedEKU) {
            if (ekuList.contains(eku.toLowerCase())) continue;
            String message = String.format("Extended Key Usage '%s' is missing.", eku);
            if (isCritical) {
                throw new GeneralSecurityException(message);
            }
            logger.warn(message);
        }
    }

    public CertificateValidator validateKeyUsage() throws GeneralSecurityException {
        CertificateValidator.validateKeyUsage(this._certChain, this._keyUsageBits);
        return this;
    }

    public CertificateValidator validateExtendedKeyUsage() throws GeneralSecurityException {
        CertificateValidator.validateExtendedKeyUsage(this._certChain, this._extendedKeyUsage);
        return this;
    }

    private void checkRevocationUsingOCSP(X509Certificate[] certs) throws GeneralSecurityException {
        if (certs.length < 2) {
            String message = "OCSP requires a responder certificate. OCSP cannot be used to verify the revocation status of self-signed certificates.";
            throw new GeneralSecurityException(message);
        }
        for (X509Certificate cert : certs) {
            logger.debugf("Certificate: %s", cert.getSubjectDN().getName());
        }
        OCSPUtils.OCSPRevocationStatus rs = this.ocspChecker.check(certs[0], certs[1]);
        if (rs == null) {
            throw new GeneralSecurityException("Unable to check client revocation status using OCSP");
        }
        if (rs.getRevocationStatus() == OCSPUtils.RevocationStatus.UNKNOWN) {
            throw new GeneralSecurityException("Unable to determine certificate's revocation status.");
        }
        if (rs.getRevocationStatus() == OCSPUtils.RevocationStatus.REVOKED) {
            StringBuilder sb = new StringBuilder();
            sb.append("Certificate's been revoked.");
            sb.append("\n");
            sb.append(rs.getRevocationReason().toString());
            sb.append("\n");
            sb.append(String.format("Revoked on: %s", rs.getRevocationTime().toString()));
            throw new GeneralSecurityException(sb.toString());
        }
    }

    private static void checkRevocationStatusUsingCRL(X509Certificate[] certs, CRLLoaderImpl crLoader) throws GeneralSecurityException {
        Collection<X509CRL> crlColl = crLoader.getX509CRLs();
        if (crlColl != null && crlColl.size() > 0) {
            for (X509CRL it : crlColl) {
                if (!it.isRevoked(certs[0])) continue;
                String message = String.format("Certificate has been revoked, certificate's subject: %s", certs[0].getSubjectDN().getName());
                logger.debug(message);
                throw new GeneralSecurityException(message);
            }
        }
    }

    private static List<String> getCRLDistributionPoints(X509Certificate cert) {
        try {
            return CRLUtils.getCRLDistributionPoints((X509Certificate)cert);
        }
        catch (IOException e) {
            logger.error(e.getMessage());
            return new ArrayList<String>();
        }
    }

    private static void checkRevocationStatusUsingCRLDistributionPoints(X509Certificate[] certs) throws GeneralSecurityException {
        List<String> distributionPoints = CertificateValidator.getCRLDistributionPoints(certs[0]);
        if (distributionPoints == null || distributionPoints.size() == 0) {
            throw new GeneralSecurityException("Could not find any CRL distribution points in the certificate, unable to check the certificate revocation status using CRL/DP.");
        }
        for (String dp : distributionPoints) {
            logger.tracef("CRL Distribution point: \"%s\"", dp);
            CertificateValidator.checkRevocationStatusUsingCRL(certs, new CRLFileLoader(dp));
        }
    }

    public CertificateValidator checkRevocationStatus() throws GeneralSecurityException {
        if (!this._crlCheckingEnabled && !this._ocspEnabled) {
            return this;
        }
        if (this._crlCheckingEnabled) {
            if (!this._crldpEnabled) {
                CertificateValidator.checkRevocationStatusUsingCRL(this._certChain, this._crlLoader);
            } else {
                CertificateValidator.checkRevocationStatusUsingCRLDistributionPoints(this._certChain);
            }
        }
        if (this._ocspEnabled) {
            this.checkRevocationUsingOCSP(this._certChain);
        }
        return this;
    }

    public static class CertificateValidatorBuilder {
        int _keyUsageBits = 0;
        List<String> _extendedKeyUsage = new LinkedList<String>();
        boolean _crlCheckingEnabled;
        boolean _crldpEnabled;
        CRLLoaderImpl _crlLoader;
        boolean _ocspEnabled;
        String _responderUri;

        public KeyUsageValidationBuilder keyUsage() {
            return new KeyUsageValidationBuilder(this);
        }

        public ExtendedKeyUsageValidationBuilder extendedKeyUsage() {
            return new ExtendedKeyUsageValidationBuilder(this);
        }

        public RevocationStatusCheckBuilder revocation() {
            return new RevocationStatusCheckBuilder(this);
        }

        public CertificateValidator build(X509Certificate[] certs) {
            if (this._crlLoader == null) {
                this._crlLoader = new CRLFileLoader("");
            }
            return new CertificateValidator(certs, this._keyUsageBits, this._extendedKeyUsage, this._crlCheckingEnabled, this._crldpEnabled, this._crlLoader, this._ocspEnabled, new BouncyCastleOCSPChecker(this._responderUri));
        }

        public class RevocationStatusCheckBuilder {
            CertificateValidatorBuilder _parent;

            protected RevocationStatusCheckBuilder(CertificateValidatorBuilder parent) {
                this._parent = parent;
            }

            public GotCRL cRLEnabled(boolean value) {
                CertificateValidatorBuilder.this._crlCheckingEnabled = value;
                return new GotCRL();
            }

            public class GotOCSP {
                public CertificateValidatorBuilder oCSPResponderURI(String responderURI) {
                    CertificateValidatorBuilder.this._responderUri = responderURI;
                    return RevocationStatusCheckBuilder.this._parent;
                }
            }

            public class GotCRLDP {
                public GotCRLRelativePath cRLrelativePath(String value) {
                    if (value != null) {
                        CertificateValidatorBuilder.this._crlLoader = new CRLFileLoader(value);
                    }
                    return new GotCRLRelativePath();
                }

                public GotCRLRelativePath cRLLoader(CRLLoaderImpl cRLLoader) {
                    if (cRLLoader != null) {
                        CertificateValidatorBuilder.this._crlLoader = cRLLoader;
                    }
                    return new GotCRLRelativePath();
                }
            }

            public class GotCRLRelativePath {
                public GotOCSP oCSPEnabled(boolean value) {
                    CertificateValidatorBuilder.this._ocspEnabled = value;
                    return new GotOCSP();
                }
            }

            public class GotCRL {
                public GotCRLDP cRLDPEnabled(boolean value) {
                    CertificateValidatorBuilder.this._crldpEnabled = value;
                    return new GotCRLDP();
                }
            }
        }

        public class ExtendedKeyUsageValidationBuilder {
            CertificateValidatorBuilder _parent;

            protected ExtendedKeyUsageValidationBuilder(CertificateValidatorBuilder parent) {
                this._parent = parent;
            }

            public CertificateValidatorBuilder parse(String extendedKeyUsage) {
                String[] strs;
                if (extendedKeyUsage == null || extendedKeyUsage.trim().length() == 0) {
                    return this._parent;
                }
                for (String str : strs = extendedKeyUsage.split("[,;:]]")) {
                    CertificateValidatorBuilder.this._extendedKeyUsage.add(str.trim());
                }
                return this._parent;
            }
        }

        public class KeyUsageValidationBuilder {
            CertificateValidatorBuilder _parent;

            KeyUsageValidationBuilder(CertificateValidatorBuilder parent) {
                this._parent = parent;
            }

            public KeyUsageValidationBuilder enableDigitalSignatureBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.DIGITAL_SIGNATURE.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enablecRLSignBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.CRLSIGN.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enableDataEncriphermentBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.DATA_ENCIPHERMENT.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enableDecipherOnlyBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.DECIPHER_ONLY.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enableEnciphermentOnlyBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.ENCIPHERMENT_ONLY.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enableKeyAgreementBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.KEY_AGREEMENT.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enableKeyEnciphermentBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.KEY_ENCIPHERMENT.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enableKeyCertSign() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.KEYCERTSIGN.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enableNonRepudiationBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.NON_REPUDIATION.getInt();
                return this;
            }

            public CertificateValidatorBuilder back() {
                return this._parent;
            }

            CertificateValidatorBuilder parse(String keyUsage) {
                String[] strs;
                if (keyUsage == null || keyUsage.trim().length() == 0) {
                    return this._parent;
                }
                for (String s : strs = keyUsage.split("[,]")) {
                    try {
                        KeyUsageBits bit = KeyUsageBits.parse(s.trim());
                        switch (bit) {
                            case CRLSIGN: {
                                this.enablecRLSignBit();
                                break;
                            }
                            case DATA_ENCIPHERMENT: {
                                this.enableDataEncriphermentBit();
                                break;
                            }
                            case DECIPHER_ONLY: {
                                this.enableDecipherOnlyBit();
                                break;
                            }
                            case DIGITAL_SIGNATURE: {
                                this.enableDigitalSignatureBit();
                                break;
                            }
                            case ENCIPHERMENT_ONLY: {
                                this.enableEnciphermentOnlyBit();
                                break;
                            }
                            case KEY_AGREEMENT: {
                                this.enableKeyAgreementBit();
                                break;
                            }
                            case KEY_ENCIPHERMENT: {
                                this.enableKeyEnciphermentBit();
                                break;
                            }
                            case KEYCERTSIGN: {
                                this.enableKeyCertSign();
                                break;
                            }
                            case NON_REPUDIATION: {
                                this.enableNonRepudiationBit();
                            }
                        }
                    }
                    catch (IllegalArgumentException e) {
                        logger.warnf("Unable to parse key usage bit: \"%s\"", s);
                    }
                    catch (IndexOutOfBoundsException e) {
                        logger.warnf("Invalid key usage bit: \"%s\"", s);
                    }
                }
                return this._parent;
            }
        }
    }

    public static class CRLFileLoader
    extends CRLLoaderImpl {
        private final String cRLPath;
        private final LdapContext ldapContext;

        public CRLFileLoader(String cRLPath) {
            this.cRLPath = cRLPath;
            this.ldapContext = new LdapContext();
        }

        public CRLFileLoader(String cRLPath, LdapContext ldapContext) {
            this.cRLPath = cRLPath;
            this.ldapContext = ldapContext;
            if (ldapContext == null) {
                throw new NullPointerException("Context cannot be null");
            }
        }

        @Override
        public Collection<X509CRL> getX509CRLs() throws GeneralSecurityException {
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            Collection<X509CRL> crlColl = null;
            if (this.cRLPath != null) {
                if (this.cRLPath.startsWith("http") || this.cRLPath.startsWith("https")) {
                    try {
                        crlColl = this.loadFromURI(cf, new URI(this.cRLPath));
                    }
                    catch (URISyntaxException e) {
                        logger.error(e.getMessage());
                    }
                } else if (this.cRLPath.startsWith("ldap")) {
                    try {
                        crlColl = this.loadCRLFromLDAP(cf, new URI(this.cRLPath));
                    }
                    catch (URISyntaxException e) {
                        logger.error(e.getMessage());
                    }
                } else {
                    crlColl = this.loadCRLFromFile(cf, this.cRLPath);
                }
            }
            if (crlColl == null || crlColl.size() == 0) {
                String message = String.format("Unable to load CRL from \"%s\"", this.cRLPath);
                throw new GeneralSecurityException(message);
            }
            return crlColl;
        }

        private Collection<X509CRL> loadFromURI(CertificateFactory cf, URI remoteURI) throws GeneralSecurityException {
            try {
                logger.debugf("Loading CRL from %s", remoteURI.toString());
                URLConnection conn = remoteURI.toURL().openConnection();
                conn.setDoInput(true);
                conn.setUseCaches(false);
                X509CRL crl = this.loadFromStream(cf, conn.getInputStream());
                return Collections.singleton(crl);
            }
            catch (IOException ex) {
                logger.errorf(ex.getMessage(), new Object[0]);
                return Collections.emptyList();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private Collection<X509CRL> loadCRLFromLDAP(CertificateFactory cf, URI remoteURI) throws GeneralSecurityException {
            block7: {
                Set<X509CRL> set;
                Hashtable<String, String> env = new Hashtable<String, String>(2);
                env.put("java.naming.factory.initial", this.ldapContext.getLdapFactoryClassName());
                env.put("java.naming.provider.url", remoteURI.toString());
                InitialDirContext ctx = new InitialDirContext(env);
                try {
                    Attributes attrs = ctx.getAttributes("");
                    Attribute cRLAttribute = attrs.get("certificateRevocationList;binary");
                    byte[] data = (byte[])cRLAttribute.get();
                    if (data == null || data.length == 0) {
                        throw new CertificateException(String.format("Failed to download CRL from \"%s\"", remoteURI.toString()));
                    }
                    X509CRL crl = this.loadFromStream(cf, new ByteArrayInputStream(data));
                    set = Collections.singleton(crl);
                }
                catch (Throwable throwable) {
                    try {
                        ctx.close();
                        throw throwable;
                    }
                    catch (NamingException e) {
                        logger.error(e.getMessage());
                        break block7;
                    }
                    catch (IOException e) {
                        logger.error(e.getMessage());
                    }
                }
                ctx.close();
                return set;
            }
            return Collections.emptyList();
        }

        private Collection<X509CRL> loadCRLFromFile(CertificateFactory cf, String relativePath) throws GeneralSecurityException {
            try {
                File f;
                String configDir = System.getProperty("jboss.server.config.dir");
                if (configDir != null && (f = new File(configDir + File.separator + relativePath)).isFile()) {
                    logger.debugf("Loading CRL from %s", f.getAbsolutePath());
                    if (!f.canRead()) {
                        throw new IOException(String.format("Unable to read CRL from \"%s\"", f.getAbsolutePath()));
                    }
                    X509CRL crl = this.loadFromStream(cf, new FileInputStream(f.getAbsolutePath()));
                    return Collections.singleton(crl);
                }
            }
            catch (IOException ex) {
                logger.errorf(ex.getMessage(), new Object[0]);
            }
            return Collections.emptyList();
        }

        private X509CRL loadFromStream(CertificateFactory cf, InputStream is) throws IOException, CRLException {
            DataInputStream dis = new DataInputStream(is);
            X509CRL crl = (X509CRL)cf.generateCRL(dis);
            dis.close();
            return crl;
        }
    }

    public static class CRLLoaderProxy
    extends CRLLoaderImpl {
        private final X509CRL _crl;

        public CRLLoaderProxy(X509CRL crl) {
            this._crl = crl;
        }

        @Override
        public Collection<X509CRL> getX509CRLs() throws GeneralSecurityException {
            return Collections.singleton(this._crl);
        }
    }

    public static class BouncyCastleOCSPChecker
    extends OCSPChecker {
        private final String responderUri;

        BouncyCastleOCSPChecker(String responderUri) {
            this.responderUri = responderUri;
        }

        @Override
        public OCSPUtils.OCSPRevocationStatus check(X509Certificate cert, X509Certificate issuerCertificate) throws CertPathValidatorException {
            OCSPUtils.OCSPRevocationStatus ocspRevocationStatus = null;
            if (this.responderUri == null || this.responderUri.trim().length() == 0) {
                ocspRevocationStatus = OCSPUtils.check((X509Certificate)cert, (X509Certificate)issuerCertificate);
            } else {
                URI uri;
                try {
                    uri = new URI(this.responderUri);
                }
                catch (URISyntaxException e) {
                    String message = String.format("Unable to check certificate revocation status using OCSP.\n%s", e.getMessage());
                    throw new CertPathValidatorException(message, e);
                }
                logger.tracef("Responder URI \"%s\" will be used to verify revocation status of the certificate using OCSP", uri.toString());
                ocspRevocationStatus = OCSPUtils.check((X509Certificate)cert, (X509Certificate)issuerCertificate, (URI)uri, null, null);
            }
            return ocspRevocationStatus;
        }
    }

    public static abstract class CRLLoaderImpl {
        public abstract Collection<X509CRL> getX509CRLs() throws GeneralSecurityException;
    }

    public static abstract class OCSPChecker {
        public abstract OCSPUtils.OCSPRevocationStatus check(X509Certificate var1, X509Certificate var2) throws CertPathValidatorException;
    }

    public static class LdapContext {
        private final String ldapFactoryClassName;

        public LdapContext() {
            this.ldapFactoryClassName = "com.sun.jndi.ldap.LdapCtxFactory";
        }

        public LdapContext(String ldapFactoryClassName) {
            this.ldapFactoryClassName = ldapFactoryClassName;
        }

        public String getLdapFactoryClassName() {
            return this.ldapFactoryClassName;
        }
    }

    static enum KeyUsageBits {
        DIGITAL_SIGNATURE(0, "digitalSignature"),
        NON_REPUDIATION(1, "nonRepudiation"),
        KEY_ENCIPHERMENT(2, "keyEncipherment"),
        DATA_ENCIPHERMENT(3, "dataEncipherment"),
        KEY_AGREEMENT(4, "keyAgreement"),
        KEYCERTSIGN(5, "keyCertSign"),
        CRLSIGN(6, "cRLSign"),
        ENCIPHERMENT_ONLY(7, "encipherOnly"),
        DECIPHER_ONLY(8, "decipherOnly");

        private int value;
        private String name;

        private KeyUsageBits(int value, String name) {
            if (value < 0 || value > 8) {
                throw new IllegalArgumentException("value");
            }
            if (name == null || name.trim().length() == 0) {
                throw new IllegalArgumentException("name");
            }
            this.value = value;
            this.name = name.trim();
        }

        public int getInt() {
            return this.value;
        }

        public String getName() {
            return this.name;
        }

        public static KeyUsageBits parse(String name) throws IllegalArgumentException, IndexOutOfBoundsException {
            if (name == null || name.trim().length() == 0) {
                throw new IllegalArgumentException("name");
            }
            for (KeyUsageBits bit : KeyUsageBits.values()) {
                if (!bit.getName().equalsIgnoreCase(name)) continue;
                return bit;
            }
            throw new IndexOutOfBoundsException("name");
        }

        public static KeyUsageBits fromValue(int value) throws IndexOutOfBoundsException {
            if (value < 0 || value > 8) {
                throw new IndexOutOfBoundsException("value");
            }
            for (KeyUsageBits bit : KeyUsageBits.values()) {
                if (bit.getInt() != value) continue;
                return bit;
            }
            throw new IndexOutOfBoundsException("value");
        }
    }
}

