/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.services.managers;

import java.net.URI;
import java.security.PublicKey;
import java.util.HashSet;
import java.util.LinkedList;
import javax.ws.rs.core.Cookie;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.UriInfo;
import org.jboss.logging.Logger;
import org.keycloak.ClientConnection;
import org.keycloak.RSATokenVerifier;
import org.keycloak.VerificationException;
import org.keycloak.jose.jws.JWSBuilder;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RealmModel;
import org.keycloak.models.RequiredCredentialModel;
import org.keycloak.models.UserCredentialModel;
import org.keycloak.models.UserModel;
import org.keycloak.models.UserSessionModel;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.representations.AccessToken;
import org.keycloak.services.managers.BruteForceProtector;
import org.keycloak.services.managers.ResourceAdminManager;
import org.keycloak.services.resources.RealmsResource;
import org.keycloak.services.util.CookieHelper;
import org.keycloak.util.Time;

public class AuthenticationManager {
    protected static Logger logger = Logger.getLogger(AuthenticationManager.class);
    public static final String FORM_USERNAME = "username";
    public static final String KEYCLOAK_IDENTITY_COOKIE = "KEYCLOAK_IDENTITY";
    public static final String KEYCLOAK_SESSION_COOKIE = "KEYCLOAK_SESSION";
    public static final String KEYCLOAK_REMEMBER_ME = "KEYCLOAK_REMEMBER_ME";
    protected BruteForceProtector protector;

    public AuthenticationManager() {
    }

    public AuthenticationManager(BruteForceProtector protector) {
        this.protector = protector;
    }

    public static boolean isSessionValid(RealmModel realm, UserSessionModel userSession) {
        if (userSession == null) {
            logger.debug((Object)"No user session");
            return false;
        }
        int currentTime = Time.currentTime();
        int max = userSession.getStarted() + realm.getSsoSessionMaxLifespan();
        return userSession != null && userSession.getLastSessionRefresh() + realm.getSsoSessionIdleTimeout() > currentTime && max > currentTime;
    }

    public static void logout(KeycloakSession session, RealmModel realm, UserSessionModel userSession, UriInfo uriInfo, ClientConnection connection) {
        if (userSession == null) {
            return;
        }
        UserModel user = userSession.getUser();
        logger.debugv("Logging out: {0} ({1})", (Object)user.getUsername(), (Object)userSession.getId());
        session.sessions().removeUserSession(realm, userSession);
        AuthenticationManager.expireIdentityCookie(realm, uriInfo, connection);
        AuthenticationManager.expireRememberMeCookie(realm, uriInfo, connection);
        new ResourceAdminManager().logoutUser(uriInfo.getRequestUri(), realm, user.getId(), userSession.getId());
    }

    public AccessToken createIdentityToken(RealmModel realm, UserModel user, UserSessionModel session) {
        AccessToken token = new AccessToken();
        token.id(KeycloakModelUtils.generateId());
        token.issuedNow();
        token.subject(user.getId());
        token.audience(realm.getName());
        if (session != null) {
            token.setSessionState(session.getId());
        }
        if (realm.getSsoSessionIdleTimeout() > 0) {
            token.expiration(Time.currentTime() + realm.getSsoSessionIdleTimeout());
        }
        return token;
    }

    public void createLoginCookie(RealmModel realm, UserModel user, UserSessionModel session, UriInfo uriInfo, ClientConnection connection) {
        String cookiePath = AuthenticationManager.getIdentityCookiePath(realm, uriInfo);
        AccessToken identityToken = this.createIdentityToken(realm, user, session);
        String encoded = this.encodeToken(realm, identityToken);
        boolean secureOnly = realm.getSslRequired().isRequired(connection);
        int maxAge = -1;
        if (session.isRememberMe()) {
            maxAge = realm.getSsoSessionIdleTimeout();
        }
        logger.debugv("Create login cookie - name: {0}, path: {1}, max-age: {2}", (Object)KEYCLOAK_IDENTITY_COOKIE, (Object)cookiePath, (Object)maxAge);
        CookieHelper.addCookie(KEYCLOAK_IDENTITY_COOKIE, encoded, cookiePath, null, null, maxAge, secureOnly, true);
        String sessionCookieValue = realm.getName() + "/" + user.getId();
        if (session != null) {
            sessionCookieValue = sessionCookieValue + "/" + session.getId();
        }
        CookieHelper.addCookie(KEYCLOAK_SESSION_COOKIE, sessionCookieValue, cookiePath, null, null, realm.getSsoSessionMaxLifespan(), secureOnly, false);
    }

    public void createRememberMeCookie(RealmModel realm, UriInfo uriInfo, ClientConnection connection) {
        String path = AuthenticationManager.getIdentityCookiePath(realm, uriInfo);
        boolean secureOnly = realm.getSslRequired().isRequired(connection);
        CookieHelper.addCookie(KEYCLOAK_REMEMBER_ME, "true", path, null, null, realm.getSsoSessionIdleTimeout(), secureOnly, true);
    }

    protected String encodeToken(RealmModel realm, Object token) {
        String encodedToken = new JWSBuilder().jsonContent(token).rsa256(realm.getPrivateKey());
        return encodedToken;
    }

    public static void expireIdentityCookie(RealmModel realm, UriInfo uriInfo, ClientConnection connection) {
        logger.debug((Object)"Expiring identity cookie");
        String path = AuthenticationManager.getIdentityCookiePath(realm, uriInfo);
        AuthenticationManager.expireCookie(realm, KEYCLOAK_IDENTITY_COOKIE, path, true, connection);
        AuthenticationManager.expireCookie(realm, KEYCLOAK_SESSION_COOKIE, path, false, connection);
        AuthenticationManager.expireRememberMeCookie(realm, uriInfo, connection);
    }

    public static void expireRememberMeCookie(RealmModel realm, UriInfo uriInfo, ClientConnection connection) {
        logger.debug((Object)"Expiring remember me cookie");
        String path = AuthenticationManager.getIdentityCookiePath(realm, uriInfo);
        String cookieName = KEYCLOAK_REMEMBER_ME;
        AuthenticationManager.expireCookie(realm, cookieName, path, true, connection);
    }

    protected static String getIdentityCookiePath(RealmModel realm, UriInfo uriInfo) {
        URI uri = RealmsResource.realmBaseUrl(uriInfo).build(new Object[]{realm.getName()});
        return uri.getRawPath();
    }

    public static void expireCookie(RealmModel realm, String cookieName, String path, boolean httpOnly, ClientConnection connection) {
        logger.debugv("Expiring cookie: {0} path: {1}", (Object)cookieName, (Object)path);
        boolean secureOnly = realm.getSslRequired().isRequired(connection);
        CookieHelper.addCookie(cookieName, "", path, null, "Expiring cookie", 0, secureOnly, httpOnly);
    }

    public AuthResult authenticateIdentityCookie(KeycloakSession session, RealmModel realm, UriInfo uriInfo, ClientConnection connection, HttpHeaders headers) {
        return this.authenticateIdentityCookie(session, realm, uriInfo, connection, headers, true);
    }

    public AuthResult authenticateIdentityCookie(KeycloakSession session, RealmModel realm, UriInfo uriInfo, ClientConnection connection, HttpHeaders headers, boolean checkActive) {
        Cookie cookie = (Cookie)headers.getCookies().get(KEYCLOAK_IDENTITY_COOKIE);
        if (cookie == null || "".equals(cookie.getValue())) {
            logger.debugv("Could not find cookie: {0}", (Object)KEYCLOAK_IDENTITY_COOKIE);
            return null;
        }
        String tokenString = cookie.getValue();
        AuthResult authResult = this.verifyIdentityToken(session, realm, uriInfo, connection, checkActive, tokenString);
        if (authResult == null) {
            AuthenticationManager.expireIdentityCookie(realm, uriInfo, connection);
            return null;
        }
        authResult.getSession().setLastSessionRefresh(Time.currentTime());
        return authResult;
    }

    protected AuthResult verifyIdentityToken(KeycloakSession session, RealmModel realm, UriInfo uriInfo, ClientConnection connection, boolean checkActive, String tokenString) {
        try {
            UserModel user;
            AccessToken token = RSATokenVerifier.verifyToken((String)tokenString, (PublicKey)realm.getPublicKey(), (String)realm.getName(), (boolean)checkActive);
            if (checkActive) {
                if (!token.isActive() || token.getIssuedAt() < realm.getNotBefore()) {
                    logger.debug((Object)"identity cookie expired");
                    return null;
                }
                logger.debugv("token not active - active: {0}, issued-at: {1}, not-before: {2}", (Object)token.isActive(), (Object)token.getIssuedAt(), (Object)realm.getNotBefore());
            }
            if ((user = session.users().getUserById(token.getSubject(), realm)) == null || !user.isEnabled()) {
                logger.debug((Object)"Unknown user in identity token");
                return null;
            }
            UserSessionModel userSession = session.sessions().getUserSession(realm, token.getSessionState());
            if (!AuthenticationManager.isSessionValid(realm, userSession)) {
                if (userSession != null) {
                    AuthenticationManager.logout(session, realm, userSession, uriInfo, connection);
                }
                logger.debug((Object)"User session not active");
                return null;
            }
            return new AuthResult(user, userSession, token);
        }
        catch (VerificationException e) {
            logger.debug((Object)"Failed to verify identity token", (Throwable)e);
            return null;
        }
    }

    public AuthenticationStatus authenticateForm(KeycloakSession session, ClientConnection clientConnection, RealmModel realm, MultivaluedMap<String, String> formData) {
        String username = (String)formData.getFirst((Object)FORM_USERNAME);
        if (username == null) {
            logger.debug((Object)"Username not provided");
            return AuthenticationStatus.INVALID_USER;
        }
        if (realm.isBruteForceProtected() && this.protector.isTemporarilyDisabled(session, realm, username)) {
            return AuthenticationStatus.ACCOUNT_TEMPORARILY_DISABLED;
        }
        AuthenticationStatus status = this.authenticateInternal(session, realm, formData, username);
        if (realm.isBruteForceProtected()) {
            switch (status) {
                case SUCCESS: {
                    this.protector.successfulLogin(realm, username, clientConnection);
                    break;
                }
                case FAILED: 
                case MISSING_TOTP: 
                case MISSING_PASSWORD: 
                case INVALID_CREDENTIALS: {
                    this.protector.failedLogin(realm, username, clientConnection);
                    break;
                }
                case INVALID_USER: {
                    this.protector.invalidUser(realm, username, clientConnection);
                    break;
                }
            }
        }
        return status;
    }

    protected AuthenticationStatus authenticateInternal(KeycloakSession session, RealmModel realm, MultivaluedMap<String, String> formData, String username) {
        UserModel user = KeycloakModelUtils.findUserByNameOrEmail((KeycloakSession)session, (RealmModel)realm, (String)username);
        if (user == null) {
            logger.debugv("User {0} not found", (Object)username);
            return AuthenticationStatus.INVALID_USER;
        }
        if (!user.isEnabled()) {
            return AuthenticationStatus.ACCOUNT_DISABLED;
        }
        HashSet<String> types = new HashSet<String>();
        for (RequiredCredentialModel credential : realm.getRequiredCredentials()) {
            types.add(credential.getType());
        }
        if (types.contains("password")) {
            String totp;
            String passwordToken;
            LinkedList<UserCredentialModel> credentials = new LinkedList<UserCredentialModel>();
            String password = (String)formData.getFirst((Object)"password");
            if (password != null) {
                credentials.add(UserCredentialModel.password((String)password));
            }
            if ((passwordToken = (String)formData.getFirst((Object)"password-token")) != null) {
                credentials.add(UserCredentialModel.passwordToken((String)passwordToken));
            }
            if ((totp = (String)formData.getFirst((Object)"totp")) != null) {
                credentials.add(UserCredentialModel.totp((String)totp));
            }
            if (password == null && passwordToken == null) {
                logger.debug((Object)"Password not provided");
                return AuthenticationStatus.MISSING_PASSWORD;
            }
            logger.debugv("validating password for user: {0}", (Object)username);
            if (!session.users().validCredentials(realm, user, credentials)) {
                return AuthenticationStatus.INVALID_CREDENTIALS;
            }
            if (user.isTotp() && totp == null) {
                return AuthenticationStatus.MISSING_TOTP;
            }
            if (!user.getRequiredActions().isEmpty()) {
                return AuthenticationStatus.ACTIONS_REQUIRED;
            }
            return AuthenticationStatus.SUCCESS;
        }
        if (types.contains("secret")) {
            String secret = (String)formData.getFirst((Object)"secret");
            if (secret == null) {
                logger.debug((Object)"Secret not provided");
                return AuthenticationStatus.MISSING_PASSWORD;
            }
            if (!session.users().validCredentials(realm, user, new UserCredentialModel[]{UserCredentialModel.secret((String)secret)})) {
                return AuthenticationStatus.INVALID_CREDENTIALS;
            }
            if (!user.getRequiredActions().isEmpty()) {
                return AuthenticationStatus.ACTIONS_REQUIRED;
            }
            return AuthenticationStatus.SUCCESS;
        }
        logger.warn((Object)"Do not know how to authenticate user");
        return AuthenticationStatus.FAILED;
    }

    public class AuthResult {
        private final UserModel user;
        private final UserSessionModel session;
        private final AccessToken token;

        public AuthResult(UserModel user, UserSessionModel session, AccessToken token) {
            this.user = user;
            this.session = session;
            this.token = token;
        }

        public UserSessionModel getSession() {
            return this.session;
        }

        public UserModel getUser() {
            return this.user;
        }

        public AccessToken getToken() {
            return this.token;
        }
    }

    public static enum AuthenticationStatus {
        SUCCESS,
        ACCOUNT_TEMPORARILY_DISABLED,
        ACCOUNT_DISABLED,
        ACTIONS_REQUIRED,
        INVALID_USER,
        INVALID_CREDENTIALS,
        MISSING_PASSWORD,
        MISSING_TOTP,
        FAILED;

    }
}

