/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.oidc.common.runtime;

import io.quarkus.credentials.CredentialsProvider;
import io.quarkus.credentials.runtime.CredentialsProviderFinder;
import io.quarkus.oidc.common.runtime.OidcCommonConfig;
import io.quarkus.oidc.common.runtime.OidcEndpointAccessException;
import io.quarkus.runtime.TlsConfig;
import io.quarkus.runtime.configuration.ConfigurationException;
import io.smallrye.jwt.build.Jwt;
import io.smallrye.jwt.build.JwtSignatureBuilder;
import io.smallrye.jwt.util.KeyUtils;
import io.smallrye.jwt.util.ResourceUtils;
import io.smallrye.mutiny.Uni;
import io.vertx.core.http.HttpClientOptions;
import io.vertx.core.json.JsonObject;
import io.vertx.core.net.ProxyOptions;
import io.vertx.mutiny.core.MultiMap;
import io.vertx.mutiny.core.buffer.Buffer;
import io.vertx.mutiny.ext.web.client.WebClient;
import java.io.InputStream;
import java.net.ConnectException;
import java.net.URI;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.Key;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.time.Duration;
import java.util.Base64;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.function.Predicate;
import java.util.function.Supplier;
import javax.crypto.SecretKey;
import org.jboss.logging.Logger;

public class OidcCommonUtils {
    public static final Duration CONNECTION_BACKOFF_DURATION = Duration.ofSeconds(2L);
    static final byte AMP = 38;
    static final byte EQ = 61;
    private static final Logger LOG = Logger.getLogger(OidcCommonUtils.class);

    private OidcCommonUtils() {
    }

    public static void verifyCommonConfiguration(OidcCommonConfig oidcConfig, boolean clientIdOptional, boolean isServerConfig) {
        String configPrefix;
        String string = configPrefix = isServerConfig ? "quarkus.oidc." : "quarkus.oidc-client.";
        if (!oidcConfig.getAuthServerUrl().isPresent()) {
            throw new ConfigurationException(String.format("'%sauth-server-url' property must be configured", configPrefix));
        }
        if (!clientIdOptional && !oidcConfig.getClientId().isPresent()) {
            throw new ConfigurationException(String.format("'%sclient-id' property must be configured", configPrefix));
        }
        try {
            URI.create(oidcConfig.getAuthServerUrl().get()).toURL();
        }
        catch (Throwable ex) {
            throw new ConfigurationException(String.format("'%sauth-server-url' is invalid", configPrefix), ex);
        }
        OidcCommonConfig.Credentials creds = oidcConfig.getCredentials();
        if (creds.secret.isPresent() && creds.clientSecret.value.isPresent()) {
            throw new ConfigurationException(String.format("'%1$scredentials.secret' and '%1$scredentials.client-secret' properties are mutually exclusive", configPrefix));
        }
        if ((creds.secret.isPresent() || creds.clientSecret.value.isPresent()) && creds.jwt.secret.isPresent()) {
            throw new ConfigurationException(String.format("Use only '%1$scredentials.secret' or '%1$scredentials.client-secret' or '%1$scredentials.jwt.secret' property", configPrefix));
        }
    }

    public static String prependSlash(String path) {
        return !path.startsWith("/") ? "/" + path : path;
    }

    public static Buffer encodeForm(MultiMap form) {
        Buffer buffer = Buffer.buffer();
        for (Map.Entry<String, String> entry : form) {
            if (buffer.length() != 0) {
                buffer.appendByte((byte)38);
            }
            buffer.appendString(entry.getKey());
            buffer.appendByte((byte)61);
            buffer.appendString(OidcCommonUtils.urlEncode(entry.getValue()));
        }
        return buffer;
    }

    public static String urlEncode(String value) {
        try {
            return URLEncoder.encode(value, StandardCharsets.UTF_8.name());
        }
        catch (Exception ex) {
            throw new RuntimeException(ex);
        }
    }

    public static void setHttpClientOptions(OidcCommonConfig oidcConfig, TlsConfig tlsConfig, HttpClientOptions options) {
        OptionalInt maxPoolSize;
        Optional<ProxyOptions> proxyOpt;
        boolean trustAll;
        boolean bl = oidcConfig.tls.verification.isPresent() ? oidcConfig.tls.verification.get() == OidcCommonConfig.Tls.Verification.NONE : (trustAll = tlsConfig.trustAll);
        if (trustAll) {
            options.setTrustAll(true);
            options.setVerifyHost(false);
        }
        if ((proxyOpt = OidcCommonUtils.toProxyOptions(oidcConfig.getProxy())).isPresent()) {
            options.setProxyOptions(proxyOpt.get());
        }
        if ((maxPoolSize = oidcConfig.maxPoolSize).isPresent()) {
            options.setMaxPoolSize(maxPoolSize.getAsInt());
        }
        options.setConnectTimeout((int)oidcConfig.getConnectionTimeout().toMillis());
    }

    public static String getAuthServerUrl(OidcCommonConfig oidcConfig) {
        String authServerUrl = oidcConfig.getAuthServerUrl().get();
        if (authServerUrl.endsWith("/")) {
            authServerUrl = authServerUrl.substring(0, authServerUrl.length() - 1);
        }
        return authServerUrl;
    }

    public static String getOidcEndpointUrl(String authServerUrl, Optional<String> endpointPath) {
        if (endpointPath.isPresent()) {
            return authServerUrl + OidcCommonUtils.prependSlash(endpointPath.get());
        }
        return null;
    }

    private static long getConnectionDelay(OidcCommonConfig oidcConfig) {
        return oidcConfig.getConnectionDelay().isPresent() ? oidcConfig.getConnectionDelay().get().getSeconds() : 0L;
    }

    public static long getConnectionDelayInMillis(OidcCommonConfig oidcConfig) {
        long connectionRetryCount;
        long connectionDelayInSecs = OidcCommonUtils.getConnectionDelay(oidcConfig);
        long l = connectionRetryCount = connectionDelayInSecs > 1L ? connectionDelayInSecs / 2L : 1L;
        if (connectionRetryCount > 1L) {
            LOG.infof("Connecting to OpenId Connect Provider for up to %d times every 2 seconds", (Object)connectionRetryCount);
        }
        return connectionDelayInSecs * 1000L;
    }

    public static Optional<ProxyOptions> toProxyOptions(OidcCommonConfig.Proxy proxyConfig) {
        if (!proxyConfig.host.isPresent()) {
            return Optional.empty();
        }
        JsonObject jsonOptions = new JsonObject();
        jsonOptions.put("host", proxyConfig.host.get());
        jsonOptions.put("port", proxyConfig.port);
        if (proxyConfig.username.isPresent()) {
            jsonOptions.put("username", proxyConfig.username.get());
        }
        if (proxyConfig.password.isPresent()) {
            jsonOptions.put("password", proxyConfig.password.get());
        }
        return Optional.of(new ProxyOptions(jsonOptions));
    }

    public static String formatConnectionErrorMessage(String authServerUrlString) {
        return String.format("OIDC server is not available at the '%s' URL. Please make sure it is correct. Note it has to end with a realm value if you work with Keycloak, for example: 'https://localhost:8180/auth/realms/quarkus'", authServerUrlString);
    }

    public static boolean isClientSecretBasicAuthRequired(OidcCommonConfig.Credentials creds) {
        return creds.secret.isPresent() || (creds.clientSecret.value.isPresent() || creds.clientSecret.provider.key.isPresent()) && creds.clientSecret.method.orElseGet(() -> OidcCommonConfig.Credentials.Secret.Method.BASIC) == OidcCommonConfig.Credentials.Secret.Method.BASIC;
    }

    public static boolean isClientJwtAuthRequired(OidcCommonConfig.Credentials creds) {
        return creds.jwt.secret.isPresent() || creds.jwt.secretProvider.key.isPresent() || creds.jwt.keyFile.isPresent() || creds.jwt.keyStoreFile.isPresent();
    }

    public static boolean isClientSecretPostAuthRequired(OidcCommonConfig.Credentials creds) {
        return (creds.clientSecret.value.isPresent() || creds.clientSecret.provider.key.isPresent()) && creds.clientSecret.method.orElseGet(() -> OidcCommonConfig.Credentials.Secret.Method.BASIC) == OidcCommonConfig.Credentials.Secret.Method.POST;
    }

    public static String clientSecret(OidcCommonConfig.Credentials creds) {
        return creds.secret.orElse(creds.clientSecret.value.orElseGet(OidcCommonUtils.fromCredentialsProvider(creds.clientSecret.provider)));
    }

    private static Supplier<? extends String> fromCredentialsProvider(final OidcCommonConfig.Credentials.Provider provider) {
        return new Supplier<String>(){

            @Override
            public String get() {
                String providerName;
                CredentialsProvider credentialsProvider;
                if (provider.key.isPresent() && (credentialsProvider = CredentialsProviderFinder.find(providerName = (String)provider.name.orElse(null))) != null) {
                    return credentialsProvider.getCredentials(providerName).get(provider.key.get());
                }
                return null;
            }
        };
    }

    public static Key clientJwtKey(OidcCommonConfig.Credentials creds) {
        if (creds.jwt.secret.isPresent() || creds.jwt.secretProvider.key.isPresent()) {
            return KeyUtils.createSecretKeyFromSecret(creds.jwt.secret.orElseGet(OidcCommonUtils.fromCredentialsProvider(creds.jwt.secretProvider)));
        }
        Key key = null;
        try {
            if (creds.jwt.keyFile.isPresent()) {
                key = KeyUtils.readSigningKey(creds.jwt.keyFile.get(), creds.jwt.keyId.orElse(null));
            } else if (creds.jwt.keyStoreFile.isPresent()) {
                KeyStore ks = KeyStore.getInstance("JKS");
                InputStream is = ResourceUtils.getResourceStream(creds.jwt.keyStoreFile.get());
                ks.load(is, creds.jwt.keyStorePassword.toCharArray());
                key = ks.getKey(creds.jwt.keyId.get(), creds.jwt.keyPassword.toCharArray());
            }
        }
        catch (Exception ex) {
            throw new ConfigurationException("Key can not be loaded");
        }
        if (key == null) {
            throw new ConfigurationException("Key is null");
        }
        return key;
    }

    public static String signJwt(OidcCommonConfig oidcConfig) {
        return OidcCommonUtils.signJwtWithKey(oidcConfig, OidcCommonUtils.clientJwtKey(oidcConfig.credentials));
    }

    public static String signJwtWithKey(OidcCommonConfig oidcConfig, Key key) {
        JwtSignatureBuilder builder = Jwt.issuer(oidcConfig.clientId.get()).subject(oidcConfig.clientId.get()).audience(OidcCommonUtils.getAuthServerUrl(oidcConfig)).expiresIn(oidcConfig.credentials.jwt.lifespan).jws();
        if (oidcConfig.credentials.jwt.getTokenKeyId().isPresent()) {
            builder.keyId(oidcConfig.credentials.jwt.getTokenKeyId().get());
        }
        if (key instanceof SecretKey) {
            return builder.sign((SecretKey)key);
        }
        return builder.sign((PrivateKey)key);
    }

    public static void verifyConfigurationId(String defaultId, String configKey, Optional<String> configId) {
        if (configKey.equals(defaultId)) {
            throw new ConfigurationException("configuration id '" + configKey + "' duplicates the default configuration id");
        }
        if (configId.isPresent() && !configKey.equals(configId.get())) {
            throw new ConfigurationException("Configuration has 2 different id values: '" + configKey + "' and '" + configId.get() + "'");
        }
    }

    public static String initClientSecretBasicAuth(OidcCommonConfig oidcConfig) {
        if (OidcCommonUtils.isClientSecretBasicAuthRequired(oidcConfig.credentials)) {
            return "Basic " + Base64.getEncoder().encodeToString((oidcConfig.getClientId().get() + ":" + OidcCommonUtils.clientSecret(oidcConfig.credentials)).getBytes(StandardCharsets.UTF_8));
        }
        return null;
    }

    public static Key initClientJwtKey(OidcCommonConfig oidcConfig) {
        if (OidcCommonUtils.isClientJwtAuthRequired(oidcConfig.credentials)) {
            return OidcCommonUtils.clientJwtKey(oidcConfig.credentials);
        }
        return null;
    }

    public static Predicate<? super Throwable> oidcEndpointNotAvailable() {
        return t -> t instanceof ConnectException || t instanceof OidcEndpointAccessException && ((OidcEndpointAccessException)t).getErrorStatus() == 404;
    }

    public static Uni<JsonObject> discoverMetadata(WebClient client, String authServerUrl, long connectionDelayInMillisecs) {
        String discoveryUrl = authServerUrl + "/.well-known/openid-configuration";
        return client.getAbs(discoveryUrl).send().onItem().transform(resp -> {
            if (resp.statusCode() == 200) {
                return resp.bodyAsJsonObject();
            }
            LOG.tracef("Discovery has failed, status code: %d", resp.statusCode());
            throw new OidcEndpointAccessException(resp.statusCode());
        }).onFailure(OidcCommonUtils.oidcEndpointNotAvailable()).retry().withBackOff(CONNECTION_BACKOFF_DURATION, CONNECTION_BACKOFF_DURATION).expireIn(connectionDelayInMillisecs).onFailure().transform(t -> t.getCause());
    }
}

